/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d

import com.macrofocus.colormapping.implementation.SimpleColorMapping
import com.macrofocus.common.format.CPFormat
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.timer.CPExecutor
import com.macrofocus.high_d.axis.AxisFactory
import com.macrofocus.high_d.axis.hierarchy.AxisHierarchy
import com.macrofocus.high_d.axis.hierarchy.DefaultAxisHierarchy
import com.macrofocus.high_d.barchart.BarChart
import com.macrofocus.high_d.barchart.BarChartModel
import com.macrofocus.high_d.barchart.DefaultBarChartModel
import com.macrofocus.high_d.details.Details
import com.macrofocus.high_d.distributions.AbstractDistributions
import com.macrofocus.high_d.distributions.DefaultDistributionsModel
import com.macrofocus.high_d.distributions.Distributions
import com.macrofocus.high_d.distributions.DistributionsModel
import com.macrofocus.high_d.filters.CategoricalFilterComponent
import com.macrofocus.high_d.filters.Filters
import com.macrofocus.high_d.filters.OrdinalFilterComponent
import com.macrofocus.high_d.heatmap.DefaultHeatMapModel
import com.macrofocus.high_d.heatmap.HeatMap
import com.macrofocus.high_d.heatmap.HeatMapModel
import com.macrofocus.high_d.list.ListView
import com.macrofocus.high_d.parallelcoordinates.AbstractParallelCoordinates
import com.macrofocus.high_d.parallelcoordinates.DefaultParallelCoordinatesSettings
import com.macrofocus.high_d.parallelcoordinates.ParallelCoordinates
import com.macrofocus.high_d.parallelcoordinates.geometry.Geometry
import com.macrofocus.high_d.parallelcoordinatesmatrix.AbstractParallelCoordinatesMatrix
import com.macrofocus.high_d.parallelcoordinatesmatrix.DefaultParallelCoordinatesMatrixModel
import com.macrofocus.high_d.parallelcoordinatesmatrix.ParallelCoordinatesMatrix
import com.macrofocus.high_d.parallelcoordinatesmatrix.ParallelCoordinatesMatrixModel
import com.macrofocus.high_d.regression.Regression
import com.macrofocus.high_d.scatterplot.AbstractScatterPlot
import com.macrofocus.high_d.scatterplot.DefaultScatterPlotModel
import com.macrofocus.high_d.scatterplot.ScatterPlot
import com.macrofocus.high_d.scatterplot.ScatterPlotModel
import com.macrofocus.high_d.scatterplotmatrix.AbstractScatterPlotMatrix
import com.macrofocus.high_d.scatterplotmatrix.DefaultScatterPlotMatrixModel
import com.macrofocus.high_d.scatterplotmatrix.ScatterPlotMatrix
import com.macrofocus.high_d.scatterplotmatrix.ScatterPlotMatrixModel
import com.macrofocus.high_d.som.DefaultSOMModel
import com.macrofocus.high_d.som.SOM
import com.macrofocus.high_d.som.SOMModel
import com.macrofocus.high_d.statistics.Statistics
import com.macrofocus.high_d.statistics.StatisticsDataFrame
import com.macrofocus.high_d.table.Table
import com.macrofocus.high_d.tablelens.AbstractTableLens
import com.macrofocus.high_d.tablelens.DefaultTableLensModel
import com.macrofocus.high_d.tablelens.TableLens
import com.macrofocus.high_d.tablelens.TableLensModel
import com.macrofocus.license.LicenseModel
import com.macrofocus.visual.SimpleVisual
import com.macrofocus.visual.SimpleVisualObjects
import org.mkui.canvas.Rendering
import org.mkui.colormap.ColorMapFactory
import org.mkui.graphics.colortheme.ColorTheme
import org.mkui.interaction.InteractionMode
import org.mkui.visual.Visual
import org.molap.dataframe.DataFrame
import org.molap.subset.MultiBinningDimension
import org.molap.subset.Reducer
import org.molap.subset.SingleBinningDimension
import org.molap.subset.SubsetDataFrame

/**
 * Factory to ease the creation of the High-D components. A single factory will creates visual components that share
 * visual state such as probing, selection, filtering, colors, etc.
 *
 * @param <Component> the type of native components
 * @param <Row>       type type of rows
 * @param <Column>    the type of columns
</Column></Row></Component> */
abstract class HighDFactory<Row, Column, Value> protected constructor(
    settings: HighDSettings?,
    colorMapFactory: ColorMapFactory
) {
    protected var settings: HighDSettings
    /**
     * Returns the data frame containing the data.
     *
     * @return a data frame
     */
    protected var dataFrame: DataFrame<Row, Column, Value>? = null
        /**
         * Sets the data frame holding the data.
         *
         * @param dataFrame the data frame
         */
        set(value) {
            setDataFrame(
                value!!,
                SimpleVisual<Row, Column>(SimpleVisualObjects<Row>(value), SimpleColorMapping<Row, Column>(getColorMapFactory(), value))
            )
        }

    var subsetDataFrame: SubsetDataFrame<Row, Column, Value>? = null
        get() {
            if (field == null) {
                field = SubsetDataFrame<Row, Column, Value>(
                    dataFrame!!,
                    if (visual != null) visual!!.filter else null,
                    if (visual != null) visual!!.selection else null
                )
            }
            return field
        }
    /**
     * Returns the data frame holding the annotation data.
     *
     * @return the annotation data frame
     */
    protected var annotationDataFrame: DataFrame<*, Column, *>? = null
    get() {
        if (field == null) {
            field = StatisticsDataFrame(
                dataFrame!!,
                visual!!.probing,
                visual!!.selection,
                visual!!.filter,
                visual!!.colorMapping,
                visual!!.coloring
            )
        }
        return field
    }

    /**
     * Returns the visual context holding information about the interactive state.
     *
     * @return the visual context
     */
    protected var visual: Visual<Row, Column>?

    /**
     * Returns the hierarchy of axes.
     *
     * @return the hierarchy
     */
    var axisHierarchy: AxisHierarchy<Row, Column>?

    /**
     * Returns the hierarchy of axes for the filters.
     *
     * @return the hierarchy
     */
    protected var filtersAxisHierarchy: AxisHierarchy<Row, Column>?

    private val colorMapFactory: ColorMapFactory

    protected constructor(
        colorMapFactory: ColorMapFactory,
        dataFrame: DataFrame<Row, Column, Value>
    ) : this(HighDSettings(), colorMapFactory) {
        this.dataFrame = dataFrame
        //        this(factory, dataFrame, new SimpleVisual<Color,Row,Column>(new SimpleVisualObjects<Row>(dataFrame), new SimpleColorMapping<Color,Row,Column>(colorMapFactory, dataFrame)));
    }

    protected constructor(
        colorMapFactory: ColorMapFactory,
        dataFrame: DataFrame<Row, Column, Value>,
        visual: Visual<Row, Column>
    ) : this(HighDSettings(), colorMapFactory) {
        setDataFrame(dataFrame, visual)
        val axisFactory: AxisFactory<Row, Column> =
            AxisFactory(dataFrame, subsetDataFrame, annotationDataFrame, visual.probing, visual.selection, visual.filter)
        axisHierarchy = DefaultAxisHierarchy<Row, Column>(axisFactory, dataFrame, true, false, 10)
        filtersAxisHierarchy = DefaultAxisHierarchy<Row, Column>(axisFactory, dataFrame, false, true, 10)

//        this(dataFrame, null, visual, new DefaultAxisHierarchy<Row,Column>(factory, dataFrame, null, visual.getProbing(), visual.getFilter()));
    }

    protected constructor(
        colorMapFactory: ColorMapFactory,
        dataFrame: DataFrame<Row, Column, Value>,
        subsetDataFrame: SubsetDataFrame<Row, Column, Value>,
        annotationDataFrame: DataFrame<*, Column, *>,
        visual: Visual<Row, Column>,
        axisHierarchy: AxisHierarchy<Row, Column>,
        filtersAxisHierarchy: AxisHierarchy<Row, Column>,
        settings: HighDSettings
    ) : this(settings, colorMapFactory) {
        this.dataFrame = dataFrame
        this.annotationDataFrame = annotationDataFrame
        this.subsetDataFrame = subsetDataFrame
        this.visual = visual
        this.axisHierarchy = axisHierarchy
        this.filtersAxisHierarchy = filtersAxisHierarchy
    }

    fun getShowFiltered(): MutableProperty<Boolean> {
        return settings.properties.getProperty(HighDSettings.PropertyType.showFiltered) as MutableProperty<Boolean>
    }

    fun getAntialiasing(): MutableProperty<Boolean> {
        return settings.properties.getProperty(HighDSettings.PropertyType.antialiasing) as MutableProperty<Boolean>
    }

    fun getRendering(): MutableProperty<Rendering> {
        return settings.properties.getProperty(HighDSettings.PropertyType.rendering) as MutableProperty<Rendering>
    }

    fun getColorTheme(): MutableProperty<ColorTheme> {
        return settings.properties.getProperty(HighDSettings.PropertyType.ColorTheme) as MutableProperty<ColorTheme>
    }

    fun getGeometry(): MutableProperty<Geometry> {
        return settings.properties.getProperty(HighDSettings.PropertyType.geometry) as MutableProperty<Geometry>
    }

    fun setColorTheme(colorTheme: MutableProperty<ColorTheme>) {
        settings.properties.replaceProperty<ColorTheme>(HighDSettings.PropertyType.ColorTheme, colorTheme)
    }

    fun getInteractionMode(): MutableProperty<InteractionMode> {
        return settings.properties.getProperty(HighDSettings.PropertyType.InteractionMode) as MutableProperty<InteractionMode>
    }

    fun setInteractionMode(interactionMode: MutableProperty<InteractionMode>) {
        settings.properties.replaceProperty<InteractionMode>(HighDSettings.PropertyType.InteractionMode, interactionMode)
    }

    fun getAllowScrolling(): MutableProperty<Boolean> {
        return settings.properties.getProperty(HighDSettings.PropertyType.allowScrolling) as MutableProperty<Boolean>
    }

    abstract fun <Row, Column, Value> createHighDFactory(dataFrame: DataFrame<Row, Column, Value>?): HighDFactory<Row, Column, Value>?
    abstract fun createHighDFactory(
        dataFrame: DataFrame<Row, Column, Value>?,
        subsetDataFrame: SubsetDataFrame<Row, Column, Value>?,
        annotationDataFrame: DataFrame<*, Column, *>?,
        visual: Visual<Row, Column>?,
        axisHierarchy: AxisHierarchy<Row, Column>?,
        filtersAxisHierarchy: AxisHierarchy<Row, Column>?,
        settings: HighDSettings?
    ): HighDFactory<Row, Column, Value>

    fun forColumns(vararg columns: Column): HighDFactory<Row, Column, Value> {
        val axisHierarchy: DefaultAxisHierarchy<Row, Column> = DefaultAxisHierarchy(axisHierarchy!!.axisFactory, columns.asIterable())
        return createHighDFactory(
            dataFrame,
            subsetDataFrame,
            annotationDataFrame,
            visual,
            axisHierarchy,
            filtersAxisHierarchy,
            settings
        )
    }

    fun forColumns(columns: Iterable<Column>?): HighDFactory<Row, Column, Value> {
        val axisHierarchy: DefaultAxisHierarchy<Row, Column> = DefaultAxisHierarchy(axisHierarchy!!.axisFactory, columns)
        return createHighDFactory(
            dataFrame,
            subsetDataFrame,
            annotationDataFrame,
            visual,
            axisHierarchy,
            filtersAxisHierarchy,
            settings
        )
    }

    fun getColorMapFactory(): ColorMapFactory {
        return colorMapFactory
    }

    /**
     * Sets the data frame holding the data.
     *
     * @param dataFrame the data frame
     */
    fun setDataFrame(dataFrame: DataFrame<Row, Column, Value>, visual: Visual<Row, Column>) {
        this.dataFrame = dataFrame
        this.visual = visual
        val axisFactory: AxisFactory<Row, Column> = AxisFactory(
            dataFrame,
            subsetDataFrame,
            annotationDataFrame,
            visual.probing,
            visual.selection,
            visual.filter
        )
        axisHierarchy = DefaultAxisHierarchy<Row, Column>(axisFactory, dataFrame, true, false, 10)
        filtersAxisHierarchy = DefaultAxisHierarchy<Row, Column>(axisFactory, dataFrame, false, true, 10)
    }

    fun setDataFrame(
        dataFrame: DataFrame<Row, Column, Value>,
        subsetDataFrame: SubsetDataFrame<Row, Column, Value>,
        annotationDataFrame: DataFrame<*, Column, *>,
        visual: Visual<Row, Column>,
        axisHierarchy: AxisHierarchy<Row, Column>,
        filtersAxisHierarchy: AxisHierarchy<Row, Column>
    ) {
        this.dataFrame = dataFrame
        this.subsetDataFrame = subsetDataFrame
        this.annotationDataFrame = annotationDataFrame
        this.visual = visual
        this.axisHierarchy = axisHierarchy
        this.filtersAxisHierarchy = filtersAxisHierarchy
    }

    /**
     * Creates a parallel coordinates plot.
     *
     * @return a parallel coordinates plot
     */
    abstract fun createParallelCoordinates(): ParallelCoordinates<Row, Column>?
    protected fun customizeParallelCoordinates(parallelCoordinates: ParallelCoordinates<Row, Column>) {
        parallelCoordinates.view!!.setShowFiltered(settings.getShowFiltered())
        parallelCoordinates.view!!.setAntialiasing(settings.getAntialiasing())
        parallelCoordinates.view!!.setRendering(settings.getRendering())
        parallelCoordinates.view!!.setColorTheme(settings.getColorTheme())
        parallelCoordinates.view!!.setGeometry(settings.getGeometry())
        parallelCoordinates.view!!.setAllowScrolling(settings.getAllowScrolling())
    }

    /**
     * Creates a parallel coordinates matrix plot.
     *
     * @return a parallel coordinates matrix plot
     */
    abstract fun createParallelCoordinatesMatrix(): ParallelCoordinatesMatrix<Row, Column>?
    protected fun customizeParallelCoordinatesMatrix(parallelCoordinatesMatrix: ParallelCoordinatesMatrix<Row, Column>) {
        parallelCoordinatesMatrix.view!!.setShowFiltered(settings.getShowFiltered())
        parallelCoordinatesMatrix.view!!.setAntialiasing(settings.getAntialiasing())
        parallelCoordinatesMatrix.view!!.setRendering(settings.getRendering())
        parallelCoordinatesMatrix.view!!.setColorTheme(settings.getColorTheme())
    }

    fun createParallelCoordinatesMatrixModel(): ParallelCoordinatesMatrixModel<Row, Column>? {
        return if (dataFrame != null) {
            DefaultParallelCoordinatesMatrixModel<Row, Column>(
                dataFrame!!,
                visual!!,
                axisHierarchy!!.flat,
                axisHierarchy!!,
                DefaultParallelCoordinatesSettings()
            )
        } else {
            null
        }
    }

    /**
     * Creates a tables lens.
     *
     * @return a tables lens
     */
    abstract fun createTableLens(): TableLens<Row, Column>?
    fun createTableLensModel(): TableLensModel<Row, Column>? {
        return if (dataFrame != null) {
            DefaultTableLensModel<Row, Column>(dataFrame!!, visual!!, axisHierarchy!!.flat)
        } else {
            null
        }
    }

    protected fun customizeTableLens(tableLens: TableLens<Row, Column>) {
        if (tableLens.model != null) {
            tableLens.model!!.setShowFiltered(settings.getShowFiltered())
        }
        tableLens.view!!.setShowFiltered(settings.getShowFiltered())
        tableLens.view!!.setAntialiasing(settings.getAntialiasing())
        tableLens.view!!.setRendering(settings.getRendering())
        tableLens.view!!.setColorTheme(settings.getColorTheme())
    }

    /**
     * Creates a scatter plot.
     *
     * @return a scatter plot
     */
    abstract fun createScatterPlot(): ScatterPlot<Row, Column>?

    /**
     * Creates a scatter plot.
     *
     * @return a scatter plot
     */
    abstract fun createScatterPlot(model: ScatterPlotModel<Row, Column>?): ScatterPlot<Row, Column>?
    protected fun customizeScatterPlot(scatterPlot: ScatterPlot<Row, Column>) {
        scatterPlot.view!!.setShowFiltered(settings.getShowFiltered())
        scatterPlot.view!!.setAntialiasing(settings.getAntialiasing())
        scatterPlot.view!!.setRendering(settings.getRendering())
        scatterPlot.view!!.setColorTheme(settings.getColorTheme())
    }

    fun createScatterPlotModel(): ScatterPlotModel<Row, Column>? {
        return if (dataFrame != null) {
            DefaultScatterPlotModel<Row, Column>(dataFrame!!, annotationDataFrame, visual!!, axisHierarchy!!.flat)
        } else {
            null
        }
    }

    /**
     * Creates a MDS
     *
     * @return a MDS
     */
//    abstract fun createMDS(): MDS<Row, Column>?
//    protected fun customizeMDS(mds: MDS<Row, Column>) {
//        mds.view.setShowFiltered(settings.getShowFiltered())
//        mds.view.setAntialiasing(settings.getAntialiasing())
//        mds.view.setRendering(settings.getRendering())
//        mds.view.setColorTheme(settings.getColorTheme())
//    }
//
//    abstract fun createMDSModel(): MDSScatterPlotModel<Row, Column>?

    /**
     * Creates a control chart.
     *
     * @return a control chart
     */
//    abstract fun createControlChart(): ControlChart<Row, Column>?
//    protected fun customizeControlChart(controlChart: ControlChart<Row, Column>) {
//        controlChart.view.setShowFiltered(settings.getShowFiltered())
//        controlChart.view.setAntialiasing(settings.getAntialiasing())
//        controlChart.view.setRendering(settings.getRendering())
//        controlChart.view.setColorTheme(settings.getColorTheme())
//    }
//
//    fun createControlCharttModel(): ControlChartScatterPlotModel<Row, Column>? {
//        return if (getDataFrame() != null) {
//            ControlChartScatterPlotModel<Row, Column>(getDataFrame(), getAnnotationDataFrame(), getVisual(), getAxisHierarchy().getFlat())
//        } else {
//            null
//        }
//    }

    /**
     * Creates a scatter plot matrix.
     *
     * @return a scatter plot matrix
     */
    abstract fun createScatterPlotMatrix(): ScatterPlotMatrix<Row, Column>?
    protected fun customizeScatterPlotMatrix(scatterPlotMatrix: ScatterPlotMatrix<Row, Column>) {
        scatterPlotMatrix.view!!.setShowFiltered(settings.getShowFiltered())
        scatterPlotMatrix.view!!.setAntialiasing(settings.getAntialiasing())
        scatterPlotMatrix.view!!.setRendering(settings.getRendering())
        scatterPlotMatrix.view!!.setColorTheme(settings.getColorTheme())
    }

    fun createScatterPlotMatrixModel(): ScatterPlotMatrixModel<Row, Column>? {
        return if (dataFrame != null) {
            DefaultScatterPlotMatrixModel<Row, Column>(dataFrame!!, visual!!, axisHierarchy!!.flat)
        } else {
            null
        }
    }

    /**
     * Creates a multidimensional scaling plot.
     *
     * @return a multidimensional scaling plot
     */
    //    @GwtIncompatible
    //    public abstract MDS<Row,Column> createMDS();
    abstract fun createExecutor(): CPExecutor?

    /**
     * Creates a distributions plot.
     *
     * @return a distributions plot
     */
    abstract fun createDistributions(): Distributions<Row, Column, Value, *>?
    fun <Bin> createDistributionsModel(): DistributionsModel<Row, Column, Value, Bin>? {
        return if (dataFrame != null) {
            DefaultDistributionsModel<Row, Column, Value, Bin>(dataFrame!!, subsetDataFrame!!, visual!!, axisHierarchy!!.flat)
        } else {
            null
        }
    }

    protected fun customizeDistribution(distributions: Distributions<Row, Column, Value, *>) {
        distributions.view!!.setShowFiltered(settings.getShowFiltered())
        distributions.view!!.setAntialiasing(settings.getAntialiasing())
        distributions.view!!.setRendering(settings.getRendering())
        distributions.view!!.setColorTheme(settings.getColorTheme())
        distributions.view!!.setInteractionMode(getInteractionMode())
    }

    /**
     * Creates a statistic table.
     *
     * @return a statistic table
     */
    abstract fun createStatistics(): Statistics<Row, Column, Value>?
    protected fun customizeStatistics(statistics: Statistics<Row, Column, Value>) {
        statistics.setColorTheme(getColorTheme())
    }
    /**
     * Creates a treemap.
     *
     * @param <N> the type of nodes
     * @return a treemap
    </N> */
    //    public abstract <N extends AbstractTreeMapNode<N,Row,Column>> TreeMap<Component, N,Row,Column> createTreeMap();
    /**
     * Creates a table.
     *
     * @return a table
     */
    abstract fun createTable(): Table<Row, Column, Value>?
    abstract fun <Row, Column, Value> createTable(
        dataFrame: DataFrame<Row, Column, Value>?,
        visual: Visual<Row, Column>?
    ): Table<Row, Column, Value>?

    protected fun customizeTable(table: Table<Row, Column, Value>) {
        table.setShowFiltered(settings.getShowFiltered())
        table.setColorTheme(settings.getColorTheme())
    }

    abstract fun createList(column: Column): ListView<Int?, String?>?
    abstract fun createFilters(): Filters<Row, Column, Value>?
    fun customizeFilters(filters: Filters<Row, Column, Value>) {
//        filters.setShowFiltered(getShowFiltered());
        filters.colorTheme = settings.getColorTheme()
        filters.interactionMode = getInteractionMode()
    }

    abstract fun createCategoricalFilter(column: Column, reducer: Reducer<Row, Value>?): CategoricalFilterComponent?
    abstract fun <Bin> createCategoricalFilter(
        column: Column,
        reducer: Reducer<Row, Bin>?,
        binningStrategy: SingleBinningDimension.SingleBinningStrategy<Row, Bin>?
    ): CategoricalFilterComponent?

    abstract fun <Bin> createCategoricalFilter(
        column: Column,
        reducer: Reducer<Row, Bin>?,
        binningStrategy: MultiBinningDimension.MultiBinningStrategy<Row, Bin>?
    ): CategoricalFilterComponent?

    protected fun customizeCategoricalFilterComponent(categoricalFilterComponent: CategoricalFilterComponent) {
//        categoricalFilterComponent.setShowFiltered(getShowFiltered());
        categoricalFilterComponent.colorTheme = settings.getColorTheme()
        categoricalFilterComponent.interactionMode = getInteractionMode()
    }

    fun createOrdinalFilter(column: Column): OrdinalFilterComponent? {
        return createOrdinalFilter(column, null, 8)
    }

    abstract fun createOrdinalFilter(column: Column, format: CPFormat<*>?, visibleLength: Int): OrdinalFilterComponent?
    abstract fun createSOM(): SOM<Row, Column>?
    protected fun createSOMModel(): SOMModel<Row, Column>? {
        return if (dataFrame != null) {
            DefaultSOMModel()
        } else {
            null
        }
    }

    protected fun customizeSOM(som: SOM<Row, Column>?) {}
    abstract fun createBarChart(): BarChart<Row, Column>?
    protected fun createBarChartModel(): BarChartModel<Row, Column>? {
        return if (dataFrame != null) {
            DefaultBarChartModel()
        } else {
            null
        }
    }

    abstract fun createHeatMap(): HeatMap<Row, Column>?
    protected fun customizeBarChart(barChart: BarChart<Row, Column>?) {}
    protected fun createHeatMapModel(): HeatMapModel<Row, Column>? {
        return if (dataFrame != null) {
            DefaultHeatMapModel()
        } else {
            null
        }
    }

    protected fun customizeHeatMap(heatMap: HeatMap<Row, Column>?) {}
    abstract fun createDetails(): Details<Row, Column>?

//    abstract fun createClustering(): Clustering<Row, Column>?
//    protected fun createClusteringModel(): SingleBinningDimension<Row, *>? {
//        return if (getSubsetDataFrame() != null) {
//            getSubsetDataFrame()!!.getBinningDimension("Clusters" as Column, null)
//        } else {
//            null
//        }
//    }

    abstract fun createRegression(): Regression<Row, Column>?

    companion object {
        private var licenseModel: LicenseModel? = null

        /**
         * Register the license key that legally entitles the use of the High-D API. This should be called prior to the
         * first instantiation of this component.
         *
         * @param username the user name
         * @param key      the license key
         */
        fun setLicenseKey(username: String?, key: String?) {
//            licenseModel = SimpleLicenseModel(ApplicationModelFactory.HIGHD_API, username, key)
            AbstractParallelCoordinates.setLicenseKey(username, key)
            AbstractParallelCoordinatesMatrix.setLicenseKey(username, key)
            AbstractScatterPlot.setLicenseKey(username, key)
            AbstractScatterPlotMatrix.setLicenseKey(username, key)
            AbstractDistributions.setLicenseKey(username, key)
            AbstractTableLens.setLicenseKey(username, key)
        }
    }

    init {
        this.settings = if (settings != null) settings else HighDSettings()
        this.colorMapFactory = colorMapFactory
        dataFrame = null
        visual = null
        axisHierarchy = null
        filtersAxisHierarchy = null
    }
}