/*
 * Copyright (c) 2013 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.distributions

import com.macrofocus.colormapping.implementation.SimpleColorMapping
import com.macrofocus.common.filter.Filter
import com.macrofocus.common.properties.MutableProperties
import com.macrofocus.common.properties.Property
import com.macrofocus.common.properties.SimpleProperties
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.high_d.axis.Alignment
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import com.macrofocus.high_d.axis.group.DefaultAxisGroupModel
import com.macrofocus.high_d.axis.locations.AxisLocations
import com.macrofocus.high_d.axis.locations.DefaultAxisLocations
import com.macrofocus.visual.SimpleVisual
import com.macrofocus.visual.SimpleVisualObjects
import org.mkui.coloring.MutableColoring
import org.mkui.colormap.ColorMapFactory
import org.mkui.colormapping.MutableColorMapping
import org.mkui.visual.Visual
import org.mkui.visual.VisualObjects
import org.molap.dataframe.DataFrame
import org.molap.subset.DistributionDimension
import org.molap.subset.SubsetDataFrame

class DefaultDistributionsModel<Row, Column, Value, Bin>(
    dataFrame: DataFrame<Row, Column, *>,
    subsetDataFrame: SubsetDataFrame<Row, Column, Value>,
    visual: Visual<Row, Column>,
    axisGroupModel: AxisGroupModel<Row, Column>
) : AbstractDistributionsModel<Row, Column, Value, Bin>(), VisualObjects<Row> {
    private val dataFrame: DataFrame<Row, Column, *>
    private val subsetDataFrame: SubsetDataFrame<Row, Column, Value>
    private val visual: Visual<Row, Column>
    private val axisGroupModel: AxisGroupModel<Row, Column>
    private val axisLocations: AxisLocations<Row, Column>
    private val order: AxisModel<*,*>? = null
    private val sortedRow: MutableList<Row>
    private val rowComparator: RowComparator? = null
    private val properties: MutableProperties<String?> = SimpleProperties()
    private val maxDimensions: Property<Int> = properties.createProperty("maxDimensions", Int.MAX_VALUE)

    constructor(
        colorMapFactory: ColorMapFactory,
        dataFrame: DataFrame<Row, Column, *>,
        subsetDataFrame: SubsetDataFrame<Row, Column, Value>
    ) : this(
        dataFrame,
        subsetDataFrame,
        SimpleVisual<Row, Column>(SimpleVisualObjects(dataFrame), SimpleColorMapping(colorMapFactory, dataFrame))
    ) {
    }

    constructor(
        dataFrame: DataFrame<Row, Column, *>,
        subsetDataFrame: SubsetDataFrame<Row, Column, Value>,
        visual: Visual<Row, Column>
    ) : this(
        dataFrame,
        subsetDataFrame,
        visual,
        DefaultAxisGroupModel("Root", dataFrame, null, visual.probing, visual.selection, visual.filter)
    ) {
    }

    override fun getVisual(): Visual<Row, Column> {
        return visual
    }

    override val objectCount: Int
        get() = dataFrame.rowCount

    /**
     * {@inheritDoc}
     */
    override fun getProbing(): MutableSingleSelection<Row> {
        return visual.probing
    }

    /**
     * {@inheritDoc}
     */
    override fun getSelection(): MutableSelection<Row> {
        return visual.selection
    }

    /**
     * {@inheritDoc}
     */
    override fun getFilter(): Filter<Row> {
        return visual.filter
    }

    /**
     * {@inheritDoc}
     */
    override fun getColorMapping(): MutableColorMapping<Row, Column> {
        return visual.colorMapping
    }

    /**
     * {@inheritDoc}
     */
    override fun getColoring(): MutableColoring<Row> {
        return visual.coloring
    }

    /**
     * {@inheritDoc}
     */
    override fun getObject(index: Int): Row {
        return if (order != null) {
            sortedRow[index]
        } else {
            dataFrame.getRowKey(index)
        }
    }

    override fun getIndex(row: Row): Int {
        return if (order != null) {
            sortedRow.binarySearch(row, rowComparator!!)
        } else {
            dataFrame.getRowAddress(row)
        }
    }

    /**
     * {@inheritDoc}
     */
    private fun getAxisModel(c: Int): AxisModel<*,*> {
        return axisGroupModel.axisOrder!!.get(c)
    }

    /**
     * {@inheritDoc}
     */
    override fun getAxisCount(): Int {
        return axisGroupModel.axisOrder!!.size()
    }

    /**
     * {@inheritDoc}
     */
    override fun getLocation(axisModel: AxisModel<Row, Column>): Double {
        return axisLocations.getLocation(Alignment.Left, axisModel)
    }

    /**
     * {@inheritDoc}
     */
    override fun getAxisGroupModel(): AxisGroupModel<Row, Column> {
        return axisGroupModel
    }

    /**
     * {@inheritDoc}
     */
    override fun setLocation(axisModel: AxisModel<Row, Column>, l: Double) {
        axisLocations.setLocation(Alignment.Left, axisModel, l)
        firedDistributionsChanged()
    }

    fun getVisibleAxisCount(): Int {
        return axisGroupModel.axisOrder!!.size()
    }

    override fun createDistributionDimension(
        column: Column,
        distributionStrategy: DistributionDimension.DistributionStrategy<Value, Bin>
    ): DistributionDimension<Row, Value, Bin> {
        return subsetDataFrame.getDistributionDimension(column, distributionStrategy)
    }

    override fun removeDistributionDimension(distributionDimension: DistributionDimension<Row, Value, Bin>) {
        subsetDataFrame.removeDimension(distributionDimension)
    }

    private inner class RowComparator(private val axisModel: AxisModel<Row, Column>) : Comparator<Row> {
        override fun compare(o1: Row, o2: Row): Int {
            val v1: Number? = axisModel.getValue(o1)
            return if (v1 != null) {
                val v2: Number? = axisModel.getValue(o2)
                if (v2 != null) {
                    val c: Int = v1.toDouble().compareTo(v2.toDouble())
                    if (c == 0) {
//                        return o1.compareTo(o2);
                        0
                    } else {
                        c
                    }
                } else {
                    -1
                }
            } else {
                val v2: Number? = axisModel.getValue(o2)
                if (v2 != null) {
                    1
                } else {
//                    return o1.compareTo(o2);
                    0
                }
            }
        }
    }

    init {
        this.dataFrame = dataFrame
        this.subsetDataFrame = subsetDataFrame
        this.axisGroupModel = axisGroupModel
        axisLocations = DefaultAxisLocations<Row, Column>(axisGroupModel.axisOrder!!, maxDimensions)
        this.visual = visual
        sortedRow = ArrayList<Row>()
        for (r in 0 until objectCount) {
            sortedRow.add(getObject(r))
        }
    }
}