package com.macrofocus.high_d.distributions

import com.macrofocus.common.filter.Filter

/**
 * A class that models a standard histogram. A histogram consists of a number of equally sized and contiguous bins that
 * cover a certain interval. Values that are added to the histogram are classified into their corresponding bin, which
 * means that the count of the bin is incremented. Values that fall outside the interval, are classified into the
 * underflow resp. overflow bins.
 */
class FixedBinsHistogram<R>(private val numberOfBins: Int, private val minValue: Double, private val maxValue: Double, filter: Filter<R>) :
    Histogram<R> {
    private var maxCount = 0
    private var activeMaxCount = 0
    private val bins: IntArray
    private var underflowBin = 0
    private var overflowBin = 0
    private val activeBins: Array<MutableList<R>?>
    private val activeUnderflowBin: MutableList<R> = ArrayList<R>()
    private val activeOverflowBin: MutableList<R> = ArrayList<R>()
    private val filter: Filter<R>
    fun addValue(row: R, value: Double) {
        if (numberOfBins <= 0) return
        val bin = getBin(value)
        if (bin < 0) {
            when (bin) {
                -1 -> underflowBin++
                -2 -> overflowBin++
            }
        } else {
            bins[bin]++
            if (bins[bin] > maxCount) {
                maxCount = bins[bin]
            }
        }
        if (!filter.isFiltered(row)) {
            if (bin < 0) {
                when (bin) {
                    -1 -> activeUnderflowBin.add(row)
                    -2 -> activeOverflowBin.add(row)
                }
            } else {
                activeBins[bin]!!.add(row)
                if (activeBins[bin]!!.size > activeMaxCount) {
                    activeMaxCount = activeBins[bin]!!.size
                }
            }
        }
    }

    override fun getActiveDensity(index: Int): Int {
        return activeBins[index]!!.size
    }

    override fun getActiveRowAtBin(j: Int, index: Int): R {
        return activeBins[j]!![index]
    }

    override fun getDensity(index: Int): Int {
        return bins[index]
    }

    override fun getNumberOfBins(): Int {
        return numberOfBins
    }

    override fun getMinValue(): Double {
        return minValue
    }

    override fun getMaxValue(): Double {
        return maxValue
    }

    override fun getBinStartValue(bin: Int): Double {
        return bin * ((maxValue - minValue) / numberOfBins) + minValue
    }

    override fun getBinEndValue(bin: Int): Double {
        return (bin + 1) * ((maxValue - minValue) / numberOfBins) + minValue
    }

    override fun getMaxActiveDensity(): Int {
        return activeMaxCount
    }

    override fun getMaxDensity(): Int {
        return maxCount
    }

    fun getUnderflowBin(): Int {
        return underflowBin
    }

    fun getOverflowBin(): Int {
        return overflowBin
    }

    override fun getBin(value: Double): Int {
        if (value < minValue) return -1
        if (value > maxValue) return -2
        if (value == maxValue) return numberOfBins - 1
        return if (value == minValue) 0 else ((value - minValue) / ((maxValue - minValue) / numberOfBins)).toInt()
    }

    override fun toString(): String {
        var content = "------\n"
        content += this::class.simpleName + "\n"
        content += "   Number of bins: $numberOfBins\n"
        content += "   Min value:      $minValue\n"
        content += "   Max value:      $maxValue\n"
        content += "   Max count:      $maxCount\n"
        content += "   Underflow bin:  $underflowBin\n"
        content += "   Overflow bin:   $overflowBin\n"
        return content
    }

    init {
        bins = IntArray(numberOfBins)
        activeBins = arrayOfNulls(numberOfBins)
        for (i in activeBins.indices) {
            activeBins[i] = ArrayList<R>()
        }
        this.filter = filter
    }
}