/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.scatterplot

import org.mkui.canvas.*
import org.mkui.geom.Point
import org.mkui.geom.Rectangle2D
import kotlin.math.max
import kotlin.math.min

/**
 * Created by luc on 15/06/16.
 */
abstract class AbstractScatterPlotController<Row, Column>(view: ScatterPlotView<Row, Column>?) : ScatterPlotController<Row, Column> {
    override var view: ScatterPlotView<Row, Column>? = null
    set(value) {
        if (field != null) {
//            this.view.removeMouseListener(mouseListener);
//            this.view.removeMouseMotionListener(mouseListener);
        }
        field = value
        if (value != null) {
            value.addKeyListener(keyListener)
            value.addMouseListener(mouseListener)
            value.addMouseMotionListener(mouseListener)
            value.addMouseWheelListener(mouseListener)
        }
    }

    /**
     * Interaction mode.
     */
    enum class Mode {
        /**
         * Selection mode.
         */
        Selection,

        /**
         * Zooming mode.
         */
        Zooming,

        /**
         * Panning mode.
         */
        Panning
    }

    private var mode = Mode.Selection
    private val multipleSelectionEnabled = true
    private val selectOnPopupTrigger = false
    private val improvedBorderZooming = false
    private val mouseListener: DefaultMouseListener = DefaultMouseListener()
    private val keyListener: KeyListener = object : KeyListener {
        var oldMode: Mode? = null
        override fun keyTyped(event: KeyEvent) {}
        override fun keyPressed(event: KeyEvent) {
            if (event.keyCode == KeyEvent.VK_S) {
                if (oldMode == null) {
                    oldMode = mode
                }
                mode = Mode.Selection
            }
            if (event.keyCode == KeyEvent.VK_Z ||
                event.keyCode == KeyEvent.VK_SHIFT
            ) {
                if (oldMode == null) {
                    oldMode = mode
                }
                mode = Mode.Zooming
            }
            if (event.keyCode == KeyEvent.VK_P ||
                event.keyCode == KeyEvent.VK_SPACE
            ) {
                if (oldMode == null) {
                    oldMode = mode
                }
                mode = Mode.Panning
            }
        }

        override fun keyReleased(event: KeyEvent) {
            if (oldMode != null) {
                mode = oldMode!!
                oldMode = null
            }
        }
    }

    private fun zoom(amount: Double, xCenter: Double, yCenter: Double) {
        val viewport: Rectangle2D = getViewport()
        val world: Rectangle2D = getWorld()

//        if (improvedBorderZooming) {
//            final double normalizedXCenter = normalize(xCenter, world.minX, world.maxX);
//            final double normalizedYCenter = normalize(yCenter, world.minY, world.maxY);
//
//            xCenter = denormalize(zoomingCenterFunction(normalizedXCenter), world.minX, world.maxX);
//            yCenter = denormalize(zoomingCenterFunction(normalizedYCenter), world.minY, world.maxY);
//        }
//
        val xRange: Range =
            Range(amount, xCenter, viewport.minX, viewport.width, world.minX, world.maxX, view!!.getXRangeModel().maximumExtent).invoke()
        val xMin: Double = xRange.getMin()
        val xMax: Double = xRange.getMax()
        val yRange: Range =
            Range(amount, yCenter, viewport.minY, viewport.height, world.minY, world.maxY, view!!.getYRangeModel().maximumExtent).invoke()
        val yMin: Double = yRange.getMin()
        val yMax: Double = yRange.getMax()
        if (xMin < xMax && yMin < yMax) {
            view!!.zoom(false, xMin, xMax, yMin, yMax)
        }
    }

    fun getViewport(): Rectangle2D {
        return Rectangle2D.Double(
            view!!.getXRangeModel().start,
            view!!.getYRangeModel().start,
            view!!.getXRangeModel().extent,
            view!!.getYRangeModel().extent
        )
    }

    fun getWorld(): Rectangle2D {
        return Rectangle2D.Double(
            view!!.getXRangeModel().minimum,
            view!!.getYRangeModel().minimum,
            view!!.getXRangeModel().maximum - view!!.getXRangeModel().minimum,
            view!!.getYRangeModel().maximum - view!!.getYRangeModel().minimum
        )
    }

    private inner class Range(
        private val _amount: Double,
        private val _center: Double?,
        private val _start: Double,
        private val _extent: Double,
        private val _minRange: Double,
        private val _maxRange: Double,
        private val _maxExtent: Double
    ) {
        private var _min = 0.0
        private var _max = 0.0
        fun getMin(): Double {
            return _min
        }

        fun getMax(): Double {
            return _max
        }

        operator fun invoke(): Range {
            val range = _extent
            val newRange: Double = min(_maxExtent, range + range * _amount)
            if (_center != null) {
                _min = _center - newRange * ((_center - _start) / range)
                _max = _center + newRange * ((_start + _extent - _center) / range)
            } else {
                _min = _start
                _max = _start + newRange
            }
            if (_min < _minRange) {
                _max = _max + (_minRange - _min)
                _min = _minRange
            }
            if (_max > _maxRange) {
                _min = _min - (_max - _maxRange)
                _max = _maxRange
            }
            _min = max(_minRange, _min)
            _max = min(_maxRange, _max)
            return this
        }
    }

    inner class DefaultMouseListener : MouseListener, MouseMotionListener, MouseWheelListener {
        private var last: MouseEvent? = null
        private var selectOnRelease: Row? = null
        private var zoomOnRelease: Point? = null
        private val zoomingPoint: Point? = null
        private var pressedPoint: Point? = null
        override fun mouseClicked(event: MouseEvent) {}
        override fun mousePressed(event: MouseEvent) {
            if (view != null) {
                pressedPoint = Point(event.x, event.y)
                val bestrow: Row? = view!!.getClosestRow(event.x, event.y)
                if (bestrow != null) {
                    if (!event.isMultipleSelectionKey) {
                        if (!view!!.model!!.getSelection().isSelected(bestrow) ||
                            view!!.model!!.getSelection().selectedCount > 1
                        ) {
                            view!!.model!!.getSelection().clearSelection()
                            view!!.model!!.getSelection().setSelectedElement(bestrow)
                            view!!.setSelectionMode(view!!.model!!.getSelection().isSelected(bestrow))
                        }
                    } else {
                        view!!.setSelectionMode(!view!!.model!!.getSelection().isSelected(bestrow))
                        if (view!!.isSelectionMode()) {
                            view!!.model!!.getSelection().setSelectedState(bestrow, true)
                        } else {
                            view!!.model!!.getSelection().setSelectedState(bestrow, false)
                        }
                    }
                } else {
                    view!!.model!!.getSelection().clearSelection()
                }
            }
        }

        override fun mouseReleased(event: MouseEvent) {
            if (view != null) {
                if (!event.isPopupTrigger) {
                    if (selectOnRelease != null) {
                        if (event.isLeftMouseButton) {
                            view!!.model!!.getSelection().clearSelection()
                            if (multipleSelectionEnabled) {
                                addToSelection(selectOnRelease!!)
                            } else {
                                setSelection(selectOnRelease)
                            }
                            selectOnRelease = null
                        }
                    }
                    if (zoomOnRelease != null) {
                        if (event.isLeftMouseButton) {
//                            zoom(-1.0 / 20.0, _view.screenToWorldX(zoomOnRelease.x), _view.screenToWorldY(zoomOnRelease.y));
                        } else {
//                            zoom(1.0 / 20.0, _view.screenToWorldX(zoomOnRelease.x), _view.screenToWorldY(zoomOnRelease.y));
                        }
                        zoomOnRelease = null
                    }
                    view!!.getRubberBand().stopRubberBand()
                } else {
                    if (selectOnPopupTrigger) {
                        val node: Row? = view!!.getClosestRow(event.x, event.y)
                        setSelection(node)
                    }

//                    if (popupMenu != null) {
//                        popupMenu.show(_view, e.x, e.y);
//                    }
                }
                last = null
            }
        }

        override fun mouseEntered(event: MouseEvent) {
            if (view != null) {
                if (view!!.model != null) {
                    view!!.model!!.getProbing().selected = view!!.getClosestRow(event.x, event.y)
                }
            }
        }

        override fun mouseExited(event: MouseEvent) {
            if (view != null) {
                if (view!!.model != null) {
                    view!!.model!!.getProbing().clearSelection()
                }
            }
        }

        override fun mouseDragged(event: MouseEvent) {
            if (view != null) {
                if (view!!.model != null) {
                    if (last != null) {
                        when (mode) {
                            Mode.Selection -> if (event.isLeftMouseButton) {
                                if (event.isAltKeyDown || event.isShiftKeyDown) {
                                    view!!.model!!.getProbing().clearSelection()
                                    if (multipleSelectionEnabled) {
                                        if (!view!!.getRubberBand().rubberBand.isActive) {
                                            view!!.getRubberBand().startRubberBand(pressedPoint!!.ix, pressedPoint!!.iy)
                                        }
                                        view!!.getRubberBand().stretchRubberBand(event.x, event.y)
                                        val rect: Rectangle2D? = view!!.getRubberBand().rubberBandScreen
                                        if (rect != null) {
                                            val toSelect: List<Row> = view!!.getRows(rect)
                                            if (event.isMultipleSelectionKey) {
                                                addToSelection(toSelect)
                                            } else {
                                                view!!.model!!.getSelection().setSelectedIterable(toSelect)
                                            }
                                            selectOnRelease = null
                                        }
                                    }
                                } else {
                                    val observation: Row? = view!!.getClosestRow(event.x, event.y)
                                    view!!.model!!.getProbing().selected = observation
                                    selectOnRelease = null
                                    val isAlreadySelected = view!!.model!!.getSelection() != null && view!!.model!!.getSelection().isSelected(observation)
                                    if (!isAlreadySelected && !event.isMenuShortcutKeyDown) {
                                        view!!.model!!.getSelection().clearSelection()
                                    }
                                    if (event.isMenuShortcutKeyDown) {
                                        if (multipleSelectionEnabled) {
                                            addToSelection(observation!!)
                                        } else {
                                            setSelection(observation)
                                        }
                                    } else {
                                        if (isAlreadySelected) {
                                            selectOnRelease = observation
                                        } else {
                                            if (multipleSelectionEnabled) {
                                                addToSelection(observation!!)
                                            } else {
                                                setSelection(observation)
                                            }
                                        }
                                    }
                                }
                            }
                            Mode.Panning -> if (event.isLeftMouseButton) {
                                val viewport: Rectangle2D = getViewport()
                                val world: Rectangle2D = getWorld()
                                val x1: Double = view!!.screenTransform.x.screenToWorld(last!!.x)
                                val x2: Double = view!!.screenTransform.x.screenToWorld(event.x)
                                val worldDiffX = x1 - x2
                                var xMin: Double = viewport.minX + worldDiffX
                                var xMax: Double = viewport.maxX + worldDiffX
                                val y1: Double = view!!.screenTransform.y.screenToWorld(last!!.y)
                                val y2: Double = view!!.screenTransform.y.screenToWorld(event.y)
                                val worldDiffY = y1 - y2
                                var yMin: Double = viewport.minY + worldDiffY
                                var yMax: Double = viewport.maxY + worldDiffY
                                if (xMin < world.minX) {
                                    xMax = xMax + (world.minX - xMin)
                                    xMin = world.minX
                                }
                                if (xMax > world.maxX) {
                                    xMin = xMin - (xMax - world.maxX)
                                    xMax = world.maxX
                                }
                                xMin = max(world.minX, xMin)
                                xMax = min(world.maxX, xMax)
                                if (yMin < world.minY) {
                                    yMax = yMax + (world.minY - yMin)
                                    yMin = world.minY
                                }
                                if (yMax > world.maxY) {
                                    yMin = yMin - (yMax - world.maxY)
                                    yMax = world.maxY
                                }
                                yMin = max(world.minY, yMin)
                                yMax = min(world.maxY, yMax)
                                if (xMin < xMax && yMin < yMax) {
                                    view!!.zoom(false, xMin, xMax, yMin, yMax)
                                }
                            }
                            Mode.Zooming -> if (event.isMiddleMouseButton) {
                                zoom(
                                    event.y - last!!.y / 100.0, view!!.screenTransform.x.screenToWorld(event.x),
                                    view!!.screenTransform.y.screenToWorld(event.y)
                                )
                                zoomOnRelease = null
                            }
                        }
                    }
                    last = event
                }
            }
        }

        override fun mouseMoved(event: MouseEvent) {
            val closest: Row? = view!!.getClosestRow(event.x, event.y)
            view!!.model!!.getProbing().selected = closest
        }

        override fun mouseWheelMoved(event: MouseWheelEvent) {
            if (view != null) {
                val amount: Double = event.wheelRotation
                zoom(
                    amount / 500.0, view!!.screenTransform.x.screenToWorld(event.x),
                    view!!.screenTransform.y.screenToWorld(event.y)
                )
            }
        }

        private fun removeFromSelection(observation: Row) {
            view!!.model!!.getSelection().setSelectedState(observation, false)
        }

        private fun setSelection(observation: Row?) {
            if (observation != null) {
                view!!.model!!.getSelection().setSelectedElements(observation)
            } else {
                view!!.model!!.getSelection().clearSelection()
            }
        }

        private fun addToSelection(observation: Row) {
            view!!.model!!.getSelection().setSelectedState(observation, true)
        }

        private fun addToSelection(newSelection: List<Row>) {
            view!!.model!!.getSelection().setSelectedIterableState(newSelection, true)
        }
    }

    init {
        this.view = view
    }
}