/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.scatterplot

import com.macrofocus.common.command.Future
import com.macrofocus.common.concurrent.Callable
import com.macrofocus.common.concurrent.ExecutorService
import com.macrofocus.common.concurrent.Runtime
import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.interval.BoundedIntervalEvent
import com.macrofocus.common.interval.BoundedIntervalListener
import com.macrofocus.common.interval.MutableBoundedInterval
import com.macrofocus.common.interval.SimpleBoundedInterval
import com.macrofocus.common.properties.*
import com.macrofocus.common.selection.*
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import com.macrofocus.common.transform.RangesScreenTransformCoordinator
import com.macrofocus.common.transform.ScreenTransformEvent
import com.macrofocus.common.transform.ScreenTransformListener
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.plot.guide.Guide
import org.mkui.canvas.*
import org.mkui.color.MkColor
import org.mkui.color.MkColorFactory
import org.mkui.color.colorOf
import org.mkui.geom.Point
import org.mkui.geom.Point2D
import org.mkui.geom.Rectangle
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.AbstractIDrawing
import org.mkui.graphics.IDrawing
import org.mkui.graphics.IGraphics
import org.mkui.graphics.colortheme.ColorTheme
import org.mkui.graphics.colortheme.LightColorTheme
import org.mkui.graphics.pressure.LinearPressure
import org.mkui.palette.FixedPalette
import org.mkui.rubberband.RubberBand
import org.mkui.rubberband.RubberbandDrawing
import org.mkui.rubberband.ScreenTransformRubberBand
import org.mkui.transform.MutableTwoDScreenTransform
import org.mkui.transform.SimpleTwoDScreenTransform
import org.mkui.visual.VisualLayer
import org.mkui.visual.VisualListener
import kotlin.math.PI
import kotlin.math.max
import kotlin.math.min
import kotlin.math.sqrt

abstract class AbstractScatterPlotView<Row, Column>() : ScatterPlotView<Row, Column> {
    override var model: ScatterPlotModel<Row, Column>? = null
    set(value) {
        if (field != null) {
            field!!.removeScatterPlotListener(listener)
            if (field!!.getX() != null) {
                field!!.getX().removeSingleSelectionListener(axisSelectionListener)
            }
            if (field!!.getY() != null) field!!.getY().removeSingleSelectionListener(axisSelectionListener)
        }
        field = value
        createOverplots()
        if (field != null) {
            field!!.addScatterPlotListener(listener)
            if (field!!.getX() != null) {
                val x: MutableSingleSelection<AxisModel<Row, Column>> = field!!.getX()
                x.addSingleSelectionListener(axisSelectionListener)
                if (x.isActive && x.selected!!.interval != null) {
                    x.selected!!.interval.addBoundedIntervalListener(intervalListener)
                }
            }
            if (field!!.getY() != null) {
                val y: MutableSingleSelection<AxisModel<Row, Column>> = field!!.getY()
                y.addSingleSelectionListener(axisSelectionListener)
                if (y.isActive && y.selected!!.interval != null) {
                    y.selected!!.interval.addBoundedIntervalListener(intervalListener)
                }
            }
        }
        updateIntervals()
        updateAxisType()
        resetXAxis()
        resetYAxis()
        timer.restart()
    }

    override val properties: MutableProperties<ScatterPlotView.PropertyType> = EnumProperties<ScatterPlotView.PropertyType>(ScatterPlotView.PropertyType.values())
    protected val canvas: CPCanvas by lazy { CPCanvas() }
    protected val timer: CPTimer
    private val xRangeModel: MutableBoundedInterval
    private val yRangeModel: MutableBoundedInterval
    override val screenTransform: MutableTwoDScreenTransform
    protected var rangesScreenTransformCoordinator: RangesScreenTransformCoordinator? = null
    override var xGuide: Guide? = null
    override var yGuide: Guide? = null
    private var isSelectionMode = true
    private val rubberBand: RubberBand
    private val propertiesListener: PropertiesListener<ScatterPlotView.PropertyType> = object : PropertiesListener<ScatterPlotView.PropertyType> {
        override fun propertyChanged(name: ScatterPlotView.PropertyType, event: PropertyEvent<Any?>) {
            createOverplots()
            scheduleUpdate()
        }
    }
    private val listener: ScatterPlotListener = object : ScatterPlotListener {
        override fun scatterPlotChanged() {
            resetXAxis()
            resetYAxis()
            scheduleUpdate()
        }
    }
    private val intervalListener: BoundedIntervalListener = object : BoundedIntervalListener {
        override fun boundedIntervalChanged(event: BoundedIntervalEvent) {
            updateIntervals()
        }
    }
    private val axisSelectionListener: SingleSelectionListener<AxisModel<Row, Column>> = object : SingleSelectionListener<AxisModel<Row, Column>> {
        override fun selectionChanged(event: SingleSelectionEvent<AxisModel<Row, Column>>) {
            if (event.previousSelection != null && event.previousSelection!!.interval != null) {
                event.previousSelection!!.interval.removeBoundedIntervalListener(intervalListener)
            }
            if (event.currentSelection != null && event.currentSelection!!.interval != null) {
                event.currentSelection!!.interval.addBoundedIntervalListener(intervalListener)
            }
            updateAxisType()
            updateIntervals()
        }
    }

    protected abstract fun getWidth(): Int
    protected abstract fun getHeight(): Int
    protected abstract fun refresh()
    protected abstract fun repaint()
    protected abstract fun scheduleUpdate()
    override fun addMouseListener(l: MouseListener) {
        canvas.addMouseListener(l)
    }

    override fun addMouseMotionListener(l: MouseMotionListener) {
        canvas.addMouseMotionListener(l)
    }

    override fun addMouseWheelListener(l: MouseWheelListener) {
        canvas.addMouseWheelListener(l)
    }

    override fun addKeyListener(listener: KeyListener) {
        canvas.addKeyListener(listener)
    }

    fun createOverplots() {
        if (model != null) {
            canvas.removeAllLayers()
            populateOverplots()
        }
    }

    protected fun populateOverplots() {
        val filteredDrawing: IDrawing = object : AbstractVisualLayerIDrawing(model!!.getVisual().filtered) {
            override val isActive: Boolean
                get() = getShowFiltered().value && super.isActive

            override fun draw(g: IGraphics, row: Row, width: Double, height: Double) {
                val mp: Point? = getPoint(AxisModel.DATA, row)
                if (mp != null) {
                    if (model!!.getSizeAxisModel() != null) {
                        val size = getMarkerSize(AxisModel.DATA, row)
                        val radius = size / 2
                        g.fillCircle(Rectangle2D.Double(mp.x - radius, mp.y - radius, size.toDouble(), size.toDouble()))
                    } else {
                        val size: Int = max(1, getMarkerSize(AxisModel.DATA, row) / 2)
                        val radius = size / 2
                        g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                    }
                }
            }
        }
        val visibleDrawing: IDrawing = object : AbstractVisualLayerIDrawing(model!!.getVisual().visible) {
            override fun draw(g: IGraphics, row: Row, width: Double, height: Double) {
                val mp: Point? = getPoint(AxisModel.DATA, row)
                if (mp != null) {
                    val size = getMarkerSize(AxisModel.DATA, row)
                    val radius = size / 2
                    g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                }
            }
        }
        val visibleDrawing2: IDrawing = object : AbstractVisualLayerIDrawing(model!!.getVisual().visible) {
            override fun draw(g: IGraphics, row: Row, width: Double, height: Double) {
                val mp: Point? = getPoint2(AxisModel.DATA, row)
                if (mp != null) {
                    val size = getMarkerSize(AxisModel.DATA, row)
                    val radius = size / 2
                    g.setColor(MkColorFactory.instance.darkGray)
                    g.drawLine(mp.x - radius, mp.y, mp.x + radius, mp.y)
                }
            }
        }
        val colorMappedDrawing: IDrawing = object : AbstractVisualLayerIDrawing(model!!.getVisual().colorMapped) {
            override fun draw(g: IGraphics, row: Row, width: Double, height: Double) {
                val mp: Point? = getPoint(AxisModel.DATA, row)
                if (mp != null) {
                    g.setColor(model!!.getColorMapping().getColor(row)!!)
                    val size = getMarkerSize(AxisModel.DATA, row)
                    val radius = size / 2
                    g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                }
            }
        }
        val coloredDrawing: IDrawing = object : AbstractVisualLayerIDrawing(model!!.getVisual().colored) {
            override fun draw(g: IGraphics, row: Row, width: Double, height: Double) {
                val mp: Point? = getPoint(AxisModel.DATA, row)
                if (mp != null) {
                    g.setColor(model!!.getColoring().getColor(row)!!)
                    val size = getMarkerSize(AxisModel.DATA, row)
                    val radius = size / 2
                    g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                }
            }
        }
        val singleSelectedDrawing: IDrawing = object : AbstractVisualLayerIDrawing(model!!.getVisual().singleSelected) {
            override fun draw(g: IGraphics, row: Row, width: Double, height: Double) {
                val mp: Point? = getPoint(AxisModel.DATA, row)
                if (mp != null) {
                    g.setColor(getColorTheme().value.selection)
                    if (model!!.getSizeAxisModel() != null) {
                        // ToDo: Fix bigger so that they occlude the density version which is not pixel perfect
                        val size = getMarkerSize(AxisModel.DATA, row) + 2
                        val radius = size / 2
                        g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                    } else {
                        val size = getMarkerSize(AxisModel.DATA, row) + 2
                        val radius = size / 2
                        g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                    }
                }
            }
        }
        val singleSelectedDrawing2: IDrawing = object : AbstractVisualLayerIDrawing(model!!.getVisual().singleSelected) {
            override fun draw(g: IGraphics, row: Row, width: Double, height: Double) {
                val mp: Point? = getPoint2(AxisModel.DATA, row)
                if (mp != null) {
                    g.setColor(getColorTheme().value.selection)
                    if (model!!.getSizeAxisModel() != null) {
                        // ToDo: Fix bigger so that they occlude the density version which is not pixel perfect
                        val size = getMarkerSize(AxisModel.DATA, row) + 2
                        val radius = size / 2
                        g.drawLine(mp.x - radius, mp.y, mp.x + radius, mp.y)
                    } else {
                        val size = getMarkerSize(AxisModel.DATA, row) + 2
                        val radius = size / 2
                        g.drawLine(mp.x - radius, mp.y, mp.x + radius, mp.y)
                    }
                }
            }
        }
        val multiSelectedDrawing: IDrawing = object : AbstractVisualLayerIDrawing(model!!.getVisual().multipleSelected) {
            override fun draw(g: IGraphics, row: Row, width: Double, height: Double) {
                val mp: Point? = getPoint(AxisModel.DATA, row)
                if (mp != null) {
                    g.setColor(getColorTheme().value.selection)
                    if (model!!.getSizeAxisModel() != null) {
                        // ToDo: Fix bigger so that they occlude the density version which is not pixel perfect
                        val size = getMarkerSize(AxisModel.DATA, row) + 2
                        val radius = size / 2
                        g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                    } else {
                        val size = getMarkerSize(AxisModel.DATA, row) + 2
                        val radius = size / 2
                        g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                    }
                }
            }
        }
        val probedDrawing: IDrawing = object : AbstractVisualLayerIDrawing(model!!.getVisual().probed) {
            override fun draw(g: IGraphics, row: Row, width: Double, height: Double) {
                val mp: Point? = getPoint(AxisModel.DATA, row)
                if (mp != null) {
                    if (model!!.getSizeAxisModel() != null) {
                        // ToDo: Fix bigger so that they occlude the density version which is not pixel perfect
                        run {
                            g.setColor(getColorTheme().value.probing)
                            val size = getMarkerSize(AxisModel.DATA, row) + 2
                            val radius = size / 2
                            g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                        }
                        if (model!!.getSelection().isSelected(row)) {
                            g.setColor(getColorTheme().value.selection)
                            val size = getMarkerSize(AxisModel.DATA, row)
                            val radius = size / 2
                            g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                        }
                    } else {
                        run {
                            g.setColor(getColorTheme().value.probing)
                            val size = getMarkerSize(AxisModel.DATA, row) + 2
                            val radius = size / 2
                            g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                        }
                        if (model!!.getSelection().isSelected(row)) {
                            g.setColor(getColorTheme().value.selection)
                            val size = getMarkerSize(AxisModel.DATA, row)
                            val radius = size / 2
                            g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                        }
                    }
                }
            }
        }
        canvas.addLayer(object : AbstractIDrawing() {
            override fun draw(g: IGraphics, point: Point2D?, width: Double, height: Double, clipBounds: Rectangle) {
                val background: MkColor = getColorTheme().value.background
                if (background != null) {
                    g.setColor(background)
                    g.fillRectangle2D(Rectangle2D.Double(0.0, 0.0, width, height))
                }
            }
        })
        if (getShowFiltered().value) {
            canvas.addDensityLayer(getRendering().value, filteredDrawing, LinearPressure(), object : PaletteProvider {
                override val palette: FixedPalette
                    get() = getColorTheme().value.ghostedPalette
            })
        }
        canvas.addDensityLayer(getRendering().value, visibleDrawing, LinearPressure(), object : PaletteProvider {
            override val palette: FixedPalette
                get() = getColorTheme().value.visiblePalette
        })
        canvas.addAveragingLayer(getRendering().value, colorMappedDrawing)
        canvas.addBufferedLayer(coloredDrawing)
        canvas.addDensityLayer(getRendering().value, multiSelectedDrawing, LinearPressure(), object : PaletteProvider {
            override val palette: FixedPalette
                get() = getColorTheme().value.selectedPalette
        })
        canvas.addLayer(singleSelectedDrawing)
        canvas.addLayer(probedDrawing)
        canvas.addDensityLayer(getRendering().value, visibleDrawing2, LinearPressure(), object : PaletteProvider {
            override val palette: FixedPalette
                get() = getColorTheme().value.visiblePalette
        })
        canvas.addLayer(singleSelectedDrawing2)
        canvas.addLayer(AnnotationIDrawing(model!!.getVisual().annotationProbing, colorOf(255, 150, 0, 200)))
        canvas.addLayer(AnnotationIDrawing(model!!.getVisual().annotationSelection, colorOf(255, 100, 0, 200)))
        canvas.addLayer(object : RubberbandDrawing(getRubberBand()) {
            override val colorTheme: Property<ColorTheme>
                get() = this@AbstractScatterPlotView.getColorTheme()
        })
    }

    override fun getXRangeModel(): MutableBoundedInterval {
        return xRangeModel
    }

    override fun getYRangeModel(): MutableBoundedInterval {
        return yRangeModel
    }

    override fun getRubberBand(): RubberBand {
        return rubberBand
    }

    fun resetXAxis() {
        if (model!!.getXAxisModel() != null) {
            val min: Double = model!!.getXAxisModel()!!.minimum
            val max: Double = model!!.getXAxisModel()!!.maximum
            xRangeModel.setMinMax(min, max)
            xRangeModel.setValue(min, max - min)
        }
    }

    fun resetYAxis() {
        if (model!!.getYAxisModel() != null) {
            val min: Double = model!!.getYAxisModel()!!.minimum
            val max: Double = model!!.getYAxisModel()!!.maximum
            yRangeModel.setMinMax(min, max)
            yRangeModel.setValue(min, max - min)
        }
    }

    /**
     * {@inheritDoc}
     */
    override fun zoom(animate: Boolean, x1: Double, x2: Double, y1: Double, y2: Double) {
        getXRangeModel().setValue(x1, x2 - x1)
        getYRangeModel().setValue(y1, y2 - y1)
    }

    protected fun updateIntervals() {
        val x: MutableSingleSelection<AxisModel<Row, Column>> = model!!.getX()
        val y: MutableSingleSelection<AxisModel<Row, Column>> = model!!.getY()
        if (x.isActive && x.selected!!.interval != null) {
            val min: Double = x.selected!!.interval.minimum
            val max: Double = x.selected!!.interval.maximum
            xRangeModel.setMinMax(min, max)
            xRangeModel.setValue(min, max - min)
        } else {
            resetXAxis()
        }
        if (y.isActive && y.selected!!.interval != null) {
            val min: Double = y.selected!!.interval.minimum
            val max: Double = y.selected!!.interval.maximum
            yRangeModel.setMinMax(min, max)
            yRangeModel.setValue(min, max - min)
        } else {
            resetYAxis()
        }

//        screenTransform.setIntervals(x.getSelected()interval, y.getSelected()interval);
        rangesScreenTransformCoordinator?.setRanges(xRangeModel, yRangeModel)
        scheduleUpdate()
    }

    protected abstract fun updateAxisType()
    override fun getClosestRow(x: Int, y: Int): Row? {
        var bestdistance = Double.MAX_VALUE
        var bestrow: Row? = null
        val visible: VisualLayer<Row> = model!!.getVisual().active
        for (row in visible) {
            val p: Point2D? = getPoint(AxisModel.DATA, row)
            if (p != null) {
                val distance: Double = Point2D.distance(p.x, p.y, x.toDouble(), y.toDouble())
                if (distance < bestdistance) {
                    bestdistance = distance
                    bestrow = row
                }
            }
        }
        return bestrow
    }

    override fun getRows(rect: Rectangle2D): List<Row> {
        val visible: VisualLayer<Row> = model!!.getVisual().active
        val list: MutableList<Row> = ArrayList<Row>()
        for (row in visible) {
            val p: Point2D? = getPoint(AxisModel.DATA, row)
            if (p != null) {
                if (rect.contains(p)) {
                    list.add(row)
                }
            }
        }
        return list
    }

    override fun getShowFiltered(): MutableProperty<Boolean> {
        return properties.getProperty(ScatterPlotView.PropertyType.ShowFiltered) as MutableProperty<Boolean>
    }

    override fun getShowGrid(): MutableProperty<Boolean> {
        return properties.getProperty(ScatterPlotView.PropertyType.ShowGrid) as MutableProperty<Boolean>
    }

    override fun getAntialiasing(): MutableProperty<Boolean> {
        return properties.getProperty(ScatterPlotView.PropertyType.Antialiasing) as MutableProperty<Boolean>
    }

    override fun getRendering(): MutableProperty<Rendering> {
        return properties.getProperty(ScatterPlotView.PropertyType.Rendering) as MutableProperty<Rendering>
    }

    override fun getGridThickness(): MutableProperty<Float?> {
        return properties.getProperty(ScatterPlotView.PropertyType.GridThickness) as MutableProperty<Float?>
    }

    override fun getShowColor(): MutableProperty<MkColor?> {
        return properties.getProperty(ScatterPlotView.PropertyType.ShowColor) as MutableProperty<MkColor?>
    }

    override fun getRegression(): MutableProperty<Boolean> {
        return properties.getProperty(ScatterPlotView.PropertyType.Regression) as MutableProperty<Boolean>
    }

    override fun getMarkerSize(): MutableProperty<Double?> {
        return properties.getProperty(ScatterPlotView.PropertyType.MarkerSize) as MutableProperty<Double?>
    }

    override fun getColorTheme(): MutableProperty<ColorTheme> {
        return properties.getProperty(ScatterPlotView.PropertyType.ColorTheme) as MutableProperty<ColorTheme>
    }

    override fun getDisplayLabels(): MutableProperty<Boolean> {
        return properties.getProperty(ScatterPlotView.PropertyType.DisplayLabels) as MutableProperty<Boolean>
    }

    override fun setShowFiltered(showFiltered: MutableProperty<Boolean>?) {
        properties.replaceProperty<Boolean>(ScatterPlotView.PropertyType.ShowFiltered, showFiltered!!)
    }

    override fun setRendering(rendering: MutableProperty<Rendering>?) {
        properties.replaceProperty<Rendering>(ScatterPlotView.PropertyType.Rendering, rendering!!)
    }

    override fun setAntialiasing(antialiasing: MutableProperty<Boolean>?) {
        properties.replaceProperty<Boolean>(ScatterPlotView.PropertyType.Antialiasing, antialiasing!!)
    }

    override fun setColorTheme(colorTheme: MutableProperty<ColorTheme>?) {
        properties.replaceProperty<ColorTheme>(ScatterPlotView.PropertyType.ColorTheme, colorTheme!!)
    }

    override fun isSelectionMode(): Boolean {
        return isSelectionMode
    }

    override fun setSelectionMode(value: Boolean) {
        isSelectionMode = value
    }

    inner abstract class AbstractVisualLayerIDrawing protected constructor(visualLayer: VisualLayer<Row>) : AbstractIDrawing() {
        private val visualLayer: VisualLayer<Row>
        override val isActive: Boolean
            get() = visualLayer.isActive

        override fun draw(g: IGraphics, point: Point2D?, width: Double, height: Double, clipBounds: Rectangle) {
            if (visualLayer.objectCount > 0) {
                if (!MULTITHREADED || !g.isThreadSafe()) {
                    for (row in visualLayer) {
                        draw(g, row, width, height)
                    }
                } else {
                    val nTasks: Int =
                        min(Runtime.getRuntime().availableProcessors(), visualLayer.objectCount) // Determine number of tasks i.e. threads
                    val nRowsPerTask: Int = visualLayer.objectCount / nTasks
                    val todo: MutableList<Callable<Any?>> = ArrayList<Callable<Any?>>(nTasks)
                    for (nTask in 0 until nTasks)  // Iterate over all tasks
                    {
                        val fromRow = nTask * nRowsPerTask // the tasks first row
                        val toRow = if (nTask < nTasks - 1) fromRow + nRowsPerTask else visualLayer.objectCount // the tasks last row
                        todo.add(object : Callable<Any?> {
                            @Throws(Exception::class)
                            override fun call(): Any? {
                                for (row in visualLayer.iterable(
                                    // Create and define the task
                                    fromRow, toRow - 1
                                )) {
                                    draw(g, row, width, height)
                                }
                                return null
                            }
                        })
                    }
                    try {
                        val answers: List<Future<Any?>> = executor!!.invokeAll(todo)
                        for (answer in answers) {
                            try {
                                answer.get()
                            } catch (e: Exception) {
                                e.printStackTrace()
                            }
                        }
                    } catch (e: Exception) {
                        e.printStackTrace() // something wrong, maybe
                    }
                }
            }
        }

        abstract fun draw(g: IGraphics, row: Row, width: Double, height: Double)

        init {
            this.visualLayer = visualLayer
            visualLayer.addVisualListener(object : VisualListener {
                override fun visualChanged() {
                    notifyIDrawingChanged()
                }
            })
        }
    }

    fun getPoint(layer: Int, row: Row): Point? {
        return if (model!!.getXAxisModel() != null && model!!.getYAxisModel() != null) {
            val v1: Number? = model?.getXAxisModel()?.getValue(layer, row)
            val v2: Number? = model?.getYAxisModel()?.getValue(layer, row)
            if (v1 != null && v2 != null) {
                Point(screenTransform.x.worldToScreen(v1.toDouble()), screenTransform.y.worldToScreen(v2.toDouble()))
            } else {
                null
            }
        } else {
            null
        }
    }

    fun getPoint2(layer: Int, row: Row): Point? {
        return if (model!!.getXAxisModel() != null && model!!.getY2AxisModel() != null) {
            val v1: Number? = model?.getXAxisModel()?.getValue(layer, row)
            val v2: Number? = model?.getY2AxisModel()?.getValue(layer, row)
            if (v1 != null && v2 != null) {
                Point(screenTransform.x.worldToScreen(v1.toDouble()), screenTransform.y.worldToScreen(v2.toDouble()))
            } else {
                null
            }
        } else {
            null
        }
    }

    fun getXLine(layer: Int, row: Row): Int? {
        return if (model!!.getXAxisModel() != null) {
            val v1: Number? = model?.getXAxisModel()?.getValue(layer, row)
            if (v1 != null) {
                screenTransform.x.worldToScreen(v1.toDouble())
            } else {
                null
            }
        } else {
            null
        }
    }

    fun getYLine(layer: Int, row: Row): Int? {
        return if (model!!.getYAxisModel() != null) {
            val v2: Number? = model?.getYAxisModel()?.getValue(layer, row)
            if (v2 != null) {
                screenTransform.y.worldToScreen(v2.toDouble())
            } else {
                null
            }
        } else {
            null
        }
    }

    private fun getMarkerSize(layer: Int, row: Row): Int {
        val sizeAxisModel: AxisModel<Row,*>? = model!!.getSizeAxisModel()
        return if (sizeAxisModel != null) {
            val v1: Number? = sizeAxisModel.getValue(layer, row)
            if (v1 != null) {
                // Values between 1 and markerSize
                val range: Double = 2 * sqrt((sizeAxisModel.maximum - sizeAxisModel.minimum) / PI)
                val v: Double = 2 * sqrt((v1.toDouble() - sizeAxisModel.minimum) / PI)
                (v * (getMarkerSize().value!!.toInt() - 1.0) / range) as Int + 1
            } else {
                0
            }
        } else {
            getMarkerSize().value!!.toInt()
        }
    }

    private inner class AnnotationIDrawing(private val selection: Selection<Any>, color: MkColor) : AbstractIDrawing() {
        private val color: MkColor
        private val selectionListener: SelectionListener<Any> = object : SelectionListener<Any> {
            override fun selectionChanged(event: SelectionEvent<Any>) {
                notifyIDrawingChanged()
            }
        }

        override val isActive: Boolean
            get() = true

        override fun draw(g: IGraphics, point: Point2D?, width: Double, height: Double, clipBounds: Rectangle) {
            g.setColor(color)
            for (row in selection) {
                val mp: Point? = getPoint(AxisModel.ANNOTATION, row as Row)
                if (mp != null) {
                    g.drawLine(mp.ix, 0, mp.ix, height.toInt())
                    g.drawLine(0, mp.iy, width.toInt(), mp.iy)
                    if (model!!.getSizeAxisModel() != null) {
                        val size = getMarkerSize(AxisModel.ANNOTATION, row)
                        val radius = size / 2
                        g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                    } else {
                        val size = getMarkerSize(AxisModel.ANNOTATION, row) + 2
                        val radius = size / 2
                        g.fillCircle(Rectangle(mp.ix - radius, mp.iy - radius, size, size))
                    }
                } else {
                    val y = getYLine(AxisModel.ANNOTATION, row)
                    if (y != null) {
                        g.drawLine(0, y, width.toInt(), y)
                    } else {
                        val x = getXLine(AxisModel.ANNOTATION, row)
                        if (x != null) {
                            g.drawLine(x, 0, x, height.toInt())
                        }
                    }
                }
            }
        }

        init {
            this.color = color
            selection.addWeakSelectionListener(selectionListener)
        }
    }

    companion object {
        private const val MULTITHREADED = true
        private val executor: ExecutorService? = CPHelper.instance.visualizationExecutorService()
    }

    init {
        properties.createProperty<Boolean>(ScatterPlotView.PropertyType.ShowFiltered, true)
        properties.createProperty<Boolean>(ScatterPlotView.PropertyType.ShowGrid, true)
        properties.createProperty<Boolean>(ScatterPlotView.PropertyType.Antialiasing, true)
        properties.createProperty<Any>(ScatterPlotView.PropertyType.Rendering, Rendering.AlphaBlended)
        properties.createProperty<Float>(ScatterPlotView.PropertyType.GridThickness, 0.1f)
        properties.createProperty<MkColor>(ScatterPlotView.PropertyType.ShowColor, colorOf(128, 128, 128))
        properties.createProperty<Boolean>(ScatterPlotView.PropertyType.Regression, true)
        properties.createProperty<Double>(ScatterPlotView.PropertyType.MarkerSize, 8.0)
        properties.createProperty<ColorTheme>(ScatterPlotView.PropertyType.ColorTheme, LightColorTheme())
        properties.createProperty<Boolean>(ScatterPlotView.PropertyType.DisplayLabels, true)
        properties.addPropertiesListener(propertiesListener)
        timer = CPHelper.instance.createTimer("ScatterPlotResizer", 40, true, object : CPTimerListener {
            override fun timerTriggered() {
                if (getWidth() > 0 && getHeight() > 0) {
                    refresh()
                }
            }
        })
        xRangeModel = SimpleBoundedInterval(0.0, 1.0, 0.0, 1.0)
        yRangeModel = SimpleBoundedInterval(0.0, 1.0, 0.0, 1.0)
        screenTransform = SimpleTwoDScreenTransform(xRangeModel, yRangeModel)
        screenTransform.x.setScreenMargins(8)
        screenTransform.y.setScreenMargins(8)
        rubberBand = ScreenTransformRubberBand(screenTransform)
        canvas.addCanvasListener(object : CanvasListener {
            override fun sizeChange(width: Int, height: Int) {
                screenTransform.setScreenSize(width, height)
            }
        })
        screenTransform.addScreenTransformListener(object : ScreenTransformListener {
            override fun transformChanged(event: ScreenTransformEvent) {
                timer.restart()
            }
        })
    }
}