/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.scatterplotmatrix

import com.macrofocus.common.filter.MutableFilter
import com.macrofocus.common.properties.MutableProperties
import com.macrofocus.common.properties.Property
import com.macrofocus.common.properties.SimpleProperties
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.high_d.axis.Alignment
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import com.macrofocus.high_d.axis.group.DefaultAxisGroupModel
import com.macrofocus.high_d.axis.locations.AxisLocations
import com.macrofocus.high_d.axis.locations.DefaultAxisLocations
import com.macrofocus.visual.SimpleVisual
import com.macrofocus.visual.SimpleVisualObjects
import org.mkui.coloring.MutableColoring
import org.mkui.colormapping.MutableColorMapping
import org.mkui.visual.Visual
import org.molap.dataframe.DataFrame
import kotlin.math.min

class DefaultScatterPlotMatrixModel<Row, Column>(private val dataFrame: DataFrame<Row, Column, *>, private val visual: Visual<Row, Column>, private val axisGroupModel: AxisGroupModel<Row, Column>) :
    AbstractScatterPlotMatrixModel<Row, Column>() {
    private val properties: MutableProperties<String?> = SimpleProperties()
    private val maxDimensions: Property<Int> = properties.createProperty("maxDimensions", 40)
    private val axisLocations: AxisLocations<Row, Column> = DefaultAxisLocations(axisGroupModel.axisOrder!!, maxDimensions)

    constructor(dataFrame: DataFrame<Row, Column, *>, colorMapping: MutableColorMapping<Row, Column>) : this(
        dataFrame,
        SimpleVisual<Row, Column>(SimpleVisualObjects<Row>(dataFrame), colorMapping),
    ) {
    }

    constructor(dataFrame: DataFrame<Row, Column, *>, visual: Visual<Row, Column>) : this(
        dataFrame,
        visual,
        DefaultAxisGroupModel("Root", dataFrame, null, visual.probing, visual.selection, visual.filter)
    ) {
    }

    override fun getVisual(): Visual<Row, Column> {
        return visual
    }

    /**
     * {@inheritDoc}
     */
    override fun getObjectCount(): Int {
        return dataFrame.rowCount
    }

    /**
     * {@inheritDoc}
     */
    override fun getProbing(): MutableSingleSelection<Row> {
        return visual.probing
    }

    /**
     * {@inheritDoc}
     */
    override fun getSelection(): MutableSelection<Row> {
        return visual.selection
    }

    /**
     * {@inheritDoc}
     */
    override fun getFilter(): MutableFilter<Row> {
        return visual.filter
    }

    /**
     * {@inheritDoc}
     */
    override fun getColorMapping(): MutableColorMapping<Row, Column> {
        return visual.colorMapping
    }

    /**
     * {@inheritDoc}
     */
    override fun getColoring(): MutableColoring<Row> {
        return visual.coloring
    }

    /**
     * {@inheritDoc}
     */
    override fun getObject(index: Int): Row {
        return dataFrame.getRowKey(index)
    }

    /**
     * {@inheritDoc}
     */
    override fun getAxisModel(c: Int): AxisModel<Row,Column> {
        return axisGroupModel.axisOrder!!.get(c)
    }

    /**
     * {@inheritDoc}
     */
    override fun getAxisCount(): Int {
        return min(maxDimensions.value, axisGroupModel.axisOrder!!.size())
    }

    /**
     * {@inheritDoc}
     */
    override fun getLocation(axisModel: AxisModel<Row,Column>): Double {
        return axisLocations.getLocation(Alignment.Left, axisModel)
    }

    /**
     * {@inheritDoc}
     */
    override fun getAxisGroupModel(): AxisGroupModel<Row,Column> {
        return axisGroupModel
    }

    /**
     * {@inheritDoc}
     */
    override fun setLocation(axisModel: AxisModel<Row,Column>, l: Double) {
        axisLocations.setLocation(Alignment.Left, axisModel, l)
        firedScatterPlotMatrixChanged()
    }

    fun getVisibleAxisCount(): Int {
        return axisGroupModel.axisOrder!!.size()
    }
}