/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.statistics

import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.filter.Filter
import com.macrofocus.common.filter.FilterEvent
import com.macrofocus.common.filter.FilterListener
import com.macrofocus.common.selection.*
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import org.mkui.coloring.ColoringListener
import org.mkui.coloring.MutableColoring
import org.mkui.coloring.implementation.ColoringEvent
import org.mkui.colormapping.MutableColorMapping
import org.molap.aggregates.cube.UnivariateStatistics
import org.molap.dataframe.AbstractDataFrame
import org.molap.dataframe.DataFrame
import org.molap.dataframe.DataFrameEvent
import org.molap.index.DefaultUniqueIndex
import org.molap.index.UniqueIndex
import org.molap.series.Series
import kotlin.reflect.KClass

class StatisticsDataFrame<Row, C>(
    dataFrame: DataFrame<Row, C, *>,
    probing: MutableSingleSelection<Row>,
    selection: MutableSelection<Row>,
    filter: Filter<Row>,
    colorMapping: MutableColorMapping<Row, C>,
    coloring: MutableColoring<Row>
) : AbstractDataFrame<StatisticsDataFrame.Stat, C, Any?>() {
    enum class Stat {
        ThreeSignma {
            override fun toString(): String {
                return "3σ"
            }
        },
        TwoSigma {
            override fun toString(): String {
                return "2σ"
            }
        },
        OneSigma {
            override fun toString(): String {
                return "1σ"
            }
        },
        Mean {
            override fun toString(): String {
                return "Mean"
            }
        },
        MinusOneSigma {
            override fun toString(): String {
                return "-1σ"
            }
        },
        MinusTwoSigma {
            override fun toString(): String {
                return "-2σ"
            }
        },
        MinusThreeSigma {
            override fun toString(): String {
                return "-3σ"
            }
        },
        Maximum {
            override fun toString(): String {
                return "100.0% (maximum)"
            }
        },
        P995 {
            override fun toString(): String {
                return "99.5%"
            }
        },
        P975 {
            override fun toString(): String {
                return "97.5%"
            }
        },
        P900 {
            override fun toString(): String {
                return "90.0%"
            }
        },
        P750 {
            override fun toString(): String {
                return "75.0% (quartile)"
            }
        },
        Median {
            override fun toString(): String {
                return "50.0% (median)"
            }
        },
        P250 {
            override fun toString(): String {
                return "25.0% (quartile)"
            }
        },
        P100 {
            override fun toString(): String {
                return "10.0%"
            }
        },
        P025 {
            override fun toString(): String {
                return "2.5%"
            }
        },
        P050 {
            override fun toString(): String {
                return "0.5%"
            }
        },
        Minimum {
            override fun toString(): String {
                return "0.0% (minimum)"
            }
        },
        Sum {
            override fun toString(): String {
                return "Sum"
            }
        },
        Origin {
            override fun toString(): String {
                return "Origin"
            }
        },
        N {
            override fun toString(): String {
                return "N"
            }
        }
    }

    private val dataFrame: DataFrame<Row, C, *>
    private val probing: MutableSingleSelection<Row>
    private val selection: MutableSelection<Row>
    private val filter: Filter<Row>
    private val colorMapping: MutableColorMapping<Row, C>
    private val coloring: MutableColoring<Row>
    private val statisticsMap: MutableMap<C, UnivariateStatistics>
    private val probingListener: SingleSelectionListener<Row> = object : SingleSelectionListener<Row> {
        override fun selectionChanged(event: SingleSelectionEvent<Row>) {
            scheduleTableDataChanged()
        }
    }
    private val selectionListener: SelectionListener<Row> = object : SelectionListener<Row> {
        override fun selectionChanged(event: SelectionEvent<Row>) {
            scheduleTableDataChanged()
        }
    }
    private val coloringListener: ColoringListener<Row> = object : ColoringListener<Row> {
        override fun coloringChanged(event: ColoringEvent<Row>) {
            scheduleTableDataChanged()
        }
    }
    private val filterListener: FilterListener<Row> = object : FilterListener<Row> {
        override fun filterChanged(event: FilterEvent<Row>) {
            scheduleTableDataChanged()
        }
    }
    override val rowIndex: UniqueIndex<Stat> = DefaultUniqueIndex(*Stat.values())

    enum class Type {
        Probed, Selected, Colored, Active, Filtered, All
    }

    private var type: Type? = null
    private val structureChangedTimer: CPTimer
    private val dataChangedTimer: CPTimer
    private fun scheduleTableStructureChanged() {
        structureChangedTimer.restart()
    }

    private fun scheduleTableDataChanged() {
        dataChangedTimer.restart()
    }

    fun setType(type: Type) {
        if (this.type != null) {
            when (this.type) {
                Type.Probed -> probing.removeSingleSelectionListener(probingListener)
                Type.Selected -> selection.removeSelectionListener(selectionListener)
                Type.Colored -> coloring.removeColoringListener(coloringListener)
                Type.Active -> filter.removeFilterListener(filterListener)
                Type.Filtered -> filter.removeFilterListener(filterListener)
                Type.All -> {
                }
                else -> {}
            }
        }
        if (this.type != type) {
            this.type = type
            when (type) {
                Type.Probed -> probing.addSingleSelectionListener(probingListener)
                Type.Selected -> selection.addSelectionListener(selectionListener)
                Type.Colored -> coloring.addColoringListener(coloringListener)
                Type.Active -> filter.addFilterListener(filterListener)
                Type.Filtered -> filter.addFilterListener(filterListener)
                Type.All -> {
                }
            }
            statisticsMap.clear()
            notifyDataFrameChanged(DataFrameEvent(null, null, false))
        }
    }

    fun getDataFrame(): DataFrame<Row, C, *> {
        return dataFrame
    }

    override fun getColumnName(column: C): String? {
        return dataFrame.getColumnName(column)
    }

    override fun getColumnClass(column: C): KClass<out Any> {
        return dataFrame.getColumnClass(column)
    }

    override val rowCount: Int
        get() = Stat.values().size

    override val columnCount: Int
        get() = dataFrame.columnCount

    override fun getValueAt(rowIndex: Stat, columnIndex: C): Any? {
        val statistics: UnivariateStatistics = getStatistics(columnIndex)!!
        if (rowIndex === Stat.ThreeSignma) {
            return statistics.getSigma(3.0)
        } else if (rowIndex === Stat.TwoSigma) {
            return statistics.getSigma(2.0)
        } else if (rowIndex === Stat.OneSigma) {
            return statistics.getSigma(1.0)
        } else if (rowIndex === Stat.Mean) {
            return statistics.mean
        } else if (rowIndex === Stat.MinusOneSigma) {
            return statistics.getSigma(-1.0)
        } else if (rowIndex === Stat.MinusTwoSigma) {
            return statistics.getSigma(-2.0)
        } else if (rowIndex === Stat.MinusThreeSigma) {
            return statistics.getSigma(-3.0)
        } else if (rowIndex === Stat.Maximum) {
            return statistics.getPercentile(100.0)
        } else if (rowIndex === Stat.P995) {
            return statistics.getPercentile(99.5)
        } else if (rowIndex === Stat.P975) {
            return statistics.getPercentile(97.5)
        } else if (rowIndex === Stat.P900) {
            return statistics.getPercentile(90.0)
        } else if (rowIndex === Stat.P750) {
            return statistics.getPercentile(75.0)
        } else if (rowIndex === Stat.Median) {
            return statistics.getPercentile(50.0)
        } else if (rowIndex === Stat.P250) {
            return statistics.getPercentile(25.0)
        } else if (rowIndex === Stat.P100) {
            return statistics.getPercentile(10.0)
        } else if (rowIndex === Stat.P025) {
            return statistics.getPercentile(2.5)
        } else if (rowIndex === Stat.P050) {
            return statistics.getPercentile(0.5)
        } else if (rowIndex === Stat.Minimum) {
            return statistics.getPercentile(0.0)
        } else if (rowIndex === Stat.Sum) {
            return statistics.sum
        } else if (rowIndex === Stat.Origin) {
            return if (statistics.mean != null) 0.0 else null
        } else if (rowIndex === Stat.N) {
            return statistics.count
        }
        return null
    }

    override val columnIndex: UniqueIndex<C>
        get() = dataFrame.columnIndex

    override fun getRowClass(integer: Stat): KClass<out Any> {
        return Any::class
    }

    fun getRow(integer: Stat?): Series<C, *>? {
        return null
    }

    fun join(series: Series<*,*>?, cs: Array<C>?): DataFrame<*,*,*>? {
        return null
    }

    fun getStatistics(column: C): UnivariateStatistics? {
        return if (statisticsMap.containsKey(column)) {
            statisticsMap[column]
        } else {
            val univariateStatistics = UnivariateStatistics(getRows()!!, dataFrame!!.getColumn(column) as Series<Any?, Any?>)
            statisticsMap[column] = univariateStatistics
            univariateStatistics
        }
    }

    private fun getRows(): Iterable<Row>? {
        return when (type) {
            Type.Probed -> probing
            Type.Selected -> selection
            Type.Colored -> coloring
            Type.Active -> object : TableIterable(0, dataFrame.rowCount - 1) {
                override fun condition(row: Row): Boolean {
                    return !filter.isFiltered(row)
                }
            }
            Type.Filtered -> object : TableIterable(0, dataFrame.rowCount - 1) {
                override fun condition(row: Row): Boolean {
                    return filter.isFiltered(row)
                }
            }
            Type.All -> TableIterable(0, dataFrame.rowCount - 1)
            else -> null
        }
    }

    protected open inner class TableIterable(private val from: Int, private val to: Int) : Iterable<Row> {
        open fun condition(row: Row): Boolean {
            return true
        }

        override fun iterator(): Iterator<Row> {
            return object : MutableIterator<Row> {
                var index = from
                private var next: Row? = null
                var queryNext = true
                override fun hasNext(): Boolean {
                    if (queryNext) {
                        next = findNext()
                        queryNext = false
                    }
                    return next != null
                }

                override fun next(): Row {
                    if (queryNext) {
                        next = findNext()
                    }
                    queryNext = true
                    return next!!
                }

                private fun findNext(): Row? {
                    while (index <= to) {
                        val `object`: Row = dataFrame.getRowKey(index)
                        if (condition(`object`)) {
                            index++
                            return `object`
                        }
                        index++
                    }
                    return null
                }

                override fun remove() {
                    throw UnsupportedOperationException()
                }
            }
        }
    }

    init {
        this.dataFrame = dataFrame
        this.probing = probing
        this.selection = selection
        this.filter = filter
        this.colorMapping = colorMapping
        this.coloring = coloring
        statisticsMap = HashMap<C, UnivariateStatistics>()
        structureChangedTimer = CPHelper.instance.createTimer("Statistics\$Structure", 40, true, object : CPTimerListener {
            override fun timerTriggered() {
                notifyDataFrameChanged(DataFrameEvent(null, null, true))
            }
        })
        dataChangedTimer = CPHelper.instance.createTimer("Statistics\$Data", 40, true, object : CPTimerListener {
            override fun timerTriggered() {
                statisticsMap.clear()
                notifyDataFrameChanged(DataFrameEvent(null, null, false))
            }
        })
        setType(Type.Active)
    }
}