/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.tablelens

import com.macrofocus.common.command.Future
import com.macrofocus.common.concurrent.Callable
import com.macrofocus.common.concurrent.ExecutorService
import com.macrofocus.common.concurrent.Runtime
import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.properties.Property
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import com.macrofocus.high_d.axis.AxisListener
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.order.MutableVisibleOrder
import com.macrofocus.order.OrderEvent
import com.macrofocus.order.OrderListener
import org.mkui.canvas.CPCanvas
import org.mkui.canvas.PaletteProvider
import org.mkui.color.MkColor
import org.mkui.color.colorOf
import org.mkui.geom.Point2D
import org.mkui.geom.Rectangle
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.AbstractIDrawing
import org.mkui.graphics.IGraphics
import org.mkui.graphics.colortheme.ColorTheme
import org.mkui.graphics.pressure.LogPressure
import org.mkui.palette.FixedPalette
import org.mkui.rubberband.RubberbandDrawing
import org.mkui.visual.VisualLayer
import org.mkui.visual.VisualListener
import kotlin.math.ceil
import kotlin.math.max
import kotlin.math.min

abstract class AbstractTableLensComponent<Row, Column> protected constructor(
    view: TableLensView<Row, Column>,
) : TableLensComponent<Row, Column> {
    override var model: TableLensModel<Row, Column>? = null
        set(value) {
            if (field != null) {
                field!!.removeTableLensListener(listener)
                field!!.getAxisGroupModel().axisOrder!!.removeOrderListener(orderListener)
                field!!.getAxisGroupModel().removeAxisListener(axisListener)
            }
            field = value
            if (field != null) {
                createOverplots()
                field!!.addTableLensListener(listener)
                field!!.getAxisGroupModel().axisOrder!!.addOrderListener(orderListener)
                field!!.getAxisGroupModel().addAxisListener(axisListener)
            }
            timer.restart()
        }
    private val margin = 0.2
    protected val canvas: CPCanvas by lazy { CPCanvas() } 
    var view: TableLensView<Row, Column>? = null

    private val listener: TableLensListener = object : TableLensListener {
        override fun tableLensChanged() {
            timer.restart()
        }
    }
    private val axisListener: AxisListener = object : AxisListener {
        override fun axisChanged() {
            scheduleUpdate()
        }
    }
    private val orderListener: OrderListener<AxisModel<Row, Column>> = object : OrderListener<AxisModel<Row, Column>> {
        override fun orderChanged(event: OrderEvent<AxisModel<Row, Column>>?) {
            scheduleUpdate()
        }

        override fun orderVisibility(event: OrderEvent<AxisModel<Row, Column>>) {
            scheduleUpdate()
        }

        override fun orderAdded(event: OrderEvent<AxisModel<Row, Column>>) {
            scheduleUpdate()
        }

        override fun orderRemoved(event: OrderEvent<AxisModel<Row, Column>>) {
            scheduleUpdate()
        }
    }
    protected val timer: CPTimer
    protected abstract fun getWidth(): Int
    protected abstract fun getHeight(): Int
    protected abstract fun refresh()
    protected abstract fun repaint()

    override fun createOverplots() {
        if (model != null) {
            canvas.removeAllLayers()
            canvas.addLayer(object : AbstractIDrawing() {
                override fun draw(g: IGraphics, point: Point2D?, width: Double, height: Double, clipBounds: Rectangle) {
                    val background: MkColor? = view?.getColorTheme()?.value?.background
                    if (background != null) {
                        g.setColor(background)
                        g.fillRectangle2D(Rectangle2D.Double(0.0, 0.0, width, height))
                    }
                }
            })
            canvas.addDensityLayer(view!!.getRendering().value, object : AbstractVisualLayerIDrawing(model!!.getVisual().multipleSelected) {
                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.fillRectangle(x, y, getWidth(), height)
                }
            }, LogPressure(), object : PaletteProvider {
                override val palette: FixedPalette
                    get() = view!!.getColorTheme().value.selectedShadowPalette
            })
            canvas.addLayer(object : AbstractVisualLayerIDrawing(model!!.getVisual().singleSelected) {
                val color: MkColor = colorOf(255, 0, 0)
                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.setColor(color)
                    g.fillRectangle(x, y, getWidth(), height)
                }
            })
            canvas.addLayer(object : AbstractVisualLayerIDrawing(model!!.getVisual().probed) {
                val selectionColor: MkColor = colorOf(255, 0, 0)
                val color: MkColor = colorOf(255, 200, 0)
                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.setColor(color)
                    g.fillRectangle(x, y - 1, getWidth(), height + 2)
                    if (model!!.getVisual().selection.isSelected(row)) {
                        g.setColor(selectionColor)
                        g.fillRectangle(x, y, getWidth(), height)
                    }
                }
            })
            canvas.addDensityLayer(view!!.getRendering().value, object : AbstractVisualLayerIDrawing(model!!.getVisual().filtered) {
                override val isActive: Boolean
                    get() = view!!.getShowFiltered().value && super.isActive

                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.fillRectangle(x, y, width, height)
                }
            }, LogPressure(), object : PaletteProvider {
                override val palette: FixedPalette
                    get() = view!!.getColorTheme().value.ghostedPalette
            })
            canvas.addDensityLayer(view!!.getRendering().value, object : AbstractVisualLayerIDrawing(model!!.getVisual().visible) {
                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.fillRectangle(x, y, width, height)
                }
            }, LogPressure(), object : PaletteProvider {
                override val palette: FixedPalette
                    get() = view!!.getColorTheme().value.visiblePalette
            })
            canvas.addAveragingLayer(view!!.getRendering().value, object : AbstractVisualLayerIDrawing(model!!.getVisual().colorMapped) {
                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.setColor(model!!.getColorMapping().getColor(row)!!)
                    g.fillRectangle(x, y, width, height)
                }
            })
            canvas.addBufferedLayer(object : AbstractVisualLayerIDrawing(model!!.getVisual().colored) {
                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.setColor(model!!.getColoring().getColor(row)!!)
                    g.fillRectangle(x, y, width, height)
                }
            })
            canvas.addDensityLayer(view!!.getRendering().value, object : AbstractVisualLayerIDrawing(model!!.getVisual().multipleSelected) {
                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.fillRectangle(x, y, width, height)
                }
            }, LogPressure(), object : PaletteProvider {
                override val palette: FixedPalette
                    get() = view!!.getColorTheme().value.selectedPalette
            })
            canvas.addLayer(object : AbstractVisualLayerIDrawing(model!!.getVisual().singleSelected) {
                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.setColor(view!!.getColorTheme().value.selection)
                    g.fillRectangle(x, y, width, height)
                }
            })
            canvas.addLayer(object : AbstractVisualLayerIDrawing(model!!.getVisual().probed) {
                override fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row) {
                    g.setColor(view!!.getColorTheme().value.probing)
                    g.fillRectangle(x, y - 1, width, height + 2)
                    if (model!!.getVisual().selection.isSelected(row)) {
                        g.setColor(view!!.getColorTheme().value.selection)
                        g.fillRectangle(x, y, width, height)
                    }
                }
            })
            canvas.addLayer(object : RubberbandDrawing(view!!.getRubberBand()) {
                override val colorTheme: Property<ColorTheme>
                    get() = view!!.getColorTheme()
            })
        }
    }

    //    private abstract class AbstractDrawing implements Drawing<Row> {
    //        @Override
    //        public void draw(com.macrofocus.overplot.Graphics g, Row row) {
    //            int y = getY(row);
    //
    //            final MutableVisibleOrder<AxisModel<Row,C>> axisOrder = model.getAxisGroupModel().getAxisOrder();
    //            final int objectCount = model.getObjectCount();
    //            int height = (int) Math.max(Math.ceil((double)getHeight() / objectCount), 1);
    //            for (int i = 0; i < axisOrder.size(); i++) {
    //                AxisModel xAxisModel = axisOrder.get(i);
    //                final Double x1Location = model.getLocation(xAxisModel);
    //
    //                Double x2Location;
    //                if (i + 1 < axisOrder.size()) {
    //                    x2Location = model.getLocation(axisOrder.get(i + 1));
    //                } else {
    //                    x2Location = 1.0;
    //                }
    //
    //                if (x1Location != null && x2Location != null) {
    //                    final Number value = xAxisModel.getValue(row);
    //                    if(value != null) {
    //                        int x = (int) (x1Location * getWidth());
    //                        int w = (int) (((x2Location * getWidth()) - x) * (1.0 - margin));
    //                        int width = (int) ((w) * (value.doubleValue() - xAxisModel.getMinimum()) / (xAxisModel.getMaximum() - xAxisModel.getMinimum()));
    //                        width = Math.min(w, width);
    //                        width = Math.max(1, width);
    //
    //                        Rectangle mp = new Rectangle(x, y, width, height);
    //
    //                        if (mp != null) {
    //                            draw(g, mp, row);
    //                        }
    //                    }
    //                }
    //            }
    //        }
    //
    //        public abstract void draw(com.macrofocus.overplot.Graphics g, Rectangle shape, Row row);
    //    }
    abstract inner class AbstractVisualLayerIDrawing protected constructor(visualLayer: VisualLayer<Row>) : AbstractIDrawing() {
        private val visualLayer: VisualLayer<Row>

        override val isActive: Boolean
            get() = visualLayer.isActive

        override fun draw(g: IGraphics, point: Point2D?, width: Double, height: Double, clipBounds: Rectangle) {
            if (visualLayer.objectCount > 0) {
                val height = getHeight()
                val axisOrder: MutableVisibleOrder<AxisModel<Row, Column>> = model!!.getAxisGroupModel().axisOrder!!
                val objectCount: Int = model!!.objectCount
                val rowHeight = max(ceil(height.toDouble() / objectCount), 1.0).toInt()
                if (!MULTITHREADED || !g.isThreadSafe()) {
                    for (row in visualLayer) {
                        draw(g, row, height, axisOrder, objectCount, rowHeight)
                    }
                } else {
                    val nTasks: Int =
                        min(Runtime.getRuntime().availableProcessors(), visualLayer.objectCount) // Determine number of tasks i.e. threads
                    val nRowsPerTask: Int = visualLayer.objectCount / nTasks
                    val todo: MutableList<Callable<Any?>> = ArrayList<Callable<Any?>>(nTasks)
                    for (nTask in 0 until nTasks)  // Iterate over all tasks
                    {
                        val fromRow = nTask * nRowsPerTask // the tasks first row
                        val toRow = if (nTask < nTasks - 1) fromRow + nRowsPerTask else visualLayer.objectCount // the tasks last row
                        todo.add(object : Callable<Any?> {
                            @Throws(Exception::class)
                            override fun call(): Any? {
                                for (row in visualLayer.iterable(
                                    // Create and define the task
                                    fromRow, toRow - 1
                                )) {
                                    draw(g, row, height, axisOrder, objectCount, rowHeight)
                                }
                                return null
                            }
                        })
                    }
                    try {
                        val answers: List<Future<Any?>> = executor!!.invokeAll(todo)
                        for (answer in answers) {
                            try {
                                answer.get()
                            } catch (e: Exception) {
                                e.printStackTrace()
                            }
                        }
                    } catch (e: Exception) {
                        e.printStackTrace() // something wrong, maybe
                    }
                }
            }
        }

        protected fun draw(g: IGraphics, row: Row, height: Int, axisOrder: MutableVisibleOrder<AxisModel<Row, Column>>, objectCount: Int, rowHeight: Int) {
            val y = getY(row, objectCount, height)
            val axisCount: Int = axisOrder.size()
            for (i in 0 until axisCount) {
                val xAxisModel: AxisModel<Row,Column> = axisOrder.get(i)!!
                val x1Location: Double? = model?.getLocation(xAxisModel)
                var x2Location: Double?
                x2Location = if (i + 1 < axisCount) {
                    model?.getLocation(axisOrder.get(i + 1)!!)
                } else {
                    1.0
                }
                if (x1Location != null && x2Location != null) {
                    val value: Number? = xAxisModel.getValue(row)
                    if (value != null) {
                        val cw = getWidth()
                        val x = (x1Location * cw).toInt()
                        val w = ((x2Location * cw - x) * (1.0 - margin)).toInt()
                        val range: Double = xAxisModel.maximum - xAxisModel.minimum
                        var width = (w * (value.toDouble() - xAxisModel.minimum) / range).toInt()
                        width = min(w, width)
                        width = max(1, width)
                        draw(g, x, y, width, rowHeight, row)
                    }
                }
            }
        }

        //        protected void draw(IGraphics g, int i, int height, int rowHeight) {
        //            final MutableVisibleOrder<AxisModel<Row,C>> axisOrder = model.getAxisGroupModel().getAxisOrder();
        //            AxisModel xAxisModel = axisOrder.get(i);
        //            final Double x1Location = model.getLocation(xAxisModel);
        //
        //            Double x2Location;
        //            if (i + 1 < axisOrder.size()) {
        //                x2Location = model.getLocation(axisOrder.get(i + 1));
        //            } else {
        //                x2Location = 1.0;
        //            }
        //
        //            if (x1Location != null && x2Location != null) {
        //                final int cw = getWidth();
        //                int x = (int) (x1Location * cw);
        //                int w = (int) (((x2Location * cw) - x) * (1.0 - margin));
        //                final int objectCount = model.getObjectCount();
        //                final double range = xAxisModel.getMaximum() - xAxisModel.getMinimum();
        //                for (Row row : visualLayer) {
        //                    final Number value = xAxisModel.getValue(row);
        //                    if(value != null) {
        //                        int y = getY(row, objectCount, height);
        //                        int width = (int) ((w) * (value.doubleValue() - xAxisModel.getMinimum()) / range);
        //                        width = Math.min(w, width);
        //                        width = Math.max(1, width);
        //
        //                        draw(g, x, y, width, rowHeight, row);
        //                    }
        //                }
        //            }
        //        }
        abstract fun draw(g: IGraphics, x: Int, y: Int, width: Int, height: Int, row: Row)

        init {
            this.visualLayer = visualLayer
            visualLayer.addVisualListener(object : VisualListener {
                override fun visualChanged() {
                    notifyIDrawingChanged()
                }
            })
        }
    }

    private fun getY(row: Row, objectCount: Int, height: Int): Int {
        return if (objectCount > 1) {
            model!!.getIndex(row) * (height - 1) / objectCount
        } else {
            0
        }
    }

    private fun getIndex(y: Int): Int {
        return y * (model!!.objectCount - 1) / (getHeight() - 1)
    }

    override fun getClosestRow(x: Int, y: Int): Row? {
        val row: Row = model!!.getObject(getIndex(y))
        return if (!model!!.getFilter().isFiltered(row)) {
            row
        } else {
            null
        }
    }

    override fun getRows(rect: Rectangle2D): List<Row> {
        val list: MutableList<Row> = ArrayList<Row>()
        val index1 = getIndex(rect.minY.toInt())
        val index2 = getIndex(rect.maxY.toInt())
        for (i in index1..index2) {
            val row: Row = model!!.getObject(i)
            if (!model!!.getFilter().isFiltered(row)) {
                list.add(row)
            }
        }
        return list
    }
    companion object {
        private const val MULTITHREADED = true
        private val executor: ExecutorService? = CPHelper.instance.visualizationExecutorService()
    }

    init {
        this.view = view
        timer = CPHelper.instance.createTimer("TableLensResizer", 40, false, object : CPTimerListener {
            override fun timerTriggered() {
                if (getWidth() > 0 && getHeight() > 0) {
                    refresh()
                }
            }
        })
    }
}