package com.macrofocus.high_d.tablelens

import com.macrofocus.colormapping.implementation.SimpleColorMapping
import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.filter.FilterEvent
import com.macrofocus.common.filter.FilterListener
import com.macrofocus.common.filter.MutableFilter
import com.macrofocus.common.properties.*
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import com.macrofocus.high_d.axis.Alignment
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import com.macrofocus.high_d.axis.group.DefaultAxisGroupModel
import com.macrofocus.high_d.axis.locations.AxisLocations
import com.macrofocus.high_d.axis.locations.DefaultAxisLocations
import com.macrofocus.high_d.tablelens.TableLensModel.Companion.PROPERTY_SHOW_FILTERED
import com.macrofocus.visual.SimpleVisual
import com.macrofocus.visual.SimpleVisualObjects
import org.mkui.coloring.MutableColoring
import org.mkui.colormap.ColorMapFactory
import org.mkui.colormapping.MutableColorMapping
import org.mkui.visual.Visual
import org.mkui.visual.VisualObjects
import org.molap.dataframe.DataFrame
import org.molap.dataframe.DataFrameEvent
import org.molap.dataframe.DataFrameListener
import org.molap.index.DefaultUniqueIndex
import org.molap.index.UniqueIndex
import kotlin.jvm.Synchronized

class DefaultTableLensModel<Row, C>(
    dataFrame: DataFrame<Row, C, *>,
    visual: Visual<Row, C>,
    axisGroupModel: AxisGroupModel<Row, C>
) : AbstractTableLensModel<Row, C>(), VisualObjects<Row> {
    private val dataFrame: DataFrame<Row, C, *>
    private val visual: Visual<Row, C>
    private val axisGroupModel: AxisGroupModel<Row, C>
    private val axisLocations: AxisLocations<Row , C>

    //    private final List<JToggleButton.ToggleButtonModel> buttonModels;
    private var order: AxisModel<Row,*>? = null
    private var sortedRows: MutableList<Row>? = null
    private var index: UniqueIndex<Row>? = null
    private var rowComparator: RowComparator? = null
    private var rowsDirty = true
    private var sortedRowsDirty = true
    private val properties: MutableProperties<String?> = SimpleProperties()
    private val maxDimensions: Property<Int> = properties.createProperty("maxDimensions", Int.MAX_VALUE)
    private var showFiltered: MutableProperty<Boolean> = properties.createProperty(PROPERTY_SHOW_FILTERED, true)
    private val _timer: CPTimer

    constructor(colorMapFactory: ColorMapFactory, dataFrame: DataFrame<Row, C, *>) : this(
        dataFrame,
        SimpleVisual<Row, C>(SimpleVisualObjects(dataFrame), SimpleColorMapping(colorMapFactory, dataFrame))
    ) {
    }

    constructor(dataFrame: DataFrame<Row, C, *>, visual: Visual<Row, C>) : this(
        dataFrame,
        visual,
        DefaultAxisGroupModel("Root", dataFrame, null, visual.probing, visual.selection, visual.filter)
    ) {
    }

    private fun updateRows() {
        rowsDirty = true
        updateOrder()
    }

    private fun updateOrder() {
        sortedRowsDirty = true
    }

    @Synchronized
    override fun getSortedRows(): UniqueIndex<Row>? {
        if (rowsDirty || sortedRowsDirty) {
            val sortedRows: MutableList<Row>
            if (rowsDirty) {
                sortedRows = ArrayList<Row>()
                for (row in dataFrame.rows()) {
                    if (showFiltered.value || !visual.filter.isFiltered(row)) {
                        sortedRows.add(row)
                    }
                }
                rowsDirty = false
            } else {
                sortedRows = this.sortedRows!!
            }
            if (sortedRowsDirty) {
                sortedRows.sortWith(rowComparator as Comparator<in Row>)
                sortedRowsDirty = false
            }
            this.sortedRows = sortedRows
            index = DefaultUniqueIndex(sortedRows)
        }
        return index
    }

    override fun getVisual(): Visual<Row, C> {
        return visual
    }

    override val objectCount: Int
        get() = getSortedRows()!!.size

    /**
     * {@inheritDoc}
     */
    override fun getProbing(): MutableSingleSelection<Row> {
        return visual.probing
    }

    /**
     * {@inheritDoc}
     */
    override fun getSelection(): MutableSelection<Row> {
        return visual.selection
    }

    /**
     * {@inheritDoc}
     */
    override fun getFilter(): MutableFilter<Row> {
        return visual.filter
    }

    /**
     * {@inheritDoc}
     */
    override fun getColorMapping(): MutableColorMapping<Row, C> {
        return visual.colorMapping
    }

    /**
     * {@inheritDoc}
     */
    override fun getColoring(): MutableColoring<Row> {
        return visual.coloring
    }

    /**
     * {@inheritDoc}
     */
    override fun getObject(index: Int): Row {
        return getSortedRows()!!.getKey(index)
    }

    override fun getIndex(row: Row): Int {
//        if(this.order != null || visual.getFilter().isActive()) {
//            return Collections.binarySearch(getSortedRows(), row, rowComparator);
//        } else {
//            return dataFrame.getRowAddress(row);
//        }
        return getSortedRows()!!.getAddress(row)
    }

    /**
     * {@inheritDoc}
     */
    private fun getAxisModel(c: Int): AxisModel<Row, C> {
        return axisGroupModel.axisOrder!!.get(c)
    }

    /**
     * {@inheritDoc}
     */
    override fun getAxisCount(): Int {
        return axisGroupModel.axisOrder!!.size()
    }
    /**
     * {@inheritDoc}
     */
    //    @Override
    //    public ButtonModel getButtonModel(final int c) {
    //        return buttonModels.get(c);
    //    }
    /**
     * {@inheritDoc}
     */
    override fun getSelectedAxis(): Iterable<AxisModel<Row , C>> {
        //        for (int i = 0; i < getAxisCount(); i++) {
//            final ButtonModel buttonModel = getButtonModel(i);
//            if (buttonModel.isSelected()) {
//                list.add(getAxisModel(i));
//            }
//        }
        return ArrayList<AxisModel<Row , C>>()
    }

    /**
     * {@inheritDoc}
     */
    override fun getLocation(axisModel: AxisModel<Row , C>): Double {
        return axisLocations.getLocation(Alignment.Left, axisModel)
    }

    /**
     * {@inheritDoc}
     */
    override fun getAxisGroupModel(): AxisGroupModel<Row,C> {
        return axisGroupModel
    }

    /**
     * {@inheritDoc}
     */
    override fun setLocation(axisModel: AxisModel<Row, C>, l: Double) {
        axisLocations.setLocation(Alignment.Left, axisModel, l)
        firedTableLensChanged()
    }

    override fun getShowFiltered(): Property<Boolean> {
        return showFiltered
    }

    override fun setShowFiltered(showFiltered: MutableProperty<Boolean>) {
        this.showFiltered = showFiltered
        properties.replaceProperty(PROPERTY_SHOW_FILTERED, showFiltered)
    }

    override fun setOrder(axisModel: AxisModel<Row,C>) {
        if (order !== axisModel) {
            order = axisModel
            rowComparator = RowComparator(axisModel)
            updateOrder()
            firedTableLensChanged()
        }
    }

    private inner class RowComparator(private val axisModel: AxisModel<Row,*>?) : Comparator<Row> {
        override fun compare(o1: Row, o2: Row): Int {
            return if (axisModel != null) {
                val v1: Number? = axisModel.getValue(o1)
                if (v1 != null) {
                    val v2: Number? = axisModel.getValue(o2)
                    if (v2 != null) {
                        val c: Int = v1.toDouble().compareTo(v2.toDouble())
                        if (c == 0) {
                            if (o1 is Comparable<*>) {
                                (o1 as Comparable<*>).compareTo(o2 as Nothing)
                            } else {
                                dataFrame.getRowAddress(o1).compareTo(dataFrame.getRowAddress(o2))
                            }
                        } else {
                            -c
                        }
                    } else {
                        -1
                    }
                } else {
                    val v2: Number? = axisModel.getValue(o2)
                    if (v2 != null) {
                        1
                    } else {
                        if (o1 is Comparable<*>) {
                            (o1 as Comparable<*>).compareTo(o2 as Nothing)
                        } else {
                            dataFrame.getRowAddress(o1).compareTo(dataFrame.getRowAddress(o2))
                        }
                    }
                }
            } else {
                if (o1 is Comparable<*> && o2 is Comparable<*>) {
                    (o1 as Comparable<Any?>).compareTo(o2 as Comparable<Any?>)
                } else {
                    dataFrame.getRowAddress(o1).compareTo(dataFrame.getRowAddress(o2))
                }
                //                return ((Integer)o1.getRow()).compareTo(o2.getRow());
            }
        }
    }

    init {
        this.dataFrame = dataFrame
        this.axisGroupModel = axisGroupModel
        _timer = CPHelper.instance.createTimer("TableLensUpdater", 40, true, object : CPTimerListener {
            override fun timerTriggered() {
                updateRows()
                firedTableLensChanged()
            }
        })
        axisLocations = DefaultAxisLocations<Row, C>(axisGroupModel.axisOrder!!, maxDimensions)
        this.visual = visual

//        buttonModels = new ArrayList<JToggleButton.ToggleButtonModel>();
//
//        for (final AxisModel current : axisGroupModel.getAxisOrder()) {
//            buttonModels.add(new JToggleButton.ToggleButtonModel());
//        }
        visual.filter.addFilterListener(object : FilterListener<Row> {
            override fun filterChanged(event: FilterEvent<Row>) {
                if (!showFiltered.value) {
                    _timer.restart()
                }
            }
        })
        properties.addPropertiesListener(object : PropertiesListener<String?> {
            override fun propertyChanged(name: String?, event: PropertyEvent<Any?>) {
                _timer.restart()
            }
        })
        rowComparator = RowComparator(order)
        dataFrame.addDataFrameListener(object : DataFrameListener<Row, C> {
            override fun dataFrameChanged(event: DataFrameEvent<Row, C>) {
                updateRows()
                firedTableLensChanged()
            }
        })
        updateRows()
    }
}