/*
 * Copyright (c) 2013 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.parallelcoordinates.blending

import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min

enum class BlendingMode {
    SourceOver {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            throw IllegalArgumentException("Not implemented!")
        }
    },
    Add {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = min(255, src[0] + dst[0])
            result[1] = min(255, src[1] + dst[1])
            result[2] = min(255, src[2] + dst[2])
            result[3] = min(255, src[3] + dst[3])
            //            result[3] = Math.min(255, src[3] + dst[3] - (src[3] * dst[3]) / 255);
        }
    },

    /**
     * Specifies no blending. The blending formula simply selects the source color.
     */
    Normal {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = src[0]
            result[1] = src[1]
            result[2] = src[2]
            result[3] = src[3]
        }
    },  //    /**
    //     * Edits or paints each pixel to make it the result color. However, the result color is a random replacement of the
    //     * pixels with the base color or the blend color, depending on the opacity at any pixel location.
    //     */
    //    Dissolve {
    //        void blend(int[] src, int[] dst, int[] result) {
    //        }
    //    },
    /**
     * Selects the darker of the backdrop and source colors. The backdrop is replaced with the source where the source
     * is darker; otherwise, it is left unchanged.
     */
    Darken {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = min(src[0], dst[0])
            result[1] = min(src[1], dst[1])
            result[2] = min(src[2], dst[2])
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * The source color is multiplied by the destination color and replaces the destination. The resultant color is
     * always at least as dark as either the source or destination color. Multiplying any color with black results in
     * black. Multiplying any color with white preserves the original color.
     */
    Multiply {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = src[0] * dst[0] + 2 shr 8
            result[1] = src[1] * dst[1] + 2 shr 8
            result[2] = src[2] * dst[2] + 2 shr 8
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Darkens the backdrop color to reflect the source color. Painting with white produces no change.
     */
    ColorBurn {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = if (src[0] == 0) 0 else max(0, 255 - (255 - dst[0] shl 8) / src[0])
            result[1] = if (src[1] == 0) 0 else max(0, 255 - (255 - dst[1] shl 8) / src[1])
            result[2] = if (src[2] == 0) 0 else max(0, 255 - (255 - dst[2] shl 8) / src[2])
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Looks at the color information in each channel and darkens the base color to reflect the blend color by
     * decreasing the brightness. Blending with white produces no change.
     */
    LinearBurn {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = if (src[0] + dst[0] == 0) 0 else src[0] + dst[0] - 255
            result[1] = if (src[1] + dst[1] == 0) 0 else src[1] + dst[1] - 255
            result[2] = if (src[2] + dst[2] == 0) 0 else src[2] + dst[2] - 255
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },  //    /**
    //     * Compares the total of all channel values for the blend and base color and displays the lower value color.
    //     * Darker Color does not produce a third color, which can result from the Darken blend, because it chooses the
    //     * lowest channel values from both the base and the blend color to create the result color.
    //     */
    //    DarkerColor {
    //        void blend(int[] src, int[] dst, int[] result) {
    //        }
    //    },
    /**
     * Selects the lighter of the backdrop and source colors. The backdrop is replaced with the source where the source
     * is lighter; otherwise, it is left unchanged.
     */
    Lighten {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = max(src[0], dst[0])
            result[1] = max(src[1], dst[1])
            result[2] = max(src[2], dst[2])
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Multiplies the complements of the backdrop and source color values, then complements the result. The result
     * color is always at least as light as either of the two constituent colors. Screening any color with white
     * produces white; screening with black leaves the original color unchanged. The effect is similar to projecting
     * multiple photographic slides simultaneously onto a single screen.
     */
    Screen {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = 255 - ((255 - src[0]) * (255 - dst[0]) shr 8)
            result[1] = 255 - ((255 - src[1]) * (255 - dst[1]) shr 8)
            result[2] = 255 - ((255 - src[2]) * (255 - dst[2]) shr 8)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Brightens the backdrop color to reflect the source color. Painting with black produces no changes.
     */
    ColorDodge {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = if (src[0] == 255) 255 else min((dst[0] shl 8) / (255 - src[0]), 255)
            result[1] = if (src[1] == 255) 255 else min((dst[1] shl 8) / (255 - src[1]), 255)
            result[2] = if (src[2] == 255) 255 else min((dst[2] shl 8) / (255 - src[2]), 255)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Looks at the color information in each channel and brightens the base color to reflect the blend color by
     * increasing the brightness. Blending with black produces no change.
     */
    LinearDodge {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = min(255, src[0] + dst[0])
            result[1] = min(255, src[1] + dst[1])
            result[2] = min(255, src[2] + dst[2])
            result[3] = min(255, src[3] + dst[3])
        }
    },  //    /**
    //     * Compares the total of all channel values for the blend and base color and displays the higher value color.
    //     * Lighter Color does not produce a third color, which can result from the Lighten blend, because it chooses the
    //     * highest channel values from both the base and blend color to create the result color.
    //     */
    //    LighterColor {
    //        void blend(int[] src, int[] dst, int[] result) {
    //        }
    //    },
    /**
     * Multiplies or screens the colors, depending on the backdrop color value. Source colors overlay the backdrop
     * while preserving its highlights and shadows. The backdrop color is not replaced but is mixed with the source
     * color to reflect the lightness or darkness of the backdrop. Overlay is the inverse of the ‘hardlight’ blend
     * mode.
     */
    Overlay {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = if (dst[0] < 128) dst[0] * src[0] shr 7 else 255 - ((255 - dst[0]) * (255 - src[0]) shr 7)
            result[1] = if (dst[1] < 128) dst[1] * src[1] shr 7 else 255 - ((255 - dst[1]) * (255 - src[1]) shr 7)
            result[2] = if (dst[2] < 128) dst[2] * src[2] shr 7 else 255 - ((255 - dst[2]) * (255 - src[2]) shr 7)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Darkens or lightens the colors, depending on the source color value. The effect is similar to shining a diffused
     * spotlight on the backdrop
     */
    SoftLight {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            val mRed = src[0] * dst[0] / 255
            val mGreen = src[1] * dst[1] / 255
            val mBlue = src[2] * dst[2] / 255
            result[0] = mRed + dst[0] * (255 - (255 - dst[0]) * (255 - src[0]) / 255 - mRed) / 255
            result[1] = mGreen + dst[1] * (255 - (255 - dst[1]) * (255 - src[1]) / 255 - mGreen) / 255
            result[2] = mBlue + dst[2] * (255 - (255 - dst[2]) * (255 - src[2]) / 255 - mBlue) / 255
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Multiplies or screens the colors, depending on the source color value. The effect is similar to shining a harsh
     * spotlight on the backdrop.
     */
    HardLight {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = if (src[0] < 128) dst[0] * dst[0] shr 7 else 255 - ((255 - dst[0]) * (255 - src[0]) shr 7)
            result[1] = if (dst[1] < 128) dst[1] * src[1] shr 7 else 255 - ((255 - dst[1]) * (255 - src[1]) shr 7)
            result[2] = if (dst[2] < 128) dst[2] * src[2] shr 7 else 255 - ((255 - dst[2]) * (255 - src[2]) shr 7)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Burns or dodges the colors by increasing or decreasing the contrast, depending on the blend color. If the blend
     * color (light source) is lighter than 50% gray, the image is lightened by decreasing the contrast. If the blend
     * color is darker than 50% gray, the image is darkened by increasing the contrast.
     */
    VividLight {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = if (src[0] < 128) if (src[0] == 0) 0 else max(
                0,
                255 - (255 - dst[0] shl 7) / src[0]
            ) else if (src[0] == 255) 255 else min(255, (dst[0] shl 7) / (255 - src[0]))
            result[1] = if (src[1] < 128) if (src[1] == 0) 0 else max(
                0,
                255 - (255 - dst[1] shl 7) / src[1]
            ) else if (src[1] == 255) 255 else min(255, (dst[1] shl 7) / (255 - src[1]))
            result[2] = if (src[2] < 128) if (src[2] == 0) 0 else max(
                0,
                255 - (255 - dst[2] shl 7) / src[2]
            ) else if (src[2] == 255) 255 else min(255, (dst[2] shl 7) / (255 - src[2]))
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Burns or dodges the colors by decreasing or increasing the brightness, depending on the blend color. If the
     * blend color (light source) is lighter than 50% gray, the image is lightened by increasing the brightness. If the
     * blend color is darker than 50% gray, the image is darkened by decreasing the brightness.
     */
    LinearLight {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = if (src[0] < 128) max(0, dst[0] + (src[0] shl 1) - 255) else min(255, dst[0] + (src[0] - 128 shl 1))
            result[1] = if (src[1] < 128) max(0, dst[1] + (src[1] shl 1) - 255) else min(255, dst[1] + (src[1] - 128 shl 1))
            result[2] = if (src[2] < 128) max(0, dst[2] + (src[2] shl 1) - 255) else min(255, dst[2] + (src[2] - 128 shl 1))
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Replaces the colors, depending on the blend color. If the blend color (light source) is lighter than 50% gray,
     * pixels darker than the blend color are replaced, and pixels lighter than the blend color do not change. If the
     * blend color is darker than 50% gray, pixels lighter than the blend color are replaced, and pixels darker than
     * the blend color do not change. This is useful for adding special effects to an image.
     */
    PinLight {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = if (src[0] < 128) min(dst[0], src[0] shl 1) else max(dst[0], src[0] - 128 shl 1)
            result[1] = if (src[1] < 128) min(dst[1], src[1] shl 1) else max(dst[1], src[1] - 128 shl 1)
            result[2] = if (src[2] < 128) min(dst[2], src[2] shl 1) else max(dst[2], src[2] - 128 shl 1)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Adds the red, green and blue channel values of the blend color to the RGB values of the base color. If the
     * resulting sum for a channel is 255 or greater, it receives a value of 255; if less than 255, a value of 0.
     * Therefore, all blended pixels have red, green, and blue channel values of either 0 or 255. This changes all
     * ixels to primary additive colors (red, green, or blue), white, or black.
     */
    HardMix {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = if (src[0] < 256 - dst[0]) 0 else 255
            result[1] = if (src[1] < 256 - dst[1]) 0 else 255
            result[2] = if (src[2] < 256 - dst[2]) 0 else 255
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Subtracts the darker of the two constituent colors from the lighter color. Painting with white inverts the
     * backdrop color; painting with black produces no change.
     */
    Difference {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = abs(dst[0] - src[0])
            result[1] = abs(dst[1] - src[1])
            result[2] = abs(dst[2] - src[2])
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Looks at the color information in each channel and subtracts the blend color from the base color.
     */
    Substract {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[0] = max(0, src[0] + dst[0] - 256)
            result[1] = max(0, src[1] + dst[1] - 256)
            result[2] = max(0, src[2] + dst[2] - 256)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Looks at the color information in each channel and divides the blend color from the base color.
     */
    Divide {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Creates a color with the hue of the source color and the saturation and luminosity of the backdrop color.
     */
    Hue {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            val srcHSL = FloatArray(3)
            RGBtoHSL(src[0], src[1], src[2], srcHSL)
            val dstHSL = FloatArray(3)
            RGBtoHSL(dst[0], dst[1], dst[2], dstHSL)
            HSLtoRGB(srcHSL[0], dstHSL[1], dstHSL[2], result)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Creates a color with the saturation of the source color and the hue and luminosity of the backdrop color.
     * Painting with this mode in an area of the backdrop that is a pure gray (no saturation) produces no change.
     */
    Saturation {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            val srcHSL = FloatArray(3)
            RGBtoHSL(src[0], src[1], src[2], srcHSL)
            val dstHSL = FloatArray(3)
            RGBtoHSL(dst[0], dst[1], dst[2], dstHSL)
            HSLtoRGB(dstHSL[0], srcHSL[1], dstHSL[2], result)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Creates a color with the hue and saturation of the source color and the luminosity of the backdrop color. This
     * preserves the gray levels of the backdrop and is useful for coloring monochrome images or tinting color images.
     */
    Color {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            val srcHSL = FloatArray(3)
            RGBtoHSL(src[0], src[1], src[2], srcHSL)
            val dstHSL = FloatArray(3)
            RGBtoHSL(dst[0], dst[1], dst[2], dstHSL)
            HSLtoRGB(srcHSL[0], srcHSL[1], dstHSL[2], result)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    },

    /**
     * Creates a color with the luminosity of the source color and the hue and saturation of the backdrop color.
     * This produces an inverse effect to that of the Color mode.
     */
    Luminosity {
        override fun blend(src: IntArray, dst: IntArray, result: IntArray) {
            val srcHSL = FloatArray(3)
            RGBtoHSL(src[0], src[1], src[2], srcHSL)
            val dstHSL = FloatArray(3)
            RGBtoHSL(dst[0], dst[1], dst[2], dstHSL)
            HSLtoRGB(dstHSL[0], dstHSL[1], srcHSL[2], result)
            result[3] = min(255, src[3] + dst[3] - src[3] * dst[3] / 255)
        }
    };

    /**
     * Blends the input colors into the result.
     *
     * @param src    the source RGBA (top)
     * @param dst    the destination RGBA (bottom)
     * @param result the result RGBA
     */
    abstract fun blend(src: IntArray, dst: IntArray, result: IntArray)

    companion object {
        /**
         *
         * Returns the HSL (Hue/Saturation/Luminance) equivalent of a given
         * RGB color. All three HSL components are floats between 0.0 and 1.0.
         *
         * @param r   the red component, between 0 and 255
         * @param g   the green component, between 0 and 255
         * @param b   the blue component, between 0 and 255
         * @param hsl a pre-allocated array of floats; can be null
         *
         * @return `hsl` if non-null, a new array of 3 floats otherwise
         *
         * @throws IllegalArgumentException if `hsl` has a length lower
         * than 3
         */
        private fun RGBtoHSL(r: Int, g: Int, b: Int, hsl: FloatArray): FloatArray {
            var r = r
            var g = g
            var b = b
            var hsl: FloatArray? = hsl
            if (hsl == null) {
                hsl = FloatArray(3)
            } else require(hsl.size >= 3) {
                "hsl array must have a length of" +
                        " at least 3"
            }
            if (r < 0) r = 0 else if (r > 255) r = 255
            if (g < 0) g = 0 else if (g > 255) g = 255
            if (b < 0) b = 0 else if (b > 255) b = 255
            val var_R = r / 255f
            val var_G = g / 255f
            val var_B = b / 255f
            var var_Min: Float
            var var_Max: Float
            val del_Max: Float
            if (var_R > var_G) {
                var_Min = var_G
                var_Max = var_R
            } else {
                var_Min = var_R
                var_Max = var_G
            }
            if (var_B > var_Max) {
                var_Max = var_B
            }
            if (var_B < var_Min) {
                var_Min = var_B
            }
            del_Max = var_Max - var_Min
            var H: Float
            val S: Float
            val L: Float
            L = (var_Max + var_Min) / 2f
            if (del_Max - 0.01f <= 0.0f) {
                H = 0f
                S = 0f
            } else {
                S = if (L < 0.5f) {
                    del_Max / (var_Max + var_Min)
                } else {
                    del_Max / (2 - var_Max - var_Min)
                }
                val del_R = ((var_Max - var_R) / 6f + del_Max / 2f) / del_Max
                val del_G = ((var_Max - var_G) / 6f + del_Max / 2f) / del_Max
                val del_B = ((var_Max - var_B) / 6f + del_Max / 2f) / del_Max
                H = if (var_R == var_Max) {
                    del_B - del_G
                } else if (var_G == var_Max) {
                    1 / 3f + del_R - del_B
                } else {
                    2 / 3f + del_G - del_R
                }
                if (H < 0) {
                    H += 1f
                }
                if (H > 1) {
                    H -= 1f
                }
            }
            hsl[0] = H
            hsl[1] = S
            hsl[2] = L
            return hsl
        }

        /**
         *
         * Returns the RGB equivalent of a given HSL (Hue/Saturation/Luminance)
         * color. All three RGB components are integers between 0 and 255.
         *
         * @param h   the hue component, between 0.0 and 1.0
         * @param s   the saturation component, between 0.0 and 1.0
         * @param l   the luminance component, between 0.0 and 1.0
         * @param rgb a pre-allocated array of ints; can be null
         *
         * @return `rgb` if non-null, a new array of 3 ints otherwise
         *
         * @throws IllegalArgumentException if `rgb` has a length lower
         * than 3
         */
        private fun HSLtoRGB(h: Float, s: Float, l: Float, rgb: IntArray): IntArray {
            var h = h
            var s = s
            var l = l
            var rgb: IntArray? = rgb
            if (rgb == null) {
                rgb = IntArray(3)
            } else require(rgb.size >= 3) {
                "rgb array must have a length of" +
                        " at least 3"
            }
            if (h < 0) h = 0.0f else if (h > 1.0f) h = 1.0f
            if (s < 0) s = 0.0f else if (s > 1.0f) s = 1.0f
            if (l < 0) l = 0.0f else if (l > 1.0f) l = 1.0f
            val R: Int
            val G: Int
            val B: Int
            if (s - 0.01f <= 0.0f) {
                R = (l * 255.0f).toInt()
                G = (l * 255.0f).toInt()
                B = (l * 255.0f).toInt()
            } else {
                val var_1: Float
                val var_2: Float
                var_2 = if (l < 0.5f) {
                    l * (1 + s)
                } else {
                    l + s - s * l
                }
                var_1 = 2 * l - var_2
                R = (255.0f * hue2RGB(var_1, var_2, h + 1.0f / 3.0f)).toInt()
                G = (255.0f * hue2RGB(var_1, var_2, h)).toInt()
                B = (255.0f * hue2RGB(var_1, var_2, h - 1.0f / 3.0f)).toInt()
            }
            rgb[0] = R
            rgb[1] = G
            rgb[2] = B
            return rgb
        }

        private fun hue2RGB(v1: Float, v2: Float, vH: Float): Float {
            var vH = vH
            if (vH < 0.0f) {
                vH += 1.0f
            }
            if (vH > 1.0f) {
                vH -= 1.0f
            }
            if (6.0f * vH < 1.0f) {
                return v1 + (v2 - v1) * 6.0f * vH
            }
            if (2.0f * vH < 1.0f) {
                return v2
            }
            return if (3.0f * vH < 2.0f) {
                v1 + (v2 - v1) * (2.0f / 3.0f - vH) * 6.0f
            } else v1
        }
    }
}