/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.statistics

import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.delegate.mutableLazy
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import com.macrofocus.order.OrderEvent
import com.macrofocus.order.OrderListener
import org.molap.dataframe.AbstractDataFrame
import org.molap.dataframe.DataFrame
import org.molap.dataframe.DataFrameEvent
import org.molap.dataframe.DataFrameListener
import org.molap.index.DefaultUniqueIndex
import org.molap.index.UniqueIndex
import org.molap.series.Series
import kotlin.reflect.KClass

class PresentationStatisticsDataFrame<Row, C>(
    private val dataFrame: StatisticsDataFrame<Row, C>,
    axisGroupModel: AxisGroupModel<Row, C>
) : AbstractDataFrame<StatisticsDataFrame.Stat, C?, Any?>() {
    private val axisGroupModel: AxisGroupModel<Row, C>
    override val rowIndex: UniqueIndex<StatisticsDataFrame.Stat>
        get() = dataFrame.rowIndex
    val columnIndexDelegate = mutableLazy {
        val list: MutableList<C?> = ArrayList<C?>()
        list.add(null)
        for (axisModel in axisGroupModel.axisOrder!!) {
            list.add(axisModel.column)
        }
        DefaultUniqueIndex<C?>(list, true)
    }
    override val columnIndex: UniqueIndex<C?> by columnIndexDelegate
    private val structureChangedTimer: CPTimer
    private val dataChangedTimer: CPTimer
    val dataFrameListener: DataFrameListener<StatisticsDataFrame.Stat, C> = object : DataFrameListener<StatisticsDataFrame.Stat, C> {
        override fun dataFrameChanged(event: DataFrameEvent<StatisticsDataFrame.Stat, C>) {
            scheduleTableDataChanged()
        }
    }

    private fun scheduleTableStructureChanged() {
        structureChangedTimer.restart()
    }

    private fun scheduleTableDataChanged() {
        dataChangedTimer.restart()
    }

    override fun getColumnName(column: C?): String? {
        return dataFrame.getColumnName(column!!)
    }

    override fun getColumnClass(columnIndex: C?): KClass<out Any> {
        return if (columnIndex != null) {
            dataFrame.getColumnClass(columnIndex)!!
        } else String::class as KClass<Any>
    }

    override val rowCount: Int
        get() = dataFrame.rowCount

    override val columnCount: Int
        get() = axisGroupModel.axisOrder!!.size() + 1

    override fun getValueAt(rowIndex: StatisticsDataFrame.Stat, columnIndex: C?): Any? {
        return when (getColumnAddress(columnIndex)) {
            0 -> rowIndex.toString()
            else -> dataFrame.getValueAt(rowIndex, columnIndex!!)
        }
    }

    override fun getRowClass(integer: StatisticsDataFrame.Stat): KClass<Any> {
        return Any::class
    }

    fun getRow(integer: StatisticsDataFrame.Stat?): Series<C, *>? {
        return null
    }

    fun join(series: Series<*,*>?, cs: Array<Any?>?): DataFrame<*, *, *>? {
        return null
    }

    init {
        this.axisGroupModel = axisGroupModel
        structureChangedTimer = CPHelper.instance.createTimer("StatisticsPresentation\$Structure", 40, true, object : CPTimerListener {
            override fun timerTriggered() {
                columnIndexDelegate.forget()
                notifyDataFrameChanged(DataFrameEvent(null, null, true))
            }
        })
        dataChangedTimer = CPHelper.instance.createTimer("StatisticsPresentation\$Data", 40, true, object : CPTimerListener {
            override fun timerTriggered() {
                notifyDataFrameChanged(DataFrameEvent(null, null, false))
            }
        })
        dataFrame.addWeakDataFrameListener(dataFrameListener)
        axisGroupModel.axisOrder!!.addOrderListener(object : OrderListener<AxisModel<Row, C>> {
            override fun orderChanged(event: OrderEvent<AxisModel<Row, C>>?) {
                scheduleTableStructureChanged()
            }

            override fun orderVisibility(event: OrderEvent<AxisModel<Row, C>>) {
                scheduleTableStructureChanged()
            }

            override fun orderAdded(event: OrderEvent<AxisModel<Row, C>>) {
                scheduleTableStructureChanged()
            }

            override fun orderRemoved(event: OrderEvent<AxisModel<Row, C>>) {
                scheduleTableStructureChanged()
            }
        })
    }
}