/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.distributions

import com.macrofocus.common.collection.RandomAccessIterable
import com.macrofocus.common.collection.SimpleImmutableEntry
import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.properties.Property
import com.macrofocus.common.properties.PropertyEvent
import com.macrofocus.common.properties.PropertyListener
import com.macrofocus.common.selection.SelectionEvent
import com.macrofocus.common.selection.SelectionListener
import com.macrofocus.common.selection.SingleSelectionEvent
import com.macrofocus.common.selection.SingleSelectionListener
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.order.MutableVisibleOrder
import com.macrofocus.order.OrderEvent
import com.macrofocus.order.OrderListener
import org.mkui.canvas.CPCanvas
import org.mkui.color.MkColor
import org.mkui.color.MkColorFactory
import org.mkui.component.CPComponent
import org.mkui.geom.Point2D
import org.mkui.geom.Rectangle
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.AbstractIDrawing
import org.mkui.graphics.IGraphics
import org.mkui.graphics.colortheme.ColorTheme
import org.mkui.palette.Palette
import org.mkui.rubberband.RubberbandDrawing
import org.molap.subset.DimensionEvent
import org.molap.subset.DimensionListener
import org.molap.subset.DistributionDimension
import org.molap.subset.SingleBinningDimension
import kotlin.math.max

/**
 * Created by luc on 12/06/16.
 */
abstract class AbstractDistributionsComponent<Row, Column, Value, Bin>(
    protected val view: DistributionsView<Row, Column, Value, Bin>,
) : DistributionsComponent<Row, Column, Value, Bin> {
    private enum class Scale {
        Global, Local
    }

    override val component: CPComponent
        get() = canvas

    override var model: DistributionsModel<Row, Column, Value, Bin>? = null
        set(value) {
            if (field != null) {
                field!!.removeDistributionsListener(listener)
                field!!.getAxisGroupModel().axisOrder!!.removeOrderListener(orderListener)
            }
            field = value
            if (value != null) {
                createOverplots()
                field!!.addDistributionsListener(listener)
                field!!.getAxisGroupModel().axisOrder!!.addOrderListener(orderListener)
            }
            axisHistogramCache.clear()
            if (value != null) {
                timer.restart()
            }
        }
    protected val canvas: CPCanvas by lazy { CPCanvas() }
    private val axisHistogramCache: MutableMap<AxisModel<*, *>, Histogram<Row>> =
        HashMap<AxisModel<*, *>, Histogram<Row>>()
    private val margin = 0.2
    private val binSize = 10
    private val scale = Scale.Local
    protected val timer: CPTimer =
        CPHelper.instance.createTimer("DistributionResizer", 40, true, object : CPTimerListener {
            override fun timerTriggered() {
                if (getWidth() > 0 && getHeight() > 0) {
                    refresh()
                }
            }
        })

    protected enum class State {
        /**
         * No bar has been selected and there are elements of this bar that haven't been filtered out.
         */
        Active,

        /**
         * It is active and the mouse is over it.
         */
        Probed,

        /**
         * It is active and has been selected by the user, i.e. other bars in this filter are filtered out.
         */
        Selected, ProbedSelected,

        /**
         * This filter has another bar that is active, making the elements from this bar being filtered out. The user
         * could however activate it by clicking on it.
         */
        Selectable, ProbedSelectable,

        /**
         * All the elements of this bar are filtered out and the user is not allowed to activate it.
         */
        Inactive
    }

    private val listener: DistributionsListener = object : DistributionsListener {
        override fun distributionsChanged() {
            timer.restart()
        }
    }
    private val orderListener: OrderListener<AxisModel<Row, Column>> = object : OrderListener<AxisModel<Row, Column>> {
        override fun orderChanged(event: OrderEvent<AxisModel<Row, Column>>?) {
            scheduleUpdate()
        }

        override fun orderVisibility(event: OrderEvent<AxisModel<Row, Column>>) {
            scheduleUpdate()
        }

        override fun orderAdded(event: OrderEvent<AxisModel<Row, Column>>) {
            scheduleUpdate()
        }

        override fun orderRemoved(event: OrderEvent<AxisModel<Row, Column>>) {
            scheduleUpdate()
        }
    }

    protected fun getWidth(): Int {
        return canvas.getWidth().toInt()
    }

    protected fun getHeight(): Int {
        return canvas.getHeight().toInt()
    }

    protected fun repaint() {
        canvas.redraw()
    }

    override fun scheduleUpdate() {
        timer.restart()
        repaint()
    }

    private fun getHistogram(axisModel: AxisModel<Row, Column>): Histogram<Row> {
        if (!axisHistogramCache.containsKey(axisModel)) {
            val histogram: FixedBinsHistogram<*>
            if (axisModel.isNumerical) {
                val binCount: Property<Int> = axisModel.binCount!!
                binCount.addPropertyListener(object : PropertyListener<Int> {
                    override fun propertyChanged(event: PropertyEvent<Int>) {
                        axisHistogramCache.remove(axisModel)
                        timer.restart()
                    }
                })
                histogram = FixedBinsHistogram<Row>(
                    binCount?.value ?: getHeight() / binSize,
                    axisModel.minimum,
                    axisModel.maximum,
                    model!!.getFilter()
                )
            } else {
                histogram = FixedBinsHistogram<Row>(
                    (axisModel.maximum - axisModel.minimum).toInt() + 1,
                    axisModel.minimum,
                    axisModel.maximum,
                    model!!.getFilter()
                )
            }
            for (row in axisModel) {
                val value: Number? = axisModel.getValue(row)
                if (value != null) {
                    histogram.addValue(row, value.toDouble())
                }
            }
            axisHistogramCache[axisModel] = histogram
        }
        return axisHistogramCache[axisModel]!!
    }

    protected open fun refresh() {
        clearCache()
        repaint()
    }

    override fun clearCache() {
        axisHistogramCache.clear()
    }

    override fun getClosestDistribution(x: Int, y: Int): DistributionDimension<Row, Value, Bin>? {
        val axisOrder: MutableVisibleOrder<AxisModel<Row, Column>> = model!!.getAxisGroupModel().axisOrder!!
        for (i in 0 until axisOrder.size()) {
            val xAxisModel: AxisModel<Row, Column> = axisOrder.get(i)
            val x1Location: Double? = model?.getLocation(xAxisModel)
            var x2Location: Double?
            x2Location = if (i + 1 < axisOrder.size()) {
                model?.getLocation(axisOrder.get(i + 1))
            } else {
                1.0
            }
            if (x1Location != null && x2Location != null) {
                // Histograms shifted by half a plate to the right
                val x1 = (x1Location * getWidth()).toInt() + 1
                val w = ((x2Location * getWidth() - x1) * (1.0 - margin)).toInt()
                val x2 = x1 + w
                if (x >= x1 && x <= x2) {
                    return view.getDistributionDimension(xAxisModel)
                }
            }
        }
        return null
    }

    override fun getClosestBin(x: Int, y: Int): Bin? {
        val axisOrder: MutableVisibleOrder<AxisModel<Row, Column>> = model!!.getAxisGroupModel().axisOrder!!
        for (i in 0 until axisOrder.size()) {
            val xAxisModel: AxisModel<Row, Column> = axisOrder.get(i)
            val x1Location: Double? = model?.getLocation(xAxisModel)
            var x2Location: Double?
            x2Location = if (i + 1 < axisOrder.size()) {
                model?.getLocation(axisOrder.get(i + 1))
            } else {
                1.0
            }
            if (x1Location != null && x2Location != null) {
                // Histograms shifted by half a plate to the right
                val x1 = (x1Location * getWidth()).toInt() + 1
                val w = ((x2Location * getWidth() - x1) * (1.0 - margin)).toInt()
                val x2 = x1 + w
                if (x >= x1 && x <= x2) {
                    val dimension: DistributionDimension<Row, Value, Bin> = view.getDistributionDimension(xAxisModel)!!
                    val j = 0
                    for (bin in dimension.bins!!) {
                        val y1 =
                            getHeight() - ((dimension.getBinStartValue(bin) - dimension.minValue) * getHeight() / (dimension.maxValue - dimension.minValue)).toInt()
                        val y2 =
                            getHeight() - ((dimension.getBinEndValue(bin) - dimension.minValue) * getHeight() / (dimension.maxValue - dimension.minValue)).toInt()
                        if (y >= y2 && y <= y1) {
                            return bin
                        }
                    }
                }
            }
        }
        return null
    }

    override fun getBins(rect: Rectangle2D): List<Bin>? {
        var list: MutableList<Bin>? = null
        val axisOrder: MutableVisibleOrder<AxisModel<Row, Column>> = model!!.getAxisGroupModel()!!.axisOrder!!
        for (i in 0 until axisOrder.size()) {
            val xAxisModel: AxisModel<Row, Column> = axisOrder.get(i)
            val x1Location: Double? = model?.getLocation(xAxisModel)
            var x2Location: Double?
            x2Location = if (i + 1 < axisOrder.size()) {
                model?.getLocation(axisOrder.get(i + 1))
            } else {
                1.0
            }
            if (x1Location != null && x2Location != null) {
                // Histograms shifted by half a plate to the right
                val x1 = (x1Location * getWidth()).toInt() + 1
                val w = ((x2Location * getWidth() - x1) * (1.0 - margin)).toInt()
                val x2 = x1 + w
                if (rect.x >= x1 && rect.x <= x2) {
                    val dimension: DistributionDimension<Row, Value, Bin>? = view.getDistributionDimension(xAxisModel)
                    for (bin in dimension!!.bins!!) {
                        val y1 =
                            getHeight() - ((dimension.getBinStartValue(bin) - dimension.minValue) * getHeight() / (dimension.maxValue - dimension.minValue)).toInt()
                        val y2 =
                            getHeight() - ((dimension.getBinEndValue(bin) - dimension.minValue) * getHeight() / (dimension.maxValue - dimension.minValue)).toInt()
                        if (rect.y >= y2 && rect.y <= y1) {
                            if (list == null) {
                                list = ArrayList<Bin>()
                            }
                            list.add(bin)
                        }
                    }
                }
            }
        }
        return list
    }

    override fun getClosestRow(x: Int, y: Int): Row? {
        val axisOrder: MutableVisibleOrder<AxisModel<Row, Column>> = model!!.getAxisGroupModel().axisOrder!!
        var max_count = 0
        if (scale == Scale.Global) {
            for (i in 0 until axisOrder.size()) {
                val xAxisModel: AxisModel<Row, Column> = axisOrder.get(i)
                val histogram: Histogram<*> = getHistogram(xAxisModel)
                max_count = max(max_count, histogram.getMaxDensity())
            }
        }
        for (i in 0 until axisOrder.size()) {
            val xAxisModel: AxisModel<Row, Column> = axisOrder.get(i)
            val x1Location: Double = model!!.getLocation(xAxisModel)
            var x2Location: Double
            x2Location = if (i + 1 < axisOrder.size()) {
                model!!.getLocation(axisOrder.get(i + 1))
            } else {
                1.0
            }
            if (x1Location != null && x2Location != null) {
                // Histograms shifted by half a plate to the right
                val x1 = (x1Location * getWidth()).toInt() + 1
                val w = ((x2Location * getWidth() - x1) * (1.0 - margin)).toInt()
                val x2 = x1 + w
                if (x >= x1 && x <= x2) {
                    val histogram: Histogram<Row> = getHistogram(xAxisModel)
                    for (j in 0 until histogram.getNumberOfBins()) {
                        val y1 =
                            getHeight() - ((histogram.getBinStartValue(j) - histogram.getMinValue()) * getHeight() / (histogram.getMaxValue() - histogram.getMinValue())).toInt()
                        val y2 =
                            getHeight() - ((histogram.getBinEndValue(j) - histogram.getMinValue()) * getHeight() / (histogram.getMaxValue() - histogram.getMinValue())).toInt()
                        if (y >= y2 && y <= y1) {
                            val activeCountAtBin: Int = histogram.getActiveDensity(j)
                            if (activeCountAtBin > 0) {
                                val max_width = x2 - x1
                                if (scale == Scale.Local) {
                                    max_count = histogram.getMaxDensity()
                                }
                                val activeWidth = activeCountAtBin * max_width / max_count
                                if (activeWidth > 0) {
                                    val index = (x - x1) * (activeCountAtBin - 1) / activeWidth
                                    return if (index >= 0 && index < activeCountAtBin) {
                                        histogram.getActiveRowAtBin(j, index)
                                    } else {
                                        null
                                    }
                                }
                            } else {
                                return null
                            }
                        }
                    }
                }
            }
        }
        return null
    }

    protected fun getState(dimension: SingleBinningDimension<Row, Bin>, bin: Bin): State {
        val selected = !dimension.filterExact!!.isActive || dimension.filterExact!!.isSelected(bin)
        return if (selected && dimension.getActiveDensity(bin) > 0) {
            if (view.getProbing().isActive && view.getProbing()
                    .isSelected(SimpleImmutableEntry<SingleBinningDimension<Row, Bin>, Bin>(dimension, bin))
            ) {
                if (dimension.filterExact!!.isSelected(bin)) {
                    State.ProbedSelected
                } else {
                    State.Probed
                }
            } else if (dimension.filterExact!!.isSelected(bin)) {
                State.Selected
            } else {
                State.Active
            }
        } else {
            if (dimension.getActiveDensity(bin) > 0) {
                if (view.getProbing().isActive && view.getProbing()
                        .isSelected(SimpleImmutableEntry<SingleBinningDimension<Row, Bin>, Bin>(dimension, bin))
                ) {
                    State.ProbedSelectable
                } else {
                    State.Selectable
                }
            } else {
                State.Inactive
            }
        }
    }

    override fun createOverplots() {
        if (model != null) {
            canvas.removeAllLayers()
            canvas.addLayer(HistogramIDrawing())
            canvas.addLayer(object : RubberbandDrawing(view.getRubberBand()) {
                override val colorTheme: Property<ColorTheme>
                    get() = view.getColorTheme()
            })
        }
    }

    private inner class HistogramIDrawing : AbstractIDrawing() {
        private val selectionListener: SelectionListener<Bin> = object : SelectionListener<Bin> {
            override fun selectionChanged(event: SelectionEvent<Bin>) {
                notifyIDrawingChanged()
            }
        }
        val dimensionListener: DimensionListener<Row> = object : DimensionListener<Row> {
            override fun dimensionChanged(event: DimensionEvent<Row>) {
                notifyIDrawingChanged()
            }

            override fun selectedCountChanged() {
                notifyIDrawingChanged()
            }
        }
        val probingListener: SingleSelectionListener<SimpleImmutableEntry<SingleBinningDimension<Row, Bin>, Bin>> =
            object : SingleSelectionListener<SimpleImmutableEntry<SingleBinningDimension<Row, Bin>, Bin>> {
                override fun selectionChanged(event: SingleSelectionEvent<SimpleImmutableEntry<SingleBinningDimension<Row, Bin>, Bin>>) {
                    notifyIDrawingChanged()
                }
            }

        override fun draw(g: IGraphics, point: Point2D?, width: Double, height: Double, clipBounds: Rectangle) {
            g.setColor(view.getColorTheme().value.background)
            g.fillRectangle(0, 0, getWidth(), getHeight())
            val axisOrder: MutableVisibleOrder<AxisModel<Row, Column>>? = model?.getAxisGroupModel()?.axisOrder
            if (axisOrder != null) {
                var max_count = 0.0
                if (scale == Scale.Global) {
                    for (i in 0 until axisOrder.size()) {
                        val xAxisModel: AxisModel<Row, Column> = axisOrder.get(i)
                        val dimension: DistributionDimension<Row, Value, Bin> =
                            view.getDistributionDimension(xAxisModel)!!
                        max_count = if (view.getShowFiltered().value) {
                            max(max_count, dimension.maxDensity)
                        } else {
                            max(max_count, dimension.maxActiveDensity)
                        }
                    }
                }
                for (i in 0 until axisOrder.size()) {
                    val xAxisModel: AxisModel<Row, Column> = axisOrder.get(i)
                    val x1Location: Double = model!!.getLocation(xAxisModel)
                    var x2Location: Double
                    x2Location = if (i + 1 < axisOrder.size()) {
                        model!!.getLocation(axisOrder.get(i + 1))
                    } else {
                        1.0
                    }
                    if (x1Location != null && x2Location != null) {
                        val dimension: DistributionDimension<Row, Value, Bin>? =
                            view.getDistributionDimension(xAxisModel)

                        // Histograms shifted by half a plate to the right
                        val x1 = (x1Location * getWidth()).toInt() + 1
                        val w = ((x2Location * getWidth() - x1) * (1.0 - margin)).toInt()
                        val x2 = x1 + w
                        g.setColor(view.getColorTheme().value.ghostedPalette.getColor(1.0))
                        g.drawLine(x1 - 1, 0, x1 - 1, getHeight())

                        // Draw the bar dimension
                        val max_width = x2 - x1
                        if (scale == Scale.Local) {
                            max_count = if (view.getShowFiltered().value) {
                                dimension!!.maxDensity
                            } else {
                                dimension!!.maxActiveDensity
                            }
                        }
                        var x_prev = x1
                        var active_x_prev = x1
                        val bins: RandomAccessIterable<Bin> = dimension!!.bins!!
                        val it: Iterator<Bin> = bins.iterator()
                        var j = 0
                        while (it.hasNext()) {
                            val bin = it.next()
                            val state = getState(dimension, bin)
                            val outlineColor: MkColor?
                            val fillColor: MkColor?
                            val labelColor: MkColor?
                            val palette: Palette = view.getColorTheme().value.palette
                            when (state) {
                                State.Active -> {
                                    outlineColor = null
                                    fillColor = palette.getColor(0.3)
                                    labelColor = palette.getColor(0.7)
                                }

                                State.Probed -> {
                                    outlineColor = view.getColorTheme().value.probing
                                    fillColor = palette.getColor(0.3)
                                    labelColor = view.getColorTheme().value.probing
                                }

                                State.Selected -> {
                                    outlineColor = null
                                    fillColor = palette.getColor(0.55)
                                    labelColor = palette.getColor(1.0)
                                }

                                State.ProbedSelected -> {
                                    outlineColor = view.getColorTheme().value.probing
                                    fillColor = palette.getColor(0.55)
                                    labelColor = view.getColorTheme().value.probing
                                }

                                State.Selectable -> {
                                    // Not active, but nevertheless selectable as alternate to the current filter, i.e. this item can be selected by the user
                                    outlineColor = null
                                    fillColor = palette.getColor(0.1)
                                    labelColor = palette.getColor(0.3)
                                }

                                State.ProbedSelectable -> {
                                    outlineColor = view.getColorTheme().value.probing
                                    fillColor = palette.getColor(0.1)
                                    labelColor = palette.getColor(0.7)
                                }

                                State.Inactive -> {
                                    outlineColor = null
                                    fillColor = palette.getColor(0.05)
                                    labelColor = palette.getColor(0.2)
                                }

                                else -> {
                                    outlineColor = null
                                    fillColor = MkColorFactory.instance.gray
                                    labelColor = null
                                }
                            }
                            if (dimension.getDensity(bin) > 0) {
                                val width = (dimension.getDensity(bin) * max_width / max_count) as Int
                                val binStartValue: Double = dimension.getBinStartValue(bin)
                                val y1 =
                                    getHeight() - ((binStartValue - dimension.minValue) * getHeight() / (dimension.maxValue - dimension.minValue)) as Int
                                val binEndValue: Double = dimension.getBinEndValue(bin)
                                val y2 =
                                    getHeight() - ((binEndValue - dimension.minValue) * getHeight() / (dimension.maxValue - dimension.minValue)) as Int
                                if (view.getShowFiltered().value) {
                                    g.setColor(view.getColorTheme().value.ghostedPalette.getColor(0.0))
                                    g.fillRectangle(x1, y2, width, y1 - y2)
                                }
                                val activeCountAtBin: Double = dimension.getActiveDensity(bin)
                                val activeWidth = (activeCountAtBin * max_width / max_count).toInt()
                                g.setColor(fillColor!!)
                                g.fillRectangle(x1, y2, activeWidth, y1 - y2)
                                if (outlineColor != null) {
                                    g.setColor(outlineColor)
                                    g.drawRectange(x1, y2, activeWidth, y1 - y2)
                                }
                                val selectionCountAtBin: Double = dimension.getSelectionDensity(bin)
                                if (selectionCountAtBin > 0) {
                                    val selectionWidth = (selectionCountAtBin * max_width / max_count).toInt()
                                    g.setColor(view.getColorTheme().value.selection)
                                    g.fillRectangle(x1, y2, selectionWidth, y1 - y2)
                                }
                                if (view.getShowFiltered().value) {
                                    g.setColor(view.getColorTheme().value.ghostedPalette.getColor(1.0))
                                    g.drawLine(x_prev, y1, x1 + width, y1)
                                    g.drawLine(x1 + width, y1, x1 + width, y2)
                                    x_prev =
                                        if (j == bins.size() - 1 || dimension.getBinStartValue(bins.get(j + 1)) !== binEndValue) {
                                            g.drawLine(x1 + width, y2, x1, y2)
                                            x1
                                        } else x1 + width
                                }
                                g.setColor(view.getColorTheme().value.visiblePalette.getColor(1.0))
                                g.drawLine(active_x_prev, y1, x1 + activeWidth, y1)
                                g.drawLine(x1 + activeWidth, y1, x1 + activeWidth, y2)
                                active_x_prev =
                                    if (!it.hasNext() || dimension!!.getBinStartValue(bins.get(j + 1)) !== binEndValue) {
                                        g.drawLine(x1 + activeWidth, y2, x1, y2)
                                        x1
                                    } else x1 + activeWidth
                                if (activeCountAtBin > 0) {
                                    var r = 0
                                    while (r < activeCountAtBin) {
                                        r++
                                    }
                                }
                            }
                            j++
                        }
                    }
                }
            }
        }

        init {
            val axisOrder: MutableVisibleOrder<AxisModel<Row, Column>>? = model!!.getAxisGroupModel().axisOrder
            for (i in 0 until axisOrder!!.size()) {
                val xAxisModel: AxisModel<Row, Column> = axisOrder.get(i)
                val dimension: SingleBinningDimension<Row, Bin>? = view.getDistributionDimension(xAxisModel)
                dimension!!.addDimensionListener(dimensionListener)
                dimension!!.filterExact!!.addWeakSelectionListener(selectionListener)
            }
            view.getProbing().addWeakSingleSelectionListener(probingListener)
        }
    }
}