/*
 * Copyright (c) 2015 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.distributions

import com.macrofocus.common.collection.SimpleImmutableEntry
import com.macrofocus.common.properties.*
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.common.selection.SimpleSingleSelection
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.BinType
import com.macrofocus.high_d.distributions.DistributionsView.Orientation
import com.macrofocus.high_d.distributions.DistributionsView.PropertyType
import com.macrofocus.order.OrderEvent
import com.macrofocus.order.OrderListener
import org.mkui.canvas.Rendering
import org.mkui.graphics.colortheme.ColorTheme
import org.mkui.graphics.colortheme.LightColorTheme
import org.mkui.interaction.InteractionMode
import org.mkui.rubberband.RubberBand
import org.mkui.rubberband.SimpleRubberBand
import org.molap.subset.*
import kotlin.math.sqrt

abstract class AbstractDistributionsView<Row, Column, Value, Bin>() :
    DistributionsView<Row, Column, Value, Bin> {
    override var model: DistributionsModel<Row, Column, Value, Bin>? = null
    set(value) {
        if (field != null) {
            field!!.getAxisGroupModel().axisOrder!!.removeOrderListener(orderListener)
        }
        field = value
        distibutionsComponent!!.model = value
        if (field != null) {
            field!!.getAxisGroupModel().axisOrder!!.addOrderListener(orderListener)
        }
    }
    private val rubberBand: RubberBand
    private var isSelectionMode = true
    protected val properties: MutableProperties<PropertyType> = EnumProperties<PropertyType>(PropertyType.values())
    private val binnedDimensions: MutableMap<Column, DistributionDimension<Row, Value, Bin>?> = mutableMapOf<Column, DistributionDimension<Row, Value, Bin>?>()
    val probing: SimpleSingleSelection<SimpleImmutableEntry<SingleBinningDimension<Row, Bin>, Bin>> =
        SimpleSingleSelection<SimpleImmutableEntry<SingleBinningDimension<Row, Bin>, Bin>>()
    private val propertiesListener: PropertiesListener<PropertyType> = object : PropertiesListener<PropertyType> {
        override fun propertyChanged(name: PropertyType, event: PropertyEvent<Any?>) {
            createOverplots()
        }
    }
    private val orderListener: OrderListener<AxisModel<Row, Column>> = object : OrderListener<AxisModel<Row, Column>> {
        override fun orderChanged(event: OrderEvent<AxisModel<Row, Column>>?) {
            refresh()
        }

        override fun orderVisibility(event: OrderEvent<AxisModel<Row, Column>>) {
            refresh()
        }

        override fun orderAdded(event: OrderEvent<AxisModel<Row, Column>>) {
            refresh()
        }

        override fun orderRemoved(event: OrderEvent<AxisModel<Row, Column>>) {
            for (axisModel in event.affected) {
                removeBinnedDimension(axisModel.column)
            }
            refresh()
        }
    }

    override fun isSelectionMode(): Boolean {
        return isSelectionMode
    }

    override fun setSelectionMode(value: Boolean) {
        isSelectionMode = value
    }

    override fun getProbing(): MutableSingleSelection<SimpleImmutableEntry<SingleBinningDimension<Row, Bin>, Bin>> {
        return probing
    }

    protected abstract fun refresh()
    protected abstract val distibutionsComponent: DistributionsComponent<Row, Column, Value, Bin>?
    override fun getOrientation(): Property<Orientation?> {
        return properties.getProperty(PropertyType.Orientation) as Property<Orientation?>
    }

    override fun getAntialiasing(): MutableProperty<Boolean> {
        return properties.getProperty(PropertyType.Antialiasing) as MutableProperty<Boolean>
    }

    override fun getShowFiltered(): Property<Boolean> {
        return properties.getProperty(PropertyType.ShowFiltered) as MutableProperty<Boolean>
    }

    override fun getShowTiming(): MutableProperty<Boolean> {
        return properties.getProperty(PropertyType.ShowTiming) as MutableProperty<Boolean>
    }

    override fun getColorTheme(): MutableProperty<ColorTheme> {
        return properties.getProperty(PropertyType.ColorTheme) as MutableProperty<ColorTheme>
    }

    override fun getInteractionMode(): MutableProperty<InteractionMode?> {
        return properties.getProperty(PropertyType.InteractionMode) as MutableProperty<InteractionMode?>
    }

    override fun setInteractionMode(interactionMode: MutableProperty<InteractionMode>?) {
        properties.replaceProperty<InteractionMode>(PropertyType.InteractionMode, interactionMode!!)
    }

    override fun setColorTheme(colorTheme: MutableProperty<ColorTheme>?) {
        properties.replaceProperty<ColorTheme>(PropertyType.ColorTheme, colorTheme!!)
    }

    override fun setShowFiltered(showFiltered: MutableProperty<Boolean>?) {
        properties.replaceProperty<Boolean>(PropertyType.ShowFiltered, showFiltered!!)
    }

    override fun setAntialiasing(antialiasing: MutableProperty<Boolean>?) {
        properties.replaceProperty<Boolean>(PropertyType.Antialiasing, antialiasing!!)
    }

    override fun setShowTiming(showTiming: MutableProperty<Boolean>?) {
        properties.replaceProperty<Boolean>(PropertyType.ShowTiming, showTiming!!)
    }

    override fun getRendering(): MutableProperty<Rendering> {
        return properties.getProperty(PropertyType.Rendering) as MutableProperty<Rendering>
    }

    override fun setRendering(rendering: MutableProperty<Rendering>?) {
        properties.replaceProperty<Rendering>(PropertyType.Rendering, rendering!!)
    }

    override fun getDistributionDimension(axisModel: AxisModel<Row, Column>): DistributionDimension<Row, Value, Bin>? {
        val column: Column = axisModel.column
        return if (binnedDimensions.containsKey(column)) {
            binnedDimensions[column]
        } else {
            val dimension: DistributionDimension<Row, Value, Bin>?
            if (axisModel.isNumerical) {
                val binType: MutableProperty<BinType>? = axisModel.binType
                val binCount: MutableProperty<Int>? = axisModel.binCount
                if (binType != null) {
                    val propertyListener: PropertyListener<BinType> = object : PropertyListener<BinType> {
                        override fun propertyChanged(event: PropertyEvent<BinType>) {
                            binType.removePropertyListener(this)
                            removeBinnedDimension(axisModel.column)
                            refresh()
                        }
                    }
                    binType.addPropertyListener(propertyListener)
                }
                if (binCount != null) {
                    val propertyListener: PropertyListener<Int> = object : PropertyListener<Int> {
                        override fun propertyChanged(event: PropertyEvent<Int>) {
                            binCount.removePropertyListener(this)
                            removeBinnedDimension(axisModel.column)
                            refresh()
                        }
                    }
                    binCount.addPropertyListener(propertyListener)
                }
                val type: BinType
                type = if (binType != null) {
                    when (binType.value) {
                        BinType.Auto -> binType.value
                        BinType.Fixed -> if (binCount != null && binCount.value > 1) {
                            binType.value
                        } else {
                            BinType.Auto
                        }
                        BinType.AutoSigma -> binType.value
                        BinType.Sigma -> if (binCount != null && binCount.value > 1) {
                            binType.value
                        } else {
                            BinType.AutoSigma
                        }
                        else -> BinType.Auto
                    }
                } else {
                    BinType.Auto
                }
                when (type) {
                    BinType.Auto -> {
                        val values : Array<Number?> = arrayOfNulls<Number>(axisModel.rowCount)
                        var count = 0
                        var i = 0
                        while (i < axisModel.rowCount) {
                            val value: Number? = axisModel.getValue(axisModel.getRow(i))
                            if (value != null) {
                                values[count] = value
                                count++
                            }
                            i++
                        }
                        if (count > 0) {
                            var notNullValues = values.filterNotNull().requireNoNulls()
                            notNullValues = notNullValues.sortedWith(compareBy({ it.toDouble() }))
//                            values.sortWith(nullsLast(compareBy({ it.toDouble() })), 0, count)

                            val distributionStrategy = GenedataDistributionStrategy<Number?>(notNullValues.toTypedArray(), count) as DistributionDimension.DistributionStrategy<Value,Bin>
                            dimension = model!!.createDistributionDimension(column, distributionStrategy)
                        } else {
                            // Fallback to Fixed scheme
                            dimension = model!!.createDistributionDimension(
                                column,
                                FixedDistributionStrategy<Number?>(
                                    if (binCount != null) binCount.value else sqrt(axisModel.rowCount.toDouble()).toInt(),
                                    axisModel.minimum,
                                    axisModel.maximum
                                ) as DistributionDimension.DistributionStrategy<Value,Bin>
                            )
                        }
                    }
                    BinType.Fixed -> dimension = model!!.createDistributionDimension(
                        column,
                        FixedDistributionStrategy<Number?>(
                            if (binCount != null) binCount.value else sqrt(axisModel.rowCount.toDouble()).toInt(),
                            axisModel.minimum,
                            axisModel.maximum
                        ) as DistributionDimension.DistributionStrategy<Value,Bin>
                    )
                    BinType.AutoSigma -> dimension = model!!.createDistributionDimension(
                        column,
                        AutoSigmaDistributionStrategy<Number?>(
                            if (binCount != null) binCount.value else sqrt(axisModel.rowCount.toDouble()).toInt(),
                            axisModel.minimum,
                            axisModel.mean,
                            axisModel.stdDev,
                            axisModel.maximum
                        ) as DistributionDimension.DistributionStrategy<Value,Bin>
                    )
                    BinType.Sigma -> dimension = model!!.createDistributionDimension(
                        column,
                        FixedSigmaDistributionStrategy<Number?>(
                            if (binCount != null) binCount.value else sqrt(axisModel.rowCount.toDouble()).toInt(),
                            axisModel.minimum,
                            axisModel.mean,
                            axisModel.stdDev,
                            axisModel.maximum
                        ) as DistributionDimension.DistributionStrategy<Value,Bin>
                    )
                    else -> dimension = null
                }
            } else {
                val distributionStrategy =
                    FixedDistributionStrategy<Number?>((axisModel.maximum - axisModel.minimum).toInt() + 2, axisModel.minimum, axisModel.maximum) as DistributionDimension.DistributionStrategy<Value,Bin>
                dimension = model!!.createDistributionDimension(column, distributionStrategy)
                dimension.binningStrategy =object : SingleBinningDimension.SingleBinningStrategy<Row, Bin> {
                    override fun isBinnable(row: Row): Boolean {
                        return distributionStrategy.isBinnable(axisModel.getValue(row) as Value)
                    }

                    override fun rowToBin(row: Row): Bin {
                        return distributionStrategy.valueToBin(axisModel.getValue(row) as Value) as Bin
                    }
                }
            }
            binnedDimensions[column] = dimension
            dimension
        }
    }

    protected fun removeBinnedDimension(column: Column) {
        if (binnedDimensions.containsKey(column)) {
            model!!.removeDistributionDimension(binnedDimensions[column]!!)
            binnedDimensions.remove(column)
        }
    }

    private fun createOverplots() {
        if (distibutionsComponent != null) {
            distibutionsComponent!!.createOverplots()
            distibutionsComponent!!.scheduleUpdate()
        }
    }

    override fun getRubberBand(): RubberBand {
        return rubberBand
    }

    init {
        properties.createProperty<Any>(PropertyType.Orientation, Orientation.Horizontal)
        properties.createProperty<Boolean>(PropertyType.ShowFiltered, true)
        properties.createProperty<Boolean>(PropertyType.ShowTiming, false)
        properties.createProperty<Boolean>(PropertyType.Antialiasing, true)
        properties.createProperty<ColorTheme>(PropertyType.ColorTheme, LightColorTheme())
        properties.createProperty<Any>(PropertyType.Rendering, Rendering.Density)
        properties.createProperty<Any>(PropertyType.InteractionMode, InteractionMode.Filter)
        properties.addPropertiesListener(propertiesListener)
        rubberBand = SimpleRubberBand()
    }
}