/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.interaction

import com.macrofocus.common.filter.MutableFilter
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.common.selection.SingleSelectionEvent
import com.macrofocus.common.selection.SingleSelectionListener
import org.mkui.canvas.MouseEvent
import org.mkui.canvas.MouseListener
import org.mkui.canvas.MouseMotionListener
import org.mkui.geom.Point
import org.mkui.geom.Rectangle2D
import org.mkui.interaction.InteractionMode
import org.mkui.rubberband.RubberBand

/**
 * Created by luc on 27/07/16.
 */
abstract class GenericMouseListener<Row>(rubberBand: RubberBand) : MouseListener, MouseMotionListener {
    private val rubberBand: RubberBand
    private val multipleSelectionEnabled = true
    private val selectOnPopupTrigger = false
    private var isSelectionMode = true
    private var last: Point? = null
    private var selectOnRelease: Row? = null
    private var zoomOnRelease: Point? = null
    private val zoomingPoint: Point? = null
    private var pressedPoint: Point? = null
    protected abstract fun getMode(): InteractionMode?
    protected abstract fun setMode(mode: InteractionMode?)
    fun isSelectionMode(): Boolean {
        return isSelectionMode
    }

    fun setSelectionMode(value: Boolean) {
        isSelectionMode = value
    }

    protected abstract fun getProbing(): MutableSingleSelection<Row>?
    protected abstract fun getSelection(): MutableSelection<Row>?
    protected abstract fun getFilter(): MutableFilter<Row>?
    protected abstract fun getClosestRow(x: Int, y: Int): Row
    protected abstract fun getRows(rect: Rectangle2D?): Collection<Row>
    protected fun getRubberBand(): RubberBand {
        return rubberBand
    }

    override fun mouseClicked(e: MouseEvent) {}
    override fun mousePressed(event: MouseEvent) {
        pressedPoint = Point(event.x, event.y)

//                getView().getNativeComponent().requestFocus();
        val bestrow: Row? = getClosestRow(event.x, event.y)
        when (getMode()) {
            InteractionMode.Selection -> if (bestrow != null) {
                if (!event.isMultipleSelectionKey) {
                    if (!getSelection()!!.isSelected(bestrow) ||
                        getSelection()!!.selectedCount > 1
                    ) {
                        getSelection()!!.clearSelection()
                        getSelection()!!.setSelectedElement(bestrow)
                        setSelectionMode(getSelection()!!.isSelected(bestrow))
                    }
                } else {
                    setSelectionMode(!getSelection()!!.isSelected(bestrow))
                    if (isSelectionMode()) {
                        getSelection()!!.setSelectedState(bestrow, true)
                    } else {
                        getSelection()!!.setSelectedState(bestrow, false)
                    }
                }
            } else {
                getSelection()!!.clearSelection()
            }
            InteractionMode.Toggle -> {
                val selected: Boolean = getSelection()!!.isSelected(bestrow)
                getSelection()!!.setSelectedState(bestrow!!, !selected)
            }
            InteractionMode.Filter -> {
                val filtered: Boolean = getFilter()!!.isFilteredBy(bestrow!!, this)
                getFilter()!!.setFiltered(bestrow!!, !filtered, this)
            }
            InteractionMode.DoNothing -> {}
            else -> {}
        }
    }

    override fun mouseReleased(event: MouseEvent) {
        if (!event.isPopupTrigger) {
            when (getMode()) {
                InteractionMode.Selection -> {
                    if (selectOnRelease != null) {
                        if (event.isLeftMouseButton) {
                            getSelection()!!.clearSelection()
                            if (multipleSelectionEnabled) {
                                addToSelection(selectOnRelease!!)
                            } else {
                                setSelection(selectOnRelease)
                            }
                            selectOnRelease = null
                        }
                    }
                    if (zoomOnRelease != null) {
                        if (event.isLeftMouseButton) {
                            //                            zoom(-1.0 / 20.0, getView().screenToWorldX(zoomOnRelease.x), getView().screenToWorldY(zoomOnRelease.y));
                        } else {
                            //                            zoom(1.0 / 20.0, getView().screenToWorldX(zoomOnRelease.x), getView().screenToWorldY(zoomOnRelease.y));
                        }
                        zoomOnRelease = null
                    }
                    getRubberBand().stopRubberBand()
                }
                InteractionMode.Toggle -> {
                    if (selectOnRelease != null) {
                        if (event.isLeftMouseButton) {
                            getSelection()!!.clearSelection()
                            if (multipleSelectionEnabled) {
                                addToSelection(selectOnRelease!!)
                            } else {
                                setSelection(selectOnRelease)
                            }
                            selectOnRelease = null
                        }
                    }
                    if (zoomOnRelease != null) {
                        if (event.isLeftMouseButton) {
                            //                            zoom(-1.0 / 20.0, getView().screenToWorldX(zoomOnRelease.x), getView().screenToWorldY(zoomOnRelease.y));
                        } else {
                            //                            zoom(1.0 / 20.0, getView().screenToWorldX(zoomOnRelease.x), getView().screenToWorldY(zoomOnRelease.y));
                        }
                        zoomOnRelease = null
                    }
                    getRubberBand().stopRubberBand()
                }
                InteractionMode.Filter -> {
                }
                InteractionMode.DoNothing -> {}
                else -> {}
            }
        } else {
            if (selectOnPopupTrigger) {
                val node = getClosestRow(event.x, event.y)
                setSelection(node)
            }

//                    if (popupMenu != null) {
//                        popupMenu.show(_view, e.x, e.y);
//                    }
        }
        last = null
    }

    override fun mouseEntered(e: MouseEvent) {}
    override fun mouseExited(e: MouseEvent) {
        getProbing()!!.clearSelection()
    }

    override fun mouseDragged(event: MouseEvent) {
        val point = Point(event.x, event.y)
        if (last != null) {
            when (getMode()) {
                InteractionMode.Selection -> if (event.isLeftMouseButton) {
                    if (event.isAltKeyDown || event.isShiftKeyDown) {
                        getProbing()!!.clearSelection()
                        if (multipleSelectionEnabled) {
                            if (!getRubberBand().rubberBand.isActive) {
                                getRubberBand().startRubberBand(pressedPoint!!.ix, pressedPoint!!.iy)
                            }
                            getRubberBand().stretchRubberBand(event.x, event.y)
                            val rect: Rectangle2D? = getRubberBand().rubberBandScreen
                            if (rect != null) {
                                val toSelect = getRows(rect)
                                if (event.isMultipleSelectionKey) {
                                    addToSelection(toSelect)
                                } else {
                                    getSelection()!!.setSelectedIterable(toSelect)
                                }
                                selectOnRelease = null
                            }
                        }
                    } else {
                        val observation = getClosestRow(point.ix, point.iy)
                        getProbing()!!.selected = observation
                        selectOnRelease = null
                        val isAlreadySelected = getSelection() != null && getSelection()!!.isSelected(observation)
                        if (!isAlreadySelected && !event.isMenuShortcutKeyDown) {
                            getSelection()!!.clearSelection()
                        }
                        if (event.isMenuShortcutKeyDown) {
                            if (multipleSelectionEnabled) {
                                addToSelection(observation)
                            } else {
                                setSelection(observation)
                            }
                        } else {
                            if (isAlreadySelected) {
                                selectOnRelease = observation
                            } else {
                                if (multipleSelectionEnabled) {
                                    addToSelection(observation)
                                } else {
                                    setSelection(observation)
                                }
                            }
                        }
                    }
                }
                else -> {}
            }
        }
        last = Point(event.x, event.y)
    }

    override fun mouseMoved(event: MouseEvent) {
        val closest = getClosestRow(event.x, event.y)
        getProbing()!!.selected = closest
    }

    private fun removeFromSelection(observation: Row) {
        getSelection()!!.setSelectedState(observation, false)
    }

    private fun setSelection(observation: Row?) {
        if (observation != null) {
            getSelection()!!.setSelectedElements(observation)
        } else {
            getSelection()!!.clearSelection()
        }
    }

    private fun addToSelection(observation: Row) {
        getSelection()!!.setSelectedState(observation, true)
    }

    private fun addToSelection(newSelection: Collection<Row>) {
        getSelection()!!.setSelectedIterableState(newSelection, true)
    }

    init {
        this.rubberBand = rubberBand
        rubberBand.rubberBand.addSingleSelectionListener(object : SingleSelectionListener<Rectangle2D?> {
            override fun selectionChanged(event: SingleSelectionEvent<Rectangle2D?>) {
                if (event.currentSelection != null && event.currentSelection!!.width > 0 && event.currentSelection!!.height > 0) {
                    val rect: Rectangle2D? = getRubberBand().rubberBandScreen
                    if (rect != null) {
                        val toSelect = getRows(rect)
                        //                        if (event.isMultipleSelectionKey()) {
//                            addToSelection(toSelect);
//                        } else {
                        getSelection()!!.setSelectedIterable(toSelect)
                        //                        }
//                        selectOnRelease = null;
                    }
                } else {
                    getSelection()!!.clearSelection()
                }
            }
        })
    }
}