/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.parallelcoordinates

import com.macrofocus.common.properties.EnumProperties
import com.macrofocus.common.properties.MutableProperties
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.high_d.axis.Alignment
import com.macrofocus.high_d.axis.AxisView
import com.macrofocus.high_d.parallelcoordinates.geometry.Geometry
import org.mkui.canvas.Rendering
import org.mkui.graphics.colortheme.ColorTheme
import org.mkui.graphics.colortheme.LightColorTheme
import org.mkui.rubberband.RubberBand
import org.mkui.rubberband.SimpleRubberBand

abstract class AbstractParallelCoordinatesView<Row, Column>() :
    ParallelCoordinatesView<Row, Column> {
    protected val properties: MutableProperties<ParallelCoordinatesView.PropertyType> = EnumProperties<ParallelCoordinatesView.PropertyType>(
        ParallelCoordinatesView.PropertyType.values())
    private val dragListeners: MutableList<ParallelCoordinatesView.DragListener> = ArrayList<ParallelCoordinatesView.DragListener>()
    private val headerListeners: MutableList<ParallelCoordinatesView.HeaderListener> = ArrayList<ParallelCoordinatesView.HeaderListener>()
    private val rubberBand: RubberBand = SimpleRubberBand()
    override fun getTooltip(): MutableProperty<Boolean> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.Tooltip) as MutableProperty<Boolean>
    }

    override fun setTooltip(tooltip: MutableProperty<Boolean>) {
        properties.replaceProperty<Boolean>(ParallelCoordinatesView.PropertyType.Tooltip, tooltip)
    }

    override fun addDragListener(listener: ParallelCoordinatesView.DragListener) {
        dragListeners.add(listener)
    }

    override fun removeDragListener(listener: ParallelCoordinatesView.DragListener) {
        dragListeners.remove(listener)
    }

    override fun startDragging(view: AxisView<*, *>) {
        for (dragListener in dragListeners) {
            dragListener.startDragging(view)
        }
    }

    override fun stopDragging(view: AxisView<*, *>) {
        for (dragListener in dragListeners) {
            dragListener.stopDragging(view)
        }
    }

    override fun dragAxisTo(view: AxisView<*, *>, shift: Int) {
        for (dragListener in dragListeners) {
            dragListener.dragAxisTo(view, shift)
        }
    }

    override fun addHeaderListener(listener: ParallelCoordinatesView.HeaderListener) {
        headerListeners.add(listener)
    }

    override fun removeHeaderListener(listener: ParallelCoordinatesView.HeaderListener) {
        headerListeners.remove(listener)
    }

    override fun headerSelected(view: AxisView<*, *>, clickCount: Int) {
        for (listener in headerListeners) {
            listener.headerSelected(view, clickCount)
        }
    }

    override fun getShowFiltered(): MutableProperty<Boolean> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.ShowFiltered) as MutableProperty<Boolean>
    }

    override fun getRendering(): MutableProperty<Rendering> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.Rendering) as MutableProperty<Rendering>
    }

    override fun getAntialiasing(): MutableProperty<Boolean> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.Antialiasing) as MutableProperty<Boolean>
    }

    override fun getAxisLine(): MutableProperty<Boolean> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.AxisLine) as MutableProperty<Boolean>
    }

    override fun getGeometry(): MutableProperty<Geometry> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.Geometry) as MutableProperty<Geometry>
    }

    override fun getAlignment(): MutableProperty<Alignment> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.Alignment) as MutableProperty<Alignment>
    }

    override fun getShowVisible(): MutableProperty<Boolean> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.ShowVisible) as MutableProperty<Boolean>
    }

    override fun getColoredTransparency(): MutableProperty<Float?> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.ColoredTransparency) as MutableProperty<Float?>
    }

    override fun getProbeFiltered(): MutableProperty<Boolean> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.ProbeFiltered) as MutableProperty<Boolean>
    }

    override fun getColorTheme(): MutableProperty<ColorTheme> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.ColorTheme) as MutableProperty<ColorTheme>
    }

    override fun setColorTheme(colorTheme: MutableProperty<ColorTheme>) {
        properties.replaceProperty<ColorTheme>(ParallelCoordinatesView.PropertyType.ColorTheme, colorTheme)
    }

    override fun getShowProbedValues(): MutableProperty<Boolean> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.ShowProbedValues) as MutableProperty<Boolean>
    }

    override fun getColumnSize(): MutableProperty<Int?> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.ColumnSize) as MutableProperty<Int?>
    }

    override fun getAllowScrolling(): MutableProperty<Boolean> {
        return properties.getProperty(ParallelCoordinatesView.PropertyType.AllowScrolling) as MutableProperty<Boolean>
    }

    override fun setAllowScrolling(allowScrolling: MutableProperty<Boolean>) {
        properties.replaceProperty<Boolean>(ParallelCoordinatesView.PropertyType.AllowScrolling, allowScrolling)
    }

    /**
     * {@inheritDoc}
     */
    override fun getRubberBand(): RubberBand {
        return rubberBand
    }

    companion object {
        /** Convenience object defining the Default layer. Equivalent to new Integer(0).  */
        const val DEFAULT_LAYER = 0

        /** Convenience object defining the Palette layer. Equivalent to new Integer(100).  */
        const val PALETTE_LAYER = 100

        /** Convenience object defining the Modal layer. Equivalent to new Integer(200).  */
        const val MODAL_LAYER = 200

        /** Convenience object defining the Popup layer. Equivalent to new Integer(300).  */
        const val POPUP_LAYER = 300

        /** Convenience object defining the Drag layer. Equivalent to new Integer(400).  */
        const val DRAG_LAYER = 400
    }

    init {
        properties.createProperty<Boolean>(ParallelCoordinatesView.PropertyType.Tooltip, true)
        properties.createProperty<Boolean>(ParallelCoordinatesView.PropertyType.ShowTiming, false)
        properties.createProperty<Boolean>(ParallelCoordinatesView.PropertyType.ShowFiltered, true)
        properties.createProperty<Boolean>(ParallelCoordinatesView.PropertyType.Antialiasing, true)
        properties.createProperty<Any>(ParallelCoordinatesView.PropertyType.Geometry, Geometry.Polylines)
        properties.createProperty<Boolean>(ParallelCoordinatesView.PropertyType.AxisLine, true)
        properties.createProperty<Any>(ParallelCoordinatesView.PropertyType.Alignment, Alignment.Center)
        properties.createProperty<Boolean>(ParallelCoordinatesView.PropertyType.ShowVisible, true)
        properties.createProperty<Float>(ParallelCoordinatesView.PropertyType.ColoredTransparency, 0.5f)
        properties.createProperty<Boolean>(ParallelCoordinatesView.PropertyType.ProbeFiltered, false)
        properties.createProperty<Any>(ParallelCoordinatesView.PropertyType.ColorTheme, LightColorTheme())
        properties.createProperty<Any>(ParallelCoordinatesView.PropertyType.Rendering, Rendering.Density)
        properties.createProperty<Boolean>(ParallelCoordinatesView.PropertyType.ShowProbedValues, true)
        properties.createProperty<Int>(ParallelCoordinatesView.PropertyType.ColumnSize, 45)
        properties.createProperty<Boolean>(ParallelCoordinatesView.PropertyType.AllowScrolling, false)
    }
}