/*
 * Copyright (c) 2012 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.parallelcoordinates

import com.macrofocus.colormapping.implementation.SimpleColorMapping
import com.macrofocus.common.collection.WeakHashMap
import com.macrofocus.common.filter.MutableFilter
import com.macrofocus.common.math.isNaN
import com.macrofocus.common.properties.MutableProperties
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.properties.Property
import com.macrofocus.common.properties.SimpleProperties
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.high_d.axis.AxisFactory
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import com.macrofocus.high_d.axis.hierarchy.AxisHierarchy
import com.macrofocus.high_d.axis.hierarchy.DefaultAxisHierarchy
import com.macrofocus.high_d.axis.locations.AxisLocations
import com.macrofocus.high_d.axis.locations.DefaultAxisLocations
import com.macrofocus.order.MutableVisibleOrder
import com.macrofocus.visual.SimpleVisual
import com.macrofocus.visual.SimpleVisualObjects
import org.mkui.coloring.MutableColoring
import org.mkui.colormap.ColorMapFactory
import org.mkui.colormapping.MutableColorMapping
import org.mkui.visual.Visual
import org.molap.dataframe.ColumnStatistics
import org.molap.dataframe.DataFrame

/**
 * Default implementation of a ParallelCoordinatesModel. This class provides an easy way to integration with Swing's
 * TableModel.
 */
class DefaultParallelCoordinatesModel<Row, Column>(
    dataFrame: DataFrame<Row, Column, *>,
    visual: Visual<Row, Column>,
    axisHierarchy: AxisHierarchy<Row, Column>,
    settings: ParallelCoordinatesSettings
) : AbstractParallelCoordinatesModel<Row, Column>() {
    private var dataFrame: DataFrame<Row, Column, *>? = null
    private val visual: Visual<Row, Column>
    private val settings: ParallelCoordinatesSettings
    private val axisHierarchy: AxisHierarchy<Row, Column>
    private val axisLocationsMap: MutableMap<AxisGroupModel<Row, Column>, AxisLocations<Row, Column>> = WeakHashMap<AxisGroupModel<Row, Column>, AxisLocations<Row, Column>>()
    private val properties: MutableProperties<String?> = SimpleProperties()
    private val maxDimensions: Property<Int> = properties.createProperty("maxDimensions", Int.MAX_VALUE)

    constructor(colorMapFactory: ColorMapFactory, dataFrame: DataFrame<Row, Column, *>) : this(
        dataFrame,
        SimpleVisual<Row, Column>(SimpleVisualObjects<Row>(dataFrame), SimpleColorMapping<Row, Column>(colorMapFactory, dataFrame))
    ) {
    }

    constructor(dataFrame: DataFrame<Row, Column, *>, visual: Visual<Row, Column>) : this(
        dataFrame, visual, DefaultAxisHierarchy<Row, Column>(
            AxisFactory<Row, Column>(dataFrame, null, null, visual.probing, visual.selection, visual.filter),
            dataFrame, true, false, 10
        ), DefaultParallelCoordinatesSettings()
    ) {
    }

    fun setDataFrame(dataFrame: DataFrame<Row, Column, *>) {
        this.dataFrame = dataFrame
    }

    override val objectCount: Int
        get() = dataFrame!!.rowCount

    override fun getVisual(): Visual<Row, Column> {
        return visual
    }

    /**
     * {@inheritDoc}
     */
    override fun getProbing(): MutableSingleSelection<Row> {
        return visual.probing
    }

    /**
     * {@inheritDoc}
     */
    override fun getSelection(): MutableSelection<Row> {
        return visual.selection
    }

    /**
     * {@inheritDoc}
     */
    override fun getFilter(): MutableFilter<Row> {
        return visual.filter
    }

    /**
     * {@inheritDoc}
     */
    override fun getColorMapping(): MutableColorMapping<Row, Column> {
        return visual.colorMapping
    }

    /**
     * {@inheritDoc}
     */
    override fun getColoring(): MutableColoring<Row> {
        return visual.coloring
    }

    /**
     * {@inheritDoc}
     */
    override fun getObject(index: Int): Row {
        return dataFrame!!.getRowKey(index)
    }

    /**
     * {@inheritDoc}
     */
    override fun getSettings(): ParallelCoordinatesSettings {
        return settings
    }

    /**
     * {@inheritDoc}
     */
    override fun getAxisHierarchy(): AxisHierarchy<Row, Column> {
        return axisHierarchy
    }

    /**
     * {@inheritDoc}
     */
    override fun getSelectedAxis(): Iterable<AxisModel<Row, Column>> {
        val list: MutableList<AxisModel<Row, Column>> = ArrayList<AxisModel<Row, Column>>()
        for (c in dataFrame!!.columns()) {
            val axisModel: AxisModel<Row, Column> = getAxisHierarchy().getAxisModel(c)!!
            val buttonModel: MutableProperty<Boolean> = axisModel.buttonModel
            if (buttonModel.value) {
                list.add(axisModel)
            }
        }
        return list
    }

    /**
     * {@inheritDoc}
     */
    override fun isHidden(axisModel: AxisModel<Row, Column>): Boolean {
        return !axisHierarchy.getAxisGroup(axisModel)!!.axisOrder!!.isVisible(axisModel)
    }

    /**
     * {@inheritDoc}
     */
    override fun setHidden(axisModel: AxisModel<Row, Column>, hidden: Boolean) {
        axisHierarchy.getAxisGroup(axisModel)!!.axisOrder!!.setVisible(axisModel, !hidden)
        fireParallelCoordinatesChanged()
    }

    /**
     * {@inheritDoc}
     */
    override fun unhideAllAxis() {
        for (axisGroup in axisHierarchy.axisGroupHierarchy.breadthFirstIterator()) {
            for (axisModel in axisGroup.axisModels) {
                setHidden(axisModel, false)
            }
        }
        fireParallelCoordinatesChanged()
    }

    override fun getAxisLocations(group: AxisGroupModel<Row, Column>): AxisLocations<Row, Column>? {
        if (!axisLocationsMap.containsKey(group)) {
            val value = DefaultAxisLocations(group.axisOrder!!, maxDimensions)
            axisLocationsMap[group] = value
            return value
        }
        return axisLocationsMap[group]
    }

    override fun showAllNumerical() {
        val axisOrder: MutableVisibleOrder<AxisModel<Row, Column>> = getAxisHierarchy().axisGroupHierarchy.root.axisOrder!!
        for (axisModel in axisOrder.iterableAll()) {
            if (axisModel.isNumerical && axisModel.name != null) {
                axisOrder.show(axisModel)
            }
        }
    }

    override fun makeDataRange(axisModels: Iterable<AxisModel<Row, Column>>) {
        for (axisModel in axisModels) {
            val statistics: ColumnStatistics = axisModel.columnStatistics
            if (statistics.min != null && statistics.max != null) {
                axisModel.setMinMax(statistics.min!!.toDouble(), statistics.max!!.toDouble())
            } else {
                axisModel.setMinMax(Double.NaN, Double.NaN)
            }
        }
    }

    override fun makeCommonRange(axisModels: Iterable<AxisModel<Row, Column>>) {
        var sharedMin: Number? = null
        var sharedMax: Number? = null
        for (axisModel in axisModels) {
            val min: Number = axisModel.minimum
            val max: Number = axisModel.maximum
            if (min != null && max != null) {
                val minimum = min.toDouble()
                val maximum = max.toDouble()
                if (!isNaN(minimum) && !isNaN(maximum)) {
                    if (sharedMin == null || sharedMin.toDouble() > minimum) {
                        sharedMin = min
                    }
                    if (sharedMax == null || sharedMax.toDouble() < maximum) {
                        sharedMax = max
                    }
                }
            }
        }
        if (sharedMin != null && sharedMax != null) {
            for (axisModel in axisModels) {
                axisModel.setMinMax(sharedMin.toDouble(), sharedMax.toDouble())
                axisModel.interval.setValue(sharedMin.toDouble(), sharedMax.toDouble() - sharedMin.toDouble())
            }
        }
    }

    override fun resetRanges(axisModels: Iterable<AxisModel<Row, Column>>) {
        for (axisModel in axisModels) {
            axisModel.interval.setValue(axisModel.minimum, axisModel.maximum - axisModel.minimum)
            //            axisModel.getRangeSliderModel().reset(this);
        }
    }

    init {
        this.dataFrame = dataFrame
        this.visual = visual
        this.axisHierarchy = axisHierarchy
        this.settings = settings
    }
}