/*
 * Copyright (c) 2012 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.parallelcoordinates

import com.macrofocus.common.filter.MutableFilter
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import com.macrofocus.high_d.axis.hierarchy.AxisHierarchy
import com.macrofocus.high_d.axis.locations.AxisLocations
import org.mkui.coloring.MutableColoring
import org.mkui.colormapping.MutableColorMapping
import org.mkui.visual.Visual
import org.mkui.visual.VisualObjects
import kotlin.js.JsExport

/**
 * Main interface for the model part of ParallelCoordinates.
 */
@JsExport
interface ParallelCoordinatesModel<O, C> : VisualObjects<O> {
    fun getVisual(): Visual<O, C>

    /**
     * Obtains the model used for probing objects in the parallel coordinates (aka mouse over).
     *
     * @return the model used for probing.
     */
    fun getProbing(): MutableSingleSelection<O>

    /**
     * Obtains the model used for selecting objects in the parallel coordinates.
     *
     * @return the model used for selection.
     */
    fun getSelection(): MutableSelection<O>

    /**
     * Obtains the model used for filtering objects in the parallel coordinates.
     *
     * @return the model used for filtering.
     */
    fun getFilter(): MutableFilter<O>

    /**
     * Gets model holding the color mapping of each object.
     *
     * @return the model holding the color of each object.
     */
    fun getColorMapping(): MutableColorMapping<O, C>

    /**
     * Gets model holding the colors of each object.
     *
     * @return the model holding the color of each object.
     */
    fun getColoring(): MutableColoring<O>

    /**
     * Returns the number of data objects (typically rows).
     *
     * @return the number of data objects
     */
    override val objectCount: Int

    /**
     * @param index
     *
     * @return
     */
    override fun getObject(index: Int): O

    /**
     * Gets the currently selected settings.
     *
     * @return the settings.
     */
    fun getSettings(): com.macrofocus.high_d.parallelcoordinates.ParallelCoordinatesSettings
    fun getAxisHierarchy(): AxisHierarchy<O, C>

    /**
     * Add a listener for receiving interesting events about changes in the model.
     *
     * @param listener the listener to add.
     */
    fun addParallelCoordinatesListener(listener: com.macrofocus.high_d.parallelcoordinates.ParallelCoordinatesListener)

    /**
     * Remove a listener from the list of listeners that should be notified about changes in the model.
     *
     * @param listener the listener to remove.
     */
    fun removeParallelCoordinateListener(listener: com.macrofocus.high_d.parallelcoordinates.ParallelCoordinatesListener)

    /**
     * Returns the list of currently selected axis.
     *
     * @return the axis that are currently selected
     */
    fun getSelectedAxis(): Iterable<AxisModel<O, C>>

    /**
     * Indicates whether an axis is visible or not.
     *
     * @param axisModel the axis
     *
     * @return false if it is visible; true otherwise
     */
    fun isHidden(axisModel: AxisModel<O, C>): Boolean

    /**
     * Turns on or off the visibility of an axis.
     *
     * @param axisModel the axis
     * @param hidden    false if it should be visible; true otherwise
     */
    fun setHidden(axisModel: AxisModel<O, C>, hidden: Boolean)

    /**
     * Make all the axis visible.
     */
    fun unhideAllAxis()
    fun getAxisLocations(group: AxisGroupModel<O, C>): AxisLocations<O, C>?
    fun showAllNumerical()
    fun makeCommonRange(axisModels: Iterable<AxisModel<O, C>>)
    fun makeDataRange(axisModels: Iterable<AxisModel<O, C>>)
    fun resetRanges(axisModels: Iterable<AxisModel<O, C>>)
}