/*
 * Copyright (c) 2009 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.parallelcoordinates

import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.high_d.axis.Alignment
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.AxisView
import com.macrofocus.high_d.axis.group.AxisGroupModel
import com.macrofocus.high_d.axis.group.AxisGroupView
import com.macrofocus.high_d.parallelcoordinates.geometry.Geometry
import com.macrofocus.high_d.parallelcoordinates.layout.ParallelCoordinatesLayout
import com.macrofocus.license.LicenseModel
import org.mkui.canvas.Rendering
import org.mkui.component.CPInteractiveComponentProvider
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.colortheme.ColorTheme
import org.mkui.rubberband.RubberBand

/**
 * Main class for the view part of ParallelCoordinates.
 */
interface ParallelCoordinatesView<Row, Column> : CPInteractiveComponentProvider {
    fun getColumnSize(): MutableProperty<Int?>
    enum class PropertyType {
        ShowTiming, ShowFiltered, Antialiasing, Geometry, AxisLine, Alignment, ShowVisible, ColoredTransparency, Tooltip, Rendering, ProbeFiltered, ColorTheme, ShowProbedValues, ColumnSize, AllowScrolling
    }

    fun getClosestRow(x: Int, y: Int): Row?

    /**
     * Returns the nodes overlapping the specified rectangle.
     *
     * @param rect the rectangle in screen coordinates
     *
     * @return the list of nodes
     */
    fun getRows(rect: Rectangle2D?): Collection<Row>?
    fun getRubberBand(): RubberBand
    fun isSelectionMode(): Boolean
    fun setSelectionMode(value: Boolean)
    fun getWidth(): Int
    fun getHeight(): Int
    fun getAlignment(): MutableProperty<Alignment>
    fun getShowVisible(): MutableProperty<Boolean>
    fun getColoredTransparency(): MutableProperty<Float?>
    fun getProbeFiltered(): MutableProperty<Boolean>
    fun getAllowScrolling(): MutableProperty<Boolean>

    /**
     * Returns whether information about refresh rate should be displayed.
     *
     * @return true to display refresh rate information, false otherwise
     */
    fun isShowTiming(): Boolean

    /**
     * Sets whether information about refresh rate should be displayed.
     *
     * @param showTiming true to display refresh rate information, false otherwise
     */
    fun setShowTiming(showTiming: Boolean)
    fun getAxisGroupView(axisGroup: AxisGroupModel<Row, Column>?): AxisGroupView?

    /**
     * Gets the view associated with a given axis.
     *
     * @param axisModel the axis
     *
     * @return
     */
    fun getAxisView(axisModel: AxisModel<*, *>?): AxisView<*, *>?
    fun setLicenseModel(licenseModel: LicenseModel?)

    /**
     * Returns the ParallelCoordinatesModel that provides the data displayed by this ParallelCoordinatesView.
     *
     * @return the ParallelCoordinatesModel that provides the data displayed by this ParallelCoordinatesView
     *
     * @see .setModel
     */
    var model: ParallelCoordinatesModel<Row, Column>?

    /**
     * Start dragging an axis.
     *
     * @param view the axis to drag
     */
    fun startDragging(view: AxisView<*, *>)

    /**
     * Stop dragging an axis. This will usually trigger a layout of the parallel coordinates plot.
     *
     * @param view the axis that has been dragged
     */
    fun stopDragging(view: AxisView<*, *>)

    /**
     * Drag an axis with the specified offset.
     *
     * @param view  the axis
     * @param shift the offset
     */
    fun dragAxisTo(view: AxisView<*, *>, shift: Int)
    fun headerSelected(view: AxisView<*, *>, clickCount: Int)

    /**
     * Returns the color theme to use.
     *
     * @return the color theme
     */
    fun getColorTheme(): MutableProperty<ColorTheme>

    /**
     * Sets the color theme to use.
     *
     * @param colorTheme the color theme to use
     */
    fun setColorTheme(colorTheme: MutableProperty<ColorTheme>)

    /**
     * Wait until the parallel coordinates plot has been fully rendered.
     */
    fun waitUntilReady()
    fun getHeaderAxisGroupMaximumHeight(): Int
    fun getHeaderAxisMaximumHeight(): Int

    /**
     * Returns the class responsible for laying out the components of the parallel coordinates plot.
     *
     * @return the layout engine
     */
    fun getParallelCoordinatesLayout(): ParallelCoordinatesLayout<Row, Column>

    /**
     * Gets the position along the X axis of an axis.
     *
     * @param axisModel the axis
     *
     * @return its position along the X axis
     */
    fun getAxisX(axisGroup: AxisGroupModel<Row,Column>, axisModel: AxisModel<Row,Column>): Int
    fun getShowFiltered(): MutableProperty<Boolean>
    fun setShowFiltered(showFiltered: MutableProperty<Boolean>?)
    fun getRendering(): MutableProperty<Rendering>
    fun setRendering(rendering: MutableProperty<Rendering>?)
    fun getAntialiasing(): MutableProperty<Boolean>
    fun setAntialiasing(antialiasing: MutableProperty<Boolean>?)
    fun getAxisLine(): MutableProperty<Boolean>
    fun getGeometry(): MutableProperty<Geometry>
    fun setGeometry(geometry: MutableProperty<Geometry>)
    fun getTooltip(): MutableProperty<Boolean>
    fun setTooltip(tooltip: MutableProperty<Boolean>)
    fun getShowProbedValues(): MutableProperty<Boolean>
    fun setAllowScrolling(allowScrolling: MutableProperty<Boolean>)
    fun addDragListener(listener: DragListener)
    fun removeDragListener(listener: DragListener)
    interface DragListener {
        /**
         * Start dragging an axis.
         *
         * @param view the axis to drag
         */
        fun startDragging(view: AxisView<*, *>)

        /**
         * Stop dragging an axis. This will usually trigger a layout of the parallel coordinates plot.
         *
         * @param view the axis that has been dragged
         */
        fun stopDragging(view: AxisView<*, *>)

        /**
         * Drag an axis with the specified offset.
         *
         * @param view  the axis
         * @param shift the offset
         */
        fun dragAxisTo(view: AxisView<*, *>, shift: Int)
    }

    fun addHeaderListener(listener: HeaderListener)
    fun removeHeaderListener(listener: HeaderListener)
    interface HeaderListener {
        fun headerSelected(view: AxisView<*, *>, clickCount: Int)
    }
}