/*
 * Copyright (c) 2013 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.parallelcoordinates.geometry

import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.parallelcoordinates.ParallelCoordinatesModel
import com.macrofocus.high_d.parallelcoordinates.ParallelCoordinatesView
import com.macrofocus.high_d.parallelcoordinates.layout.ParallelCoordinatesLayout
import org.mkui.geom.GeneralPath
import org.mkui.geom.Rectangle2D
import org.mkui.geom.Shape
import org.mkui.geom.curve.CatmullRomSpline
import org.mkui.geom.curve.ControlPath
import org.mkui.geom.curve.GroupIterator
import org.mkui.geom.curve.ShapeMultiPath

enum class Geometry {
    Polylines {
        override fun <R,C> createGeometry(
            view: ParallelCoordinatesView<R, C>,
            model: ParallelCoordinatesModel<R, C>?,
            fullHeight: Int,
            layer: Int,
            row: R,
            level: Int
        ): Shape? {
            var path: GeneralPath? = null
            var previousAvailable = false
            val layout: ParallelCoordinatesLayout<R,C> = view.getParallelCoordinatesLayout()
            for (axisGroup in layout.getAxisGroups(level)) {
                for (axisModel in axisGroup.axisOrder!!) {
                    val trackBounds: Rectangle2D = layout.getTrackBounds(axisGroup, axisModel)
                    val x: Int = trackBounds.x.toInt() + layout.getAxisPreferredWidth() / 2
                    val value: Number? = axisModel.getValue(layer, row)
                    val currentAvailable = value != null
                    if (currentAvailable) {
                        val height = trackBounds.height.toInt()
                        val y = getY(fullHeight, height, layout.getAxisAfterTrackGap(), value!!.toDouble(), axisModel)
                        if (path != null) {
                            if (previousAvailable) {
                                path.lineTo(x.toDouble(), y)
                            } else {
                                path.moveTo((x - 1).toDouble(), y)
                                path.lineTo(x.toDouble(), y)
                            }
                        } else {
                            path = GeneralPath()
                            path.moveTo(x.toDouble(), y)
                        }
                    }
                    previousAvailable = currentAvailable
                }
            }
            return path
        }
    },
    Steps {
        override fun <R,C> createGeometry(
            view: ParallelCoordinatesView<R, C>,
            model: ParallelCoordinatesModel<R, C>?,
            fullHeight: Int,
            layer: Int,
            row: R,
            level: Int
        ): Shape? {
            var path: GeneralPath? = null
            var previousAvailable = false
            val layout: ParallelCoordinatesLayout<R,C> = view.getParallelCoordinatesLayout()
            for (axisGroup in layout.getAxisGroups(level)) for (axisModel in axisGroup.axisOrder!!) {
                val trackBounds: Rectangle2D = layout.getTrackBounds(axisGroup, axisModel)
                val x: Int = view.getAxisX(axisGroup, axisModel)
                val value: Number? = axisModel.getValue(layer, row)
                val currentAvailable = value != null
                if (currentAvailable) {
                    val height = trackBounds.height.toInt()
                    val y = getY(fullHeight, height, layout.getAxisAfterTrackGap(), value!!.toDouble(), axisModel)
                    if (path != null) {
                        if (previousAvailable) {
                            path.lineTo(x.toDouble(), y)
                        } else {
                            path.moveTo(x.toDouble(), y)
                        }
                        path.lineTo(x.toDouble() + layout.getAxisPreferredWidth(), y)
                    } else {
                        path = GeneralPath()
                        path.moveTo(x.toDouble(), y)
                        path.lineTo(x.toDouble() + layout.getAxisPreferredWidth(), y)
                    }
                }
                previousAvailable = currentAvailable
            }
            return path
        }
    },
    Polycurves {
        override fun <R,C> createGeometry(
            view: ParallelCoordinatesView<R, C>,
            model: ParallelCoordinatesModel<R, C>?,
            fullHeight: Int,
            layer: Int,
            row: R,
            level: Int
        ): Shape? {
            var x = 0.0
            var y = 0.0
            val curves = ControlPath()
            var cp: ControlPath? = null
            var previousAvailable = false
            val layout: ParallelCoordinatesLayout<R,C> = view.getParallelCoordinatesLayout()
            for (axisGroup in layout.getAxisGroups(level)) for (axisModel in axisGroup.axisOrder!!) {
                val trackBounds: Rectangle2D = layout.getTrackBounds(axisGroup, axisModel)
                x = view.getAxisX(axisGroup, axisModel) + layout.getAxisPreferredWidth() / 2.0
                val value: Number? = axisModel.getValue(layer, row)
                val currentAvailable = value != null
                if (currentAvailable) {
                    val height = trackBounds.height.toInt()
                    y = getY(fullHeight, height, layout.getAxisAfterTrackGap(), value!!.toDouble(), axisModel)
                    if (cp != null) {
                        if (previousAvailable) {
                            cp.addPoint(createPoint2(x, y))
                        } else {
                            println("Error")
                        }
                    } else {
                        cp = ControlPath()
                        cp.addPoint(createPoint2(x, y))
                        cp.addPoint(createPoint2(x, y))
                    }
                    previousAvailable = currentAvailable
                } else {
                    if (cp != null) {
                        cp.addPoint(createPoint2(x, y))
                        val cardinalSpline = CatmullRomSpline(cp, GroupIterator(intArrayOf(0, cp.numPoints() - 1)))
                        curves.addCurve(cardinalSpline)
                    }
                    cp = null
                }
            }
            if (cp != null) {
                cp.addPoint(createPoint2(x, y))
                val cardinalSpline = CatmullRomSpline(cp, GroupIterator(intArrayOf(0, cp.numPoints() - 1)))
                //            cardinalSpline.setAlpha(0.5);
                curves.addCurve(cardinalSpline)
            }
            val dimension = 2
            val mp = ShapeMultiPath(dimension)
            mp.flatness = 1.0 // default flatness is 1.0
            for (i in 0 until curves.numCurves()) {
                val curve: org.mkui.geom.curve.Curve = curves.getCurve(i)
                curve.appendTo(mp)
            }
            return mp
        }
    };

    abstract fun <R,C> createGeometry(
        view: ParallelCoordinatesView<R, C>,
        model: ParallelCoordinatesModel<R, C>?,
        fullHeight: Int,
        layer: Int,
        row: R,
        level: Int
    ): Shape?

    companion object {
        fun getY(fullHeight: Int, height: Int, offset: Int, value: Double, axisModel: AxisModel<*,*>): Double {
            return if (!axisModel.isDegenerate) {
                fullHeight - 1 - (height - 1) * axisModel.getNormalizedValue(value)!! - offset
            } else {
                fullHeight - height * 0.5 - offset
            }
        }

        private fun createPoint2(x: Double, y: Double): org.mkui.geom.curve.Point {
            val arr = doubleArrayOf(x, y)
            return object : org.mkui.geom.curve.Point {
                override val location: DoubleArray
                    get() = arr
            }
        }
    }
}