/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.parallelcoordinatesmatrix

import com.macrofocus.common.properties.MutableProperties
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.properties.Property
import com.macrofocus.common.properties.SimpleProperties
import com.macrofocus.hierarchy.Hierarchy
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import com.macrofocus.high_d.axis.hierarchy.AxisHierarchy
import com.macrofocus.high_d.axis.hierarchy.DefaultAxisHierarchy
import com.macrofocus.high_d.parallelcoordinates.DefaultParallelCoordinatesModel
import com.macrofocus.high_d.parallelcoordinates.DefaultParallelCoordinatesSettings
import com.macrofocus.high_d.parallelcoordinates.ParallelCoordinates
import com.macrofocus.license.LicenseModel
import com.macrofocus.order.MutableVisibleOrder
import com.macrofocus.order.OrderEvent
import com.macrofocus.order.OrderListener
import org.mkui.canvas.Rendering
import org.mkui.graphics.colortheme.ColorTheme
import kotlin.math.min
import kotlin.math.pow

abstract class AbstractParallelCoordinatesMatrixView<Row, Column> : ParallelCoordinatesMatrixView<Row,Column> {
    override var model: ParallelCoordinatesMatrixModel<Row, Column>? = null
    set(value) {
        field?.getAxisGroupModel()!!.axisOrder!!.removeOrderListener(orderListener)
        field = value
        if (field != null) {
            updateParallelCoordinatesModel()
            field!!.getAxisGroupModel().axisOrder!!.addOrderListener(orderListener)
        }

    }
    protected var parallelCoordinates: ParallelCoordinates<Row, Column>
    private val properties: MutableProperties<String?> = SimpleProperties()
    private val maxDimensions: Property<Int> = properties.createProperty("maxDimensions", 20)
    private val licenseModel: LicenseModel? = null
    private val orderListener: OrderListener<AxisModel<Row,Column>> = object : OrderListener<AxisModel<Row,Column>> {
        override fun orderChanged(event: OrderEvent<AxisModel<Row,Column>>?) {
            refresh()
        }

        override fun orderVisibility(event: OrderEvent<AxisModel<Row,Column>>) {
            refresh()
        }

        override fun orderAdded(event: OrderEvent<AxisModel<Row,Column>>) {
            refresh()
        }

        override fun orderRemoved(event: OrderEvent<AxisModel<Row,Column>>) {
            refresh()
        }
    }

    protected abstract fun createParallelCoordinates(): ParallelCoordinates<Row, Column>
    
    protected abstract fun refresh()
    protected open fun updateParallelCoordinatesModel() {
        val axisHierarchy: AxisHierarchy<Row, Column> =
            DefaultAxisHierarchy<Row, Column>(model!!.getAxisHierarchy().axisFactory, model!!.getDataFrame(), true, 10)
        val matrixModel = DefaultParallelCoordinatesModel(model!!.getDataFrame(), model!!.getVisual(), axisHierarchy, DefaultParallelCoordinatesSettings())
        val axisOrder: MutableVisibleOrder<AxisModel<Row, Column>> = model!!.getAxisGroupModel().axisOrder!!
        val size: Int = min(maxDimensions.value, axisOrder.size())
        val h = hamiltonianDecompositions(size)
        val axisGroupHierarchy: Hierarchy<AxisGroupModel<Row, Column>> = matrixModel.getAxisHierarchy().axisGroupHierarchy
        var root: AxisGroupModel<Row, Column> = axisGroupHierarchy.root
        val axes: MutableList<AxisModel<Row, Column>> = ArrayList<AxisModel<Row, Column>>()
        for (axisModel in root.axisOrder!!) {
            axes.add(axisModel)
        }
        for (axisModel in axes) {
            matrixModel.getAxisHierarchy().removeFromGroup(root, axisModel)
        }
        for (i in h.indices) {
            val c = h[i]
            val axisModels: Array<AxisModel<Row, Column>?> = arrayOfNulls<AxisModel<Row, Column>>(c.size)
            for (j in c.indices) {
                val index = c[j]
                axisModels[j] = matrixModel.getAxisHierarchy().createClonedAxisModel(axisOrder.get(index).column)
            }
            root = matrixModel.getAxisHierarchy().createGroup(root, "Group " + (i + 1), *axisModels.requireNoNulls())!!
        }
        parallelCoordinates.model = matrixModel

//        nativeComponent.revalidate();
    }

    override fun isShowTiming(): Boolean {
        return parallelCoordinates.view!!.isShowTiming()
    }

    override fun setShowTiming(showTiming: Boolean) {
        parallelCoordinates.view!!.setShowTiming(showTiming)
    }

    override fun isSelectionMode(): Boolean {
        return parallelCoordinates.view!!.isSelectionMode()
    }

    override fun setSelectionMode(value: Boolean) {
        parallelCoordinates.view!!.setSelectionMode(value)
    }

    override fun setLicenseModel(licenseModel: LicenseModel?) {
        parallelCoordinates.view!!.setLicenseModel(licenseModel)
    }

    override fun getColorTheme(): MutableProperty<ColorTheme> {
        return parallelCoordinates.view!!.getColorTheme()
    }

    override fun setColorTheme(colorTheme: MutableProperty<ColorTheme>) {
        parallelCoordinates.view!!.setColorTheme(colorTheme)
    }

    override fun setShowFiltered(showFiltered: MutableProperty<Boolean>?) {
        parallelCoordinates.view!!.setShowFiltered(showFiltered)
    }

    override fun setRendering(rendering: MutableProperty<Rendering>?) {
        parallelCoordinates.view!!.setRendering(rendering)
    }

    override fun setAntialiasing(antialiasing: MutableProperty<Boolean>?) {
        parallelCoordinates.view!!.setAntialiasing(antialiasing)
    }

    companion object {
        /**
         * Hamiltonian decompositions
         *
         * @param n Number of axes
         *
         * @return
         */
        private fun hamiltonianDecompositions(n: Int): Array<IntArray> {
            // Number of Hamiltonian paths
            val m = getRowCount(n)

//        System.out.println("m=" + m + ", " + " n=" + n);
            val h: Array<IntArray>
            h = if (n == 2 * m) {
                Array(m) { IntArray(n) }
            } else if (n == 2 * m + 1) {
                Array(m) { IntArray(2 * m + 2) }
            } else {
                Array(m) { IntArray(2 * m + 2) }
            }
            val even = n % 2 == 0
            for (k in 0 until m) {
                for (j in 0 until 2 * m) {
                    var v = h(k + 1, j + 1, m, 2 * m)
                    if (v < 0) {
                        v = 2 * m + v
                    }
                    if (even) {
                        h[k][j] = v
                    } else {
                        if (j == 0) {
                            h[k][j] = n - 1
                            h[k][j + 1] = v
                            h[k][2 * m + 1] = n - 1
                        } else {
                            h[k][j + 1] = v
                        }
                    }
                }
            }

//        for(int k = 0; k < h.length; k++) {
//            int hk[] = h[k];
//            for(int j = 0; j < hk.length; j++) {
//                System.out.print(h[k][j] + " ");
//            }
//            System.out.println();
//        }
            return h
        }

        protected fun getRowCount(n: Int): Int {
            val m: Int
            val even = n % 2 == 0
            m = if (even) {
                // Even
                n / 2
            } else {
                // Odd
                (n - 1) / 2
            }
            return m
        }

        private fun h(k: Int, j: Int, m: Int, n: Int): Int {
            return if (k == 1 && j == 1) {
                0
            } else if (k == 1) {
                h(1, j - 1, m, n) + -1.0.pow(j.toDouble()).toInt() * (j - 1) % n
            } else {
                h(k - 1, j, m, n) + 1 % n
            }
        }
    }

    init {
        parallelCoordinates = createParallelCoordinates()
        setLicenseModel(licenseModel)
    }
}