package com.macrofocus.high_d.scatterplot

import com.macrofocus.colormapping.implementation.SimpleColorMapping
import com.macrofocus.common.filter.Filter
import com.macrofocus.common.selection.*
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import com.macrofocus.high_d.axis.group.DefaultAxisGroupModel
import com.macrofocus.visual.SimpleVisual
import com.macrofocus.visual.SimpleVisualObjects
import org.mkui.coloring.MutableColoring
import org.mkui.colormap.ColorMapFactory
import org.mkui.colormapping.MutableColorMapping
import org.mkui.visual.Visual
import org.mkui.visual.VisualObjects
import org.molap.dataframe.DataFrame

class DefaultScatterPlotModel<Row, C>(
    dataFrame: DataFrame<Row, C, *>,
    annotationDataFrame: DataFrame<*, C, *>?,
    visual: Visual<Row, C>,
    root: AxisGroupModel<Row, C>
) : AbstractScatterPlotModel<Row, C>(), VisualObjects<Row> {
    private val dataFrame: DataFrame<Row, C, *>
    private val annotationDataFrame: DataFrame<*, C, *>?
    private val visual: Visual<Row, C>
    private val x: MutableSingleSelection<AxisModel<Row, C>> = SimpleSingleSelection<AxisModel<Row, C>>()
    private val y: MutableSingleSelection<AxisModel<Row, C>> = SimpleSingleSelection<AxisModel<Row, C>>()
    private val y2: MutableSingleSelection<AxisModel<Row, C>> = SimpleSingleSelection<AxisModel<Row, C>>()
    private val size: MutableSingleSelection<AxisModel<Row, C>> = SimpleSingleSelection<AxisModel<Row, C>>()
    private val root: AxisGroupModel<Row, C>
    private var xAxisModel: AxisModel<Row, C>? = null
    private var yAxisModel: AxisModel<Row, C>? = null
    private var y2AxisModel: AxisModel<Row, C>? = null
    private var sizeAxisModel: AxisModel<Row, C>? = null

    constructor(colorMapFactory: ColorMapFactory, dataFrame: DataFrame<Row, C, *>) : this(
        colorMapFactory,
        dataFrame,
        SimpleVisual<Row, C>(SimpleVisualObjects(dataFrame), SimpleColorMapping(colorMapFactory, dataFrame))
    ) {
    }

    constructor(
        colorMapFactory: ColorMapFactory,
        dataFrame: DataFrame<Row, C, *>,
        visual: Visual<Row, C>
    ) : this(
        dataFrame,
        null,
        visual,
        DefaultAxisGroupModel("Root", dataFrame, null, visual.probing, visual.selection, visual.filter)
    ) {
    }

    override val objectCount: Int
        get() = dataFrame.rowCount

    override fun getVisual(): Visual<Row, C> {
        return visual
    }

    override fun getAxisGroupModel(): AxisGroupModel<Row, C> {
        return root
    }

    /**
     * {@inheritDoc}
     */
    override fun getProbing(): MutableSingleSelection<Row> {
        return visual.probing
    }

    /**
     * {@inheritDoc}
     */
    override fun getSelection(): MutableSelection<Row> {
        return visual.selection
    }

    /**
     * {@inheritDoc}
     */
    override fun getFilter(): Filter<Row> {
        return visual.filter
    }

    /**
     * {@inheritDoc}
     */
    override fun getColorMapping(): MutableColorMapping<Row, C> {
        return visual.colorMapping
    }

    /**
     * {@inheritDoc}
     */
    override fun getColoring(): MutableColoring<Row> {
        return visual.coloring
    }

    /**
     * {@inheritDoc}
     */
    override fun getObject(index: Int): Row {
        return dataFrame.getRowKey(index)
    }

    override fun getX(): MutableSingleSelection<AxisModel<Row, C>> {
        return x
    }

    override fun getY(): MutableSingleSelection<AxisModel<Row, C>> {
        return y
    }

    override fun getY2(): MutableSingleSelection<AxisModel<Row, C>> {
        return y2
    }

    override fun getSize(): MutableSingleSelection<AxisModel<Row, C>> {
        return size
    }

    override fun getDataFrame(): DataFrame<Row, C, *> {
        return dataFrame
    }

    override fun getAnnotationDataFrame(): DataFrame<*, C, *>? {
        return annotationDataFrame
    }

    override fun getXAxisModel(): AxisModel<Row, C>? {
        return xAxisModel
    }

    override fun getYAxisModel(): AxisModel<Row, C>? {
        return yAxisModel
    }

    override fun getY2AxisModel(): AxisModel<Row, C>? {
        return y2AxisModel
    }

    override fun getSizeAxisModel(): AxisModel<Row, C>? {
        return sizeAxisModel
    }

    init {
        this.dataFrame = dataFrame
        this.annotationDataFrame = annotationDataFrame
        this.visual = visual
        this.root = root

        // Assign numerical columns
        for (axisModel in root.axisOrder!!) {
            if (axisModel.isNumerical) {
                if (xAxisModel == null) {
                    xAxisModel = axisModel
                } else if (yAxisModel == null) {
                    yAxisModel = axisModel
                    break
                }
            }
        }

        // If two numerical columns couldn't be found, then find categorical columns
        if (yAxisModel == null) {
            for (axisModel in root.axisOrder!!) {
                if (!axisModel.isNumerical) {
                    if (xAxisModel == null) {
                        xAxisModel = axisModel
                    } else if (yAxisModel == null) {
                        yAxisModel = axisModel
                        break
                    }
                }
            }
        }
        sizeAxisModel = null
        x.addSingleSelectionListener(object : SingleSelectionListener<AxisModel<Row, C>> {
            override fun selectionChanged(event: SingleSelectionEvent<AxisModel<Row, C>>) {
                xAxisModel = event.currentSelection
                fireScatterPlotChanged()
            }
        })
        y.addSingleSelectionListener(object : SingleSelectionListener<AxisModel<Row, C>> {
            override fun selectionChanged(event: SingleSelectionEvent<AxisModel<Row, C>>) {
                yAxisModel = event.currentSelection
                fireScatterPlotChanged()
            }
        })
        y2.addSingleSelectionListener(object : SingleSelectionListener<AxisModel<Row, C>> {
            override fun selectionChanged(event: SingleSelectionEvent<AxisModel<Row, C>>) {
                y2AxisModel = event.currentSelection
                fireScatterPlotChanged()
            }
        })
        size.addSingleSelectionListener(object : SingleSelectionListener<AxisModel<Row, C>> {
            override fun selectionChanged(event: SingleSelectionEvent<AxisModel<Row, C>>) {
                sizeAxisModel = event.currentSelection
                fireScatterPlotChanged()
            }
        })
        if (xAxisModel != null) {
            x.selected = xAxisModel
        }
        if (yAxisModel != null) {
            y.selected = yAxisModel
        }
        if (sizeAxisModel != null) {
            size.selected = sizeAxisModel
        }
    }
}