/*
 * Copyright (c) 2013 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.distributions

import com.macrofocus.common.filter.Filter
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import org.mkui.coloring.MutableColoring
import org.mkui.colormapping.MutableColorMapping
import org.mkui.visual.Visual
import org.molap.subset.DistributionDimension

interface DistributionsModel<Row, Column, Value, Bin> {
    fun getVisual(): Visual<Row, Column>

    /**
     * Obtains the model used for probing objects in the scatter plot (aka mouse over).
     *
     * @return the model used for probing.
     */
    fun getProbing(): MutableSingleSelection<Row>

    /**
     * Obtains the model used for selecting objects in the scatter plot.
     *
     * @return the model used for selection.
     */
    fun getSelection(): MutableSelection<Row>

    /**
     * Obtains the model used for filtering objects in the scatter plot.
     *
     * @return the model used for filtering.
     */
    fun getFilter(): Filter<Row>

    /**
     * Gets model holding the color mapping of each object.
     *
     * @return the model holding the color of each object.
     */
    fun getColorMapping(): MutableColorMapping<Row, Column>

    /**
     * Gets model holding the colors of each object.
     *
     * @return the model holding the color of each object.
     */
    fun getColoring(): MutableColoring<Row>

    /**
     * Returns the number of data objects (typically rows).
     *
     * @return the number of data objects
     */
    val objectCount: Int

    /**
     * @param index
     *
     * @return
     */
    fun getObject(index: Int): Row
    fun getIndex(row: Row): Int

    /**
     * Gets the normalized location of an axis.
     *
     * @param axisModel the axis
     *
     * @return its normalized location within the group
     */
    fun getLocation(axisModel: AxisModel<Row, Column>): Double

    /**
     * Returns the number of axis composing the parallel coordinates plot.
     *
     * @return the number of axis
     */
    fun getAxisCount(): Int

    /**
     * Gets all the axis composing this parallel coordinates plot.
     *
     * @return
     */
    fun getAxisGroupModel(): AxisGroupModel<Row, Column>

    /**
     * Sets the location of an axis within its group.
     *
     * @param axisModel the axis
     * @param l         its normalized location
     */
    fun setLocation(axisModel: AxisModel<Row, Column>, l: Double)

    /**
     * Add a listener for receiving interesting events about changes in the model.
     *
     * @param listener the listener to add.
     */
    fun addDistributionsListener(listener: DistributionsListener)

    /**
     * Remove a listener from the list of listeners that should be notified about changes in the model.
     *
     * @param listener the listener to remove.
     */
    fun removeDistributionsListener(listener: DistributionsListener)
    fun createDistributionDimension(column: Column, distributionStrategy: DistributionDimension.DistributionStrategy<Value,Bin>): DistributionDimension<Row, Value, Bin>
    fun removeDistributionDimension(distributionDimension: DistributionDimension<Row, Value, Bin>)
}