/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.high_d.scatterplotmatrix

import com.macrofocus.common.filter.MutableFilter
import com.macrofocus.common.selection.MutableSelection
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.high_d.axis.AxisModel
import com.macrofocus.high_d.axis.group.AxisGroupModel
import org.mkui.coloring.MutableColoring
import org.mkui.colormapping.MutableColorMapping
import org.mkui.visual.Visual

interface ScatterPlotMatrixModel<Row, Column> {
    fun getVisual(): Visual<Row, Column>

    /**
     * Obtains the model used for probing objects in the scatter plot (aka mouse over).
     *
     * @return the model used for probing.
     */
    fun getProbing(): MutableSingleSelection<Row>

    /**
     * Obtains the model used for selecting objects in the scatter plot.
     *
     * @return the model used for selection.
     */
    fun getSelection(): MutableSelection<Row>

    /**
     * Obtains the model used for filtering objects in the scatter plot.
     *
     * @return the model used for filtering.
     */
    fun getFilter(): MutableFilter<Row>

    /**
     * Gets model holding the color mapping of each object.
     *
     * @return the model holding the color of each object.
     */
    fun getColorMapping(): MutableColorMapping<Row, Column>

    /**
     * Gets model holding the colors of each object.
     *
     * @return the model holding the color of each object.
     */
    fun getColoring(): MutableColoring<Row>

    /**
     * Returns the number of data objects (typically rows).
     *
     * @return the number of data objects
     */
    fun getObjectCount(): Int

    /**
     * @param index
     *
     * @return
     */
    fun getObject(index: Int): Row

    /**
     * Gets the normalized location of an axis.
     *
     * @param axisModel the axis
     *
     * @return its normalized location within the group
     */
    fun getLocation(axisModel: AxisModel<Row, Column>): Double

    /**
     * Returns the number of axis composing the parallel coordinates plot.
     *
     * @return the number of axis
     */
    fun getAxisCount(): Int
    fun getAxisModel(index: Int): AxisModel<Row, Column>

    /**
     * Gets all the axis composing this parallel coordinates plot.
     *
     * @return
     */
    fun getAxisGroupModel(): AxisGroupModel<Row,Column>

    /**
     * Sets the location of an axis within its group.
     *
     * @param axisModel the axis
     * @param l         its normalized location
     */
    fun setLocation(axisModel: AxisModel<Row,Column>, l: Double)

    /**
     * Add a listener for receiving interesting events about changes in the model.
     *
     * @param listener the listener to add.
     */
    fun addScatterPlotMatrixListener(listener: ScatterPlotMatrixListener)

    /**
     * Remove a listener from the list of listeners that should be notified about changes in the model.
     *
     * @param listener the listener to remove.
     */
    fun removeScatterPlotMatrixListener(listener: ScatterPlotMatrixListener)
}