/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

import kotlin.math.max

object CubicCurve2D {
    /**
     * Returns the square of the flatness of the cubic curve specified
     * by the control points stored in the indicated array at the
     * indicated index. The flatness is the maximum distance
     * of a control point from the line connecting the end points.
     *
     * @param coords an array containing coordinates
     * @param offset the index of `coords` from which to begin
     * getting the end points and control points of the curve
     *
     * @return the square of the flatness of the `CubicCurve2D`
     * specified by the coordinates in `coords` at
     * the specified offset.
     *
     * @since 1.2
     */
    fun getFlatnessSq(coords: DoubleArray, offset: Int): Double {
        return getFlatnessSq(
            coords[offset + 0], coords[offset + 1],
            coords[offset + 2], coords[offset + 3],
            coords[offset + 4], coords[offset + 5],
            coords[offset + 6], coords[offset + 7]
        )
    }

    /**
     * Returns the square of the flatness of the cubic curve specified
     * by the indicated control points. The flatness is the maximum distance
     * of a control point from the line connecting the end points.
     *
     * @param x1     the X coordinate that specifies the start point
     * of a `CubicCurve2D`
     * @param y1     the Y coordinate that specifies the start point
     * of a `CubicCurve2D`
     * @param ctrlx1 the X coordinate that specifies the first control point
     * of a `CubicCurve2D`
     * @param ctrly1 the Y coordinate that specifies the first control point
     * of a `CubicCurve2D`
     * @param ctrlx2 the X coordinate that specifies the second control point
     * of a `CubicCurve2D`
     * @param ctrly2 the Y coordinate that specifies the second control point
     * of a `CubicCurve2D`
     * @param x2     the X coordinate that specifies the end point
     * of a `CubicCurve2D`
     * @param y2     the Y coordinate that specifies the end point
     * of a `CubicCurve2D`
     *
     * @return the square of the flatness of the `CubicCurve2D`
     * represented by the specified coordinates.
     *
     * @since 1.2
     */
    fun getFlatnessSq(
        x1: Double, y1: Double,
        ctrlx1: Double, ctrly1: Double,
        ctrlx2: Double, ctrly2: Double,
        x2: Double, y2: Double
    ): Double {
        return max(
            Line2D.ptSegDistSq(x1, y1, x2, y2, ctrlx1, ctrly1),
            Line2D.ptSegDistSq(x1, y1, x2, y2, ctrlx2, ctrly2)
        )
    }

    /**
     * Subdivides the cubic curve specified by the coordinates
     * stored in the `src` array at indices `srcoff`
     * through (`srcoff`&nbsp;+&nbsp;7) and stores the
     * resulting two subdivided curves into the two result arrays at the
     * corresponding indices.
     * Either or both of the `left` and `right`
     * arrays may be `null` or a reference to the same array
     * as the `src` array.
     * Note that the last point in the first subdivided curve is the
     * same as the first point in the second subdivided curve. Thus,
     * it is possible to pass the same array for `left`
     * and `right` and to use offsets, such as `rightoff`
     * equals (`leftoff` + 6), in order
     * to avoid allocating extra storage for this common point.
     *
     * @param src      the array holding the coordinates for the source curve
     * @param srcoff   the offset into the array of the beginning of the
     * the 6 source coordinates
     * @param left     the array for storing the coordinates for the first
     * half of the subdivided curve
     * @param leftoff  the offset into the array of the beginning of the
     * the 6 left coordinates
     * @param right    the array for storing the coordinates for the second
     * half of the subdivided curve
     * @param rightoff the offset into the array of the beginning of the
     * the 6 right coordinates
     *
     * @since 1.2
     */
    fun subdivide(
        src: DoubleArray, srcoff: Int,
        left: DoubleArray?, leftoff: Int,
        right: DoubleArray?, rightoff: Int
    ) {
        var x1 = src[srcoff + 0]
        var y1 = src[srcoff + 1]
        var ctrlx1 = src[srcoff + 2]
        var ctrly1 = src[srcoff + 3]
        var ctrlx2 = src[srcoff + 4]
        var ctrly2 = src[srcoff + 5]
        var x2 = src[srcoff + 6]
        var y2 = src[srcoff + 7]
        if (left != null) {
            left[leftoff + 0] = x1
            left[leftoff + 1] = y1
        }
        if (right != null) {
            right[rightoff + 6] = x2
            right[rightoff + 7] = y2
        }
        x1 = (x1 + ctrlx1) / 2.0
        y1 = (y1 + ctrly1) / 2.0
        x2 = (x2 + ctrlx2) / 2.0
        y2 = (y2 + ctrly2) / 2.0
        var centerx = (ctrlx1 + ctrlx2) / 2.0
        var centery = (ctrly1 + ctrly2) / 2.0
        ctrlx1 = (x1 + centerx) / 2.0
        ctrly1 = (y1 + centery) / 2.0
        ctrlx2 = (x2 + centerx) / 2.0
        ctrly2 = (y2 + centery) / 2.0
        centerx = (ctrlx1 + ctrlx2) / 2.0
        centery = (ctrly1 + ctrly2) / 2.0
        if (left != null) {
            left[leftoff + 2] = x1
            left[leftoff + 3] = y1
            left[leftoff + 4] = ctrlx1
            left[leftoff + 5] = ctrly1
            left[leftoff + 6] = centerx
            left[leftoff + 7] = centery
        }
        if (right != null) {
            right[rightoff + 0] = centerx
            right[rightoff + 1] = centery
            right[rightoff + 2] = ctrlx2
            right[rightoff + 3] = ctrly2
            right[rightoff + 4] = x2
            right[rightoff + 5] = y2
        }
    }
}