package org.kamaeleo.geom.curve

import com.macrofocus.common.collection.arraycopy
import kotlin.jvm.Synchronized

/**
 * PascalsTriangle can be used for O(1) lookup of the nCr function.
 */
internal object PascalsTriangle {
    private var pt = arrayOf<DoubleArray?>(doubleArrayOf(1.0))

    /**
     * The nCr function returns the number of ways r things can be chosen from a set of size n.
     * Mathematically, it is defined as: n! / (r! * (n - r)!)
     * Although the result is always a whole number, double precision is used because
     * the maximum value a double can represent is larger than long.  Thus, large returned
     * values will only be an approximation of the actual value.  If the result exceeds the
     * capabilities of double precision then the result can be checked using Double.isInfinite(...).
     * For example: System.out.println(PascalsTriangle.nCr(1030, 515)); // outputs: Infinity
     * If the value of n or r is less than 0 or the value of r is greater than n then 0 is
     * returned.
     */
    @Synchronized
    fun nCr(n: Int, r: Int): Double {
        var r = r
        if (n < 0 || r < 0 || r > n) return 0.0
        if (n >= pt.size) {
            val d = 2 * pt.size
            val pt2: Array<DoubleArray?>
            pt2 = if (n > d) arrayOfNulls(n + 1) else arrayOfNulls(d + 1)
            arraycopy(pt, 0, pt2, 0, pt.size)
            for (i in pt.size until pt2.size) {
                pt2[i] = DoubleArray(i / 2 + 1)
                pt2[i]!![0] = 1.0
                for (j in 1 until pt2[i]!!.size) {
                    var x = pt2[i - 1]!![j - 1]
                    if (j < pt2[i - 1]!!.size) x += pt2[i - 1]!![j] else x = 2 * x
                    pt2[i]!![j] = x
                }
            }
            pt = pt2
        }
        if (2 * r > n) r = n - r
        return pt[n]!![r]
    }
}