package org.kamaeleo.geom.curve

/**
 *
 * The Cardinal-spline passes through the points in the control-path specified by the group-iterator.
 * However, the curve does not pass through the first or the last control-point, it begins at the
 * second control-point and ends at the second last control-point.
 *
 *
 *
 * There is a single parameter, alpha, that gives some control over the shape of the curve.  When
 * the value of alpha is 0.5 the curve becomes the CatmullRomSpline.  Figure 1 shows an example of a
 * CardinalSpline.
 *
 *
 *
 * <center><img align="center" src="doc-files/cardinal1.gif"></img></center>
 *
 * @see org.kamaeleo.geom.curve.CatmullRomSpline
 */
class CardinalSpline(cp: ControlPath?, gi: GroupIterator?) : ParametricCurve(cp, gi) {
    /**
     * Returns the value of alpha.  The default value is 0.5.
     *
     * @see .setAlpha
     */
    /**
     * Sets the value of alpha.
     *
     * @see .getAlpha
     */
    var alpha = 0.5

    /**
     * The requirements for this curve are the group-iterator must be in-range and have a group size of at least 4.
     * If these requirements are not met then this method returns quietly.
     */
    override fun appendTo(mp: MultiPath) {
        if (!gi!!.isInRange(0, cp!!.numPoints())) return
        if (gi!!.groupSize < 4) return
        gi!!.set(0, 0)
        for (i in 0..3) pt[i] = cp!!.getPoint(gi!!.next())!!.location
        val d = DoubleArray(mp.dimension + 1)
        eval(d)
        if (connect) mp.lineTo(d) else mp.moveTo(d)
        gi!!.set(0, 0)
        while (true) {
            val index_i: Int = gi!!.index_i()
            val count_j: Int = gi!!.count_j()
            for (i in 0..3) {
                if (!gi!!.hasNext()) return
                pt[i] = cp!!.getPoint(gi!!.next())!!.location
            }
            gi!!.set(index_i, count_j)
            gi!!.next()
            BinaryCurveApproximationAlgorithm.genPts(this, 0.0, 1.0, mp)
        }
    }

    override fun eval(p: DoubleArray) {
        val t = p[p.size - 1]
        val t2 = t * t
        val t3 = t2 * t
        val a = 2 * t3 - 3 * t2 + 1
        val b = -2 * t3 + 3 * t2
        val c = alpha * (t3 - 2 * t2 + t)
        val d = alpha * (t3 - t2)
        for (i in 0 until p.size - 1) p[i] =
            a * pt[1]!![i] + b * pt[2]!![i] + c * (pt[2]!![i] - pt[0]!![i]) + d * (pt[3]!![i] - pt[1]!![i])
    }

    /**
     * Returns a value of 1.
     */
    override val sampleLimit: Int
        get() = 1

    companion object {
        private val pt = arrayOfNulls<DoubleArray>(4)
    }
}