/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

class GeneralPath : Path2D.Double {
    /**
     * Constructs a new empty single precision `GeneralPath` object
     * with a default winding rule of [.WIND_NON_ZERO].
     *
     * @since 1.2
     */
    constructor() : super(WIND_NON_ZERO, INIT_SIZE) {}

    /**
     * Constructs a new `GeneralPath` object with the specified
     * winding rule to control operations that require the interior of the
     * path to be defined.
     *
     * @param rule the winding rule
     *
     * @see .WIND_EVEN_ODD
     *
     * @see .WIND_NON_ZERO
     *
     * @since 1.2
     */
    constructor(rule: Int) : super(rule, INIT_SIZE) {}

    /**
     * Constructs a new `GeneralPath` object with the specified
     * winding rule and the specified initial capacity to store path
     * coordinates.
     * This number is an initial guess as to how many path segments
     * will be added to the path, but the storage is expanded as
     * needed to store whatever path segments are added.
     *
     * @param rule            the winding rule
     * @param initialCapacity the estimate for the number of path segments
     * in the path
     *
     * @see .WIND_EVEN_ODD
     *
     * @see .WIND_NON_ZERO
     *
     * @since 1.2
     */
    constructor(rule: Int, initialCapacity: Int) : super(rule, initialCapacity) {}

    /**
     * Constructs a new `GeneralPath` object from an arbitrary
     * [Shape] object.
     * All of the initial geometry and the winding rule for this path are
     * taken from the specified `Shape` object.
     *
     * @param s the specified `Shape` object
     *
     * @since 1.2
     */
    constructor(s: Shape) : super(s.pathIterator) {}
    constructor(s: Shape, at: AffineTransform?) : super(s, at) {}
    internal constructor(
        windingRule: Int,
        pointTypes: ByteArray?,
        numTypes: Int,
        pointCoords: DoubleArray,
        numCoords: Int
    ) {
        // used to construct from native
        this.windingRule = windingRule
        this.pointTypes = pointTypes!!
        this.numTypes = numTypes
        this.doubleCoords = pointCoords
        this.numCoords = numCoords
    }
}