/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

import com.macrofocus.common.collection.arraycopy
import kotlin.jvm.Synchronized
import kotlin.jvm.Transient
import kotlin.math.min

/**
 * The `Path2D` class provides a simple, yet flexible
 * shape which represents an arbitrary geometric path.
 * It can fully represent any path which can be iterated by the
 * [PathIterator] interface including all of its segment
 * types and winding rules and it implements all of the
 * basic hit testing methods of the [Shape] interface.
 *
 *
 * Use [Path2D.Float] when dealing with data that can be represented
 * and used with floating point precision.  Use [Path2D.Double]
 * for data that requires the accuracy or range of double precision.
 *
 *
 * `Path2D` provides exactly those facilities required for
 * basic construction and management of a geometric path and
 * implementation of the above interfaces with little added
 * interpretation.
 * If it is useful to manipulate the interiors of closed
 * geometric shapes beyond simple hit testing then the
 * [Area] class provides additional capabilities
 * specifically targeted at closed figures.
 * While both classes nominally implement the `Shape`
 * interface, they differ in purpose and together they provide
 * two useful views of a geometric shape where `Path2D`
 * deals primarily with a trajectory formed by path segments
 * and `Area` deals more with interpretation and manipulation
 * of enclosed regions of 2D geometric space.
 *
 *
 * The [PathIterator] interface has more detailed descriptions
 * of the types of segments that make up a path and the winding rules
 * that control how to determine which regions are inside or outside
 * the path.
 *
 * @author Jim Graham
 * @version %I%, %G%
 * @since 1.6
 */
abstract class Path2D : Shape {
    @Transient
    var pointTypes: ByteArray

    @Transient
    var numTypes = 0

    @Transient
    var numCoords = 0

    @Transient
    var windingRule = 0

    /**
     * Constructs a new empty `Path2D` object.
     * It is assumed that the package sibling subclass that is
     * defaulting to this constructor will fill in all values.
     *
     * @since 1.6
     */
    /* private protected */
    internal constructor() {
        pointTypes = ByteArray(0);
    }

    /**
     * Constructs a new `Path2D` object from the given
     * specified initial values.
     * This method is only intended for internal use and should
     * not be made public if the other constructors for this class
     * are ever exposed.
     *
     * @param rule         the winding rule
     * @param initialTypes the size to make the initial array to
     * store the path segment types
     *
     * @since 1.6
     */
    /* private protected */
    internal constructor(rule: Int, initialTypes: Int) {
        windingRule = rule
        pointTypes = ByteArray(initialTypes)
    }

    abstract fun cloneCoordsDouble(at: AffineTransform?): DoubleArray
    abstract fun append(x: kotlin.Double, y: kotlin.Double)
    abstract fun createTransformedShape(transform: AffineTransform?): Shape?

    /**
     * Adds a point to the path by moving to the specified
     * coordinates specified in double precision.
     *
     * @param x the specified X coordinate
     * @param y the specified Y coordinate
     *
     * @since 1.6
     */
    abstract fun moveTo(x: kotlin.Double, y: kotlin.Double)

    /**
     * Adds a point to the path by drawing a straight line from the
     * current coordinates to the new specified coordinates
     * specified in double precision.
     *
     * @param x the specified X coordinate
     * @param y the specified Y coordinate
     *
     * @since 1.6
     */
    abstract fun lineTo(x: kotlin.Double, y: kotlin.Double)

    /**
     * Adds a curved segment, defined by two new points, to the path by
     * drawing a Quadratic curve that intersects both the current
     * coordinates and the specified coordinates `(x2,y2)`,
     * using the specified point `(x1,y1)` as a quadratic
     * parametric control point.
     * All coordinates are specified in double precision.
     *
     * @param x1 the X coordinate of the quadratic control point
     * @param y1 the Y coordinate of the quadratic control point
     * @param x2 the X coordinate of the final end point
     * @param y2 the Y coordinate of the final end point
     *
     * @since 1.6
     */
    abstract fun quadTo(
        x1: kotlin.Double, y1: kotlin.Double,
        x2: kotlin.Double, y2: kotlin.Double
    )

    /**
     * Adds a curved segment, defined by three new points, to the path by
     * drawing a Bzier curve that intersects both the current
     * coordinates and the specified coordinates `(x3,y3)`,
     * using the specified points `(x1,y1)` and `(x2,y2)` as
     * Bzier control points.
     * All coordinates are specified in double precision.
     *
     * @param x1 the X coordinate of the first Bzier control point
     * @param y1 the Y coordinate of the first Bzier control point
     * @param x2 the X coordinate of the second Bzier control point
     * @param y2 the Y coordinate of the second Bzier control point
     * @param x3 the X coordinate of the final end point
     * @param y3 the Y coordinate of the final end point
     *
     * @since 1.6
     */
    abstract fun curveTo(
        x1: kotlin.Double, y1: kotlin.Double,
        x2: kotlin.Double, y2: kotlin.Double,
        x3: kotlin.Double, y3: kotlin.Double
    )

    /**
     * Closes the current subpath by drawing a straight line back to
     * the coordinates of the last `moveTo`.  If the path is already
     * closed then this method has no effect.
     *
     * @since 1.6
     */
    @Synchronized
    fun closePath() {
        if (numTypes == 0 || pointTypes[numTypes - 1] != SEG_CLOSE) {
            needRoom(true, 0)
            pointTypes[numTypes++] = SEG_CLOSE
        }
    }

    abstract fun needRoom(needMove: Boolean, newCoords: Int)

    /**
     * Appends the geometry of the specified
     * [PathIterator] object
     * to the path, possibly connecting the new geometry to the existing
     * path segments with a line segment.
     * If the `connect` parameter is `true` and the
     * path is not empty then any initial `moveTo` in the
     * geometry of the appended `Shape` is turned into a
     * `lineTo` segment.
     * If the destination coordinates of such a connecting `lineTo`
     * segment match the ending coordinates of a currently open
     * subpath then the segment is omitted as superfluous.
     * The winding rule of the specified `Shape` is ignored
     * and the appended geometry is governed by the winding
     * rule specified for this path.
     *
     * @param pi      the `PathIterator` whose geometry is appended to
     * this path
     * @param connect a boolean to control whether or not to turn an initial
     * `moveTo` segment into a `lineTo` segment
     * to connect the new geometry to the existing path
     *
     * @since 1.6
     */
    abstract fun append(pi: PathIterator, connect: Boolean)

//    /**
//     * Sets the winding rule for this path to the specified value.
//     *
//     * @param rule an integer representing the specified
//     * winding rule
//     *
//     * @throws IllegalArgumentException if
//     * `rule` is not either
//     * [.WIND_EVEN_ODD] or
//     * [.WIND_NON_ZERO]
//     * @see .getWindingRule
//     *
//     * @since 1.6
//     */
//    fun setWindingRule(rule: Int) {
//        require(!(rule != WIND_EVEN_ODD && rule != WIND_NON_ZERO)) {
//            "winding rule must be " +
//                    "WIND_EVEN_ODD or " +
//                    "WIND_NON_ZERO"
//        }
//        windingRule = rule
//    }

    /**
     * Returns the coordinates most recently added to the end of the path
     * as a [Point2D] object.
     *
     * @return a `Point2D` object containing the ending coordinates of
     * the path or `null` if there are no points in the path.
     *
     * @since 1.6
     */
    @get:Synchronized
    val currentPoint: org.kamaeleo.geom.Point2D?
        get() {
            var index = numCoords
            if (numTypes < 1 || index < 1) {
                return null
            }
            if (pointTypes[numTypes - 1] == SEG_CLOSE) {
                loop@ for (i in numTypes - 2 downTo 1) {
                    when (pointTypes[i]) {
                        SEG_MOVETO -> break@loop
                        SEG_LINETO -> index -= 2
                        SEG_QUADTO -> index -= 4
                        SEG_CUBICTO -> index -= 6
                        SEG_CLOSE -> {
                        }
                    }
                }
            }
            return getPoint(index - 2)
        }

    abstract fun getPoint(coordindex: Int): Point2D?

    /**
     * Resets the path to empty.  The append position is set back to the
     * beginning of the path and all coordinates and point types are
     * forgotten.
     *
     * @since 1.6
     */
    @Synchronized
    fun reset() {
        numCoords = 0
        numTypes = numCoords
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.6
     */
    override operator fun contains(p: Point2D): Boolean {
        return contains(p.x, p.y)
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.6
     */
    fun contains(x: kotlin.Double, y: kotlin.Double): Boolean {
        return if (x * 0.0 + y * 0.0 == 0.0) {
            /* N * 0.0 is 0.0 only if N is finite.
             * Here we know that both x and y are finite.
             */
            if (numTypes < 2) {
                return false
            }
            val mask = if (windingRule == WIND_NON_ZERO) -1 else 1
            pointCrossings(x, y) and mask != 0
        } else {
            /* Either x or y was infinite or NaN.
             * A NaN always produces a negative response to any test
             * and Infinity values cannot be "inside" any path so
             * they should return false as well.
             */
            false
        }
    }

    abstract fun pointCrossings(px: kotlin.Double, py: kotlin.Double): Int

    /**
     * {@inheritDoc}
     *
     *
     * This method object may conservatively return true in
     * cases where the specified rectangular area intersects a
     * segment of the path, but that segment does not represent a
     * boundary between the interior and exterior of the path.
     * Such a case may occur if some set of segments of the
     * path are retraced in the reverse direction such that the
     * two sets of segments cancel each other out without any
     * interior area between them.
     * To determine whether segments represent true boundaries of
     * the interior of the path would require extensive calculations
     * involving all of the segments of the path and the winding
     * rule and are thus beyond the scope of this implementation.
     *
     * @since 1.6
     */
    override fun intersects(r: Rectangle2D): Boolean {
        return intersects(r.x, r.y, r.width, r.height)
    }

    /**
     * {@inheritDoc}
     *
     *
     * This method object may conservatively return true in
     * cases where the specified rectangular area intersects a
     * segment of the path, but that segment does not represent a
     * boundary between the interior and exterior of the path.
     * Such a case may occur if some set of segments of the
     * path are retraced in the reverse direction such that the
     * two sets of segments cancel each other out without any
     * interior area between them.
     * To determine whether segments represent true boundaries of
     * the interior of the path would require extensive calculations
     * involving all of the segments of the path and the winding
     * rule and are thus beyond the scope of this implementation.
     *
     * @since 1.6
     */
    fun intersects(x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double): Boolean {
        if ((x + w).isNaN() || (y + h).isNaN()) {
            /* [xy]+[wh] is NaN if any of those values are NaN,
             * or if adding the two together would produce NaN
             * by virtue of adding opposing Infinte values.
             * Since we need to add them below, their sum must
             * not be NaN.
             * We return false because NaN always produces a
             * negative response to tests
             */
            return false
        }
        if (w <= 0 || h <= 0) {
            return false
        }
        val mask = if (windingRule == WIND_NON_ZERO) -1 else 2
        val crossings = rectCrossings(x, y, x + w, y + h)
        return crossings == Curve.RECT_INTERSECTS ||
                crossings and mask != 0
    }

    /**
     * {@inheritDoc}
     *
     *
     * This method object may conservatively return false in
     * cases where the specified rectangular area intersects a
     * segment of the path, but that segment does not represent a
     * boundary between the interior and exterior of the path.
     * Such segments could lie entirely within the interior of the
     * path if they are part of a path with a [.WIND_NON_ZERO]
     * winding rule or if the segments are retraced in the reverse
     * direction such that the two sets of segments cancel each
     * other out without any exterior area falling between them.
     * To determine whether segments represent true boundaries of
     * the interior of the path would require extensive calculations
     * involving all of the segments of the path and the winding
     * rule and are thus beyond the scope of this implementation.
     *
     * @since 1.6
     */
    operator fun contains(r: Rectangle2D): Boolean {
        return contains(r.x, r.y, r.width, r.height)
    }

    /**
     * {@inheritDoc}
     *
     *
     * This method object may conservatively return false in
     * cases where the specified rectangular area intersects a
     * segment of the path, but that segment does not represent a
     * boundary between the interior and exterior of the path.
     * Such segments could lie entirely within the interior of the
     * path if they are part of a path with a [.WIND_NON_ZERO]
     * winding rule or if the segments are retraced in the reverse
     * direction such that the two sets of segments cancel each
     * other out without any exterior area falling between them.
     * To determine whether segments represent true boundaries of
     * the interior of the path would require extensive calculations
     * involving all of the segments of the path and the winding
     * rule and are thus beyond the scope of this implementation.
     *
     * @since 1.6
     */
    fun contains(x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double): Boolean {
        if ((x + w).isNaN() || (y + h).isNaN()) {
            /* [xy]+[wh] is NaN if any of those values are NaN,
             * or if adding the two together would produce NaN
             * by virtue of adding opposing Infinte values.
             * Since we need to add them below, their sum must
             * not be NaN.
             * We return false because NaN always produces a
             * negative response to tests
             */
            return false
        }
        if (w <= 0 || h <= 0) {
            return false
        }
        val mask = if (windingRule == WIND_NON_ZERO) -1 else 2
        val crossings = rectCrossings(x, y, x + w, y + h)
        return crossings != Curve.RECT_INTERSECTS &&
                crossings and mask != 0
    }

    abstract fun rectCrossings(
        rxmin: kotlin.Double, rymin: kotlin.Double,
        rxmax: kotlin.Double, rymax: kotlin.Double
    ): Int

    /**
     * The `Double` class defines a geometric path with
     * coordinates stored in double precision floating point.
     *
     * @since 1.6
     */
    open class Double : Path2D {
        @Transient
        var doubleCoords: DoubleArray
        /**
         * Constructs a new empty double precision `Path2D` object
         * with the specified winding rule and the specified initial
         * capacity to store path segments.
         * This number is an initial guess as to how many path segments
         * are in the path, but the storage is expanded as needed to store
         * whatever path segments are added to this path.
         *
         * @param rule            the winding rule
         * @param initialCapacity the estimate for the number of path segments
         * in the path
         *
         * @see .WIND_EVEN_ODD
         *
         * @see .WIND_NON_ZERO
         *
         * @since 1.6
         */
        /**
         * Constructs a new empty double precision `Path2D` object
         * with a default winding rule of [.WIND_NON_ZERO].
         *
         * @since 1.6
         */
        /**
         * Constructs a new empty double precision `Path2D` object
         * with the specified winding rule to control operations that
         * require the interior of the path to be defined.
         *
         * @param rule the winding rule
         *
         * @see .WIND_EVEN_ODD
         *
         * @see .WIND_NON_ZERO
         *
         * @since 1.6
         */
        constructor(rule: Int = WIND_NON_ZERO, initialCapacity: Int = INIT_SIZE) : super(rule, initialCapacity) {
            doubleCoords = DoubleArray(initialCapacity * 2)
        }

        /**
         * Constructs a new double precision `Path2D` object
         * from an arbitrary [Shape] object.
         * All of the initial geometry and the winding rule for this path are
         * taken from the specified `Shape` object.
         *
         * @param s the specified `Shape` object
         *
         * @since 1.6
         */
        constructor(p2d: Path2D) {
            windingRule = p2d.windingRule
            numTypes = p2d.numTypes
            pointTypes = copyOf(
                p2d.pointTypes,
                p2d.pointTypes.size
            )
            numCoords = p2d.numCoords
            doubleCoords = p2d.cloneCoordsDouble(null)
        }

        /**
         * Constructs a new double precision `Path2D` object
         * from an arbitrary [java.awt.Shape] object, transformed by an
         * [AffineTransform] object.
         * All of the initial geometry and the winding rule for this path are
         * taken from the specified `Shape` object and transformed
         * by the specified `AffineTransform` object.
         *
         * @param s  the specified `Shape` object
         * @param at the specified `AffineTransform` object
         *
         * @since 1.6
         */
        constructor(s: Shape, at: AffineTransform?) {
            if (s is Path2D) {
                val p2d = s
                windingRule = p2d.windingRule
                numTypes = p2d.numTypes
                // trim arrays:
                pointTypes = copyOf(p2d.pointTypes, p2d.numTypes)
                numCoords = p2d.numCoords
                doubleCoords = p2d.cloneCoordsDouble(at)
            } else {
                val pi: PathIterator = s.getPathIterator(at)
                windingRule = pi.windingRule
                pointTypes = ByteArray(INIT_SIZE)
                doubleCoords = DoubleArray(INIT_SIZE * 2)
                append(pi, false)
            }
        }

        constructor(pi: PathIterator) {
            windingRule = pi.windingRule
            pointTypes = ByteArray(INIT_SIZE)
            doubleCoords = DoubleArray(INIT_SIZE * 2)
            append(pi, false)
        }

        override fun cloneCoordsDouble(at: AffineTransform?): DoubleArray {
            // trim arrays:
            val ret: DoubleArray
            if (at == null) {
                ret = copyOf(doubleCoords, numCoords)
            } else {
                ret = DoubleArray(numCoords)
                at.transform(doubleCoords, 0, ret, 0, numCoords / 2)
            }
            return ret
        }

        override fun append(x: kotlin.Double, y: kotlin.Double) {
            doubleCoords[numCoords++] = x
            doubleCoords[numCoords++] = y
        }

        override fun getPoint(coordindex: Int): Point2D {
            return Point2D.Double(
                doubleCoords[coordindex],
                doubleCoords[coordindex + 1]
            )
        }

        override fun needRoom(needMove: Boolean, newCoords: Int) {
            check(!(needMove && numTypes == 0)) {
                "missing initial moveto " +
                        "in path definition"
            }
            var size = pointTypes.size
            if (numTypes >= size) {
                var grow = size
                if (grow > EXPAND_MAX) {
                    grow = EXPAND_MAX
                }
                pointTypes = copyOf(pointTypes, size + grow)
            }
            size = doubleCoords.size
            if (numCoords + newCoords > size) {
                var grow = size
                if (grow > EXPAND_MAX * 2) {
                    grow = EXPAND_MAX * 2
                }
                if (grow < newCoords) {
                    grow = newCoords
                }
                doubleCoords = copyOf(doubleCoords, size + grow)
            }
        }

        override fun pointCrossings(px: kotlin.Double, py: kotlin.Double): Int {
            if (numTypes == 0) {
                return 0
            }
            var movx: kotlin.Double
            val coords = doubleCoords
            movx = coords[0]
            var curx = movx
            var movy: kotlin.Double
            movy = coords[1]
            var cury = movy
            var crossings = 0
            var ci = 2
            for (i in 1 until numTypes) {
                var endy: kotlin.Double
                var endx: kotlin.Double
                when (pointTypes[i]) {
                    SEG_MOVETO -> {
                        if (cury != movy) {
                            crossings += Curve.pointCrossingsForLine(
                                px, py,
                                curx, cury,
                                movx, movy
                            )
                        }
                        run {
                            curx = coords[ci++]
                            movx = curx
                        }
                        run {
                            cury = coords[ci++]
                            movy = cury
                        }
                    }
                    SEG_LINETO -> {
                        crossings += Curve.pointCrossingsForLine(px, py,
                            curx, cury,
                            coords[ci++].also {
                                endx = it
                            },
                            coords[ci++].also {
                                endy = it
                            })
                        curx = endx
                        cury = endy
                    }
                    SEG_QUADTO -> {
                        crossings += Curve.pointCrossingsForQuad(
                            px, py,
                            curx, cury,
                            coords[ci++],
                            coords[ci++],
                            coords[ci++].also {
                                endx = it
                            },
                            coords[ci++].also {
                                endy = it
                            },
                            0
                        )
                        curx = endx
                        cury = endy
                    }
                    SEG_CUBICTO -> {
                        crossings += Curve.pointCrossingsForCubic(
                            px, py,
                            curx, cury,
                            coords[ci++],
                            coords[ci++],
                            coords[ci++],
                            coords[ci++],
                            coords[ci++].also {
                                endx = it
                            },
                            coords[ci++].also {
                                endy = it
                            },
                            0
                        )
                        curx = endx
                        cury = endy
                    }
                    SEG_CLOSE -> {
                        if (cury != movy) {
                            crossings += Curve.pointCrossingsForLine(
                                px, py,
                                curx, cury,
                                movx, movy
                            )
                        }
                        curx = movx
                        cury = movy
                    }
                }
            }
            if (cury != movy) {
                crossings += Curve.pointCrossingsForLine(
                    px, py,
                    curx, cury,
                    movx, movy
                )
            }
            return crossings
        }

        override fun rectCrossings(
            rxmin: kotlin.Double, rymin: kotlin.Double,
            rxmax: kotlin.Double, rymax: kotlin.Double
        ): Int {
            if (numTypes == 0) {
                return 0
            }
            val coords = doubleCoords
            var movx: kotlin.Double
            movx = coords[0]
            var curx = movx
            var movy: kotlin.Double
            movy = coords[1]
            var cury = movy
            var crossings = 0
            var ci = 2
            var i = 1
            while (crossings != Curve.RECT_INTERSECTS && i < numTypes) {
                var endy: kotlin.Double
                var endx: kotlin.Double
                when (pointTypes[i]) {
                    SEG_MOVETO -> {
                        if (curx != movx || cury != movy) {
                            crossings = Curve.rectCrossingsForLine(
                                crossings,
                                rxmin, rymin,
                                rxmax, rymax,
                                curx, cury,
                                movx, movy
                            )
                        }
                        run {
                            curx = coords[ci++]
                            movx = curx
                        }
                        run {
                            cury = coords[ci++]
                            movy = cury
                        }
                    }
                    SEG_LINETO -> {
                        endx = coords[ci++]
                        endy = coords[ci++]
                        crossings = Curve.rectCrossingsForLine(
                            crossings,
                            rxmin, rymin,
                            rxmax, rymax,
                            curx, cury,
                            endx, endy
                        )
                        curx = endx
                        cury = endy
                    }
                    SEG_QUADTO -> {
                        crossings = Curve.rectCrossingsForQuad(
                            crossings,
                            rxmin, rymin,
                            rxmax, rymax,
                            curx, cury,
                            coords[ci++],
                            coords[ci++],
                            coords[ci++].also {
                                endx = it
                            },
                            coords[ci++].also {
                                endy = it
                            },
                            0
                        )
                        curx = endx
                        cury = endy
                    }
                    SEG_CUBICTO -> {
                        crossings = Curve.rectCrossingsForCubic(
                            crossings,
                            rxmin, rymin,
                            rxmax, rymax,
                            curx, cury,
                            coords[ci++],
                            coords[ci++],
                            coords[ci++],
                            coords[ci++],
                            coords[ci++].also {
                                endx = it
                            },
                            coords[ci++].also {
                                endy = it
                            },
                            0
                        )
                        curx = endx
                        cury = endy
                    }
                    SEG_CLOSE -> {
                        if (curx != movx || cury != movy) {
                            crossings = Curve.rectCrossingsForLine(
                                crossings,
                                rxmin, rymin,
                                rxmax, rymax,
                                curx, cury,
                                movx, movy
                            )
                        }
                        curx = movx
                        cury = movy
                    }
                }
                i++
            }
            if (crossings != Curve.RECT_INTERSECTS &&
                (curx != movx || cury != movy)
            ) {
                crossings = Curve.rectCrossingsForLine(
                    crossings,
                    rxmin, rymin,
                    rxmax, rymax,
                    curx, cury,
                    movx, movy
                )
            }
            // Count should always be a multiple of 2 here.
            // assert((crossings & 1) != 0);
            return crossings
        }

        // ToDo: Implement in the context of JSweet
        override fun createTransformedShape(transform: AffineTransform?): Shape? {
            return null
        }

        /**
         * {@inheritDoc}
         *
         * @since 1.6
         */
        @Synchronized
        override fun moveTo(x: kotlin.Double, y: kotlin.Double) {
            if (numTypes > 0 && pointTypes[numTypes - 1] == SEG_MOVETO) {
                doubleCoords[numCoords - 2] = x
                doubleCoords[numCoords - 1] = y
            } else {
                needRoom(false, 2)
                pointTypes[numTypes++] = SEG_MOVETO
                doubleCoords[numCoords++] = x
                doubleCoords[numCoords++] = y
            }
        }

        /**
         * {@inheritDoc}
         *
         * @since 1.6
         */
        @Synchronized
        override fun lineTo(x: kotlin.Double, y: kotlin.Double) {
            needRoom(true, 2)
            pointTypes[numTypes++] = SEG_LINETO
            doubleCoords[numCoords++] = x
            doubleCoords[numCoords++] = y
        }

        /**
         * {@inheritDoc}
         *
         * @since 1.6
         */
        @Synchronized
        override fun quadTo(
            x1: kotlin.Double, y1: kotlin.Double,
            x2: kotlin.Double, y2: kotlin.Double
        ) {
            needRoom(true, 4)
            pointTypes[numTypes++] = SEG_QUADTO
            doubleCoords[numCoords++] = x1
            doubleCoords[numCoords++] = y1
            doubleCoords[numCoords++] = x2
            doubleCoords[numCoords++] = y2
        }

        /**
         * {@inheritDoc}
         *
         * @since 1.6
         */
        @Synchronized
        override fun curveTo(
            x1: kotlin.Double, y1: kotlin.Double,
            x2: kotlin.Double, y2: kotlin.Double,
            x3: kotlin.Double, y3: kotlin.Double
        ) {
            needRoom(true, 6)
            pointTypes[numTypes++] = SEG_CUBICTO
            doubleCoords[numCoords++] = x1
            doubleCoords[numCoords++] = y1
            doubleCoords[numCoords++] = x2
            doubleCoords[numCoords++] = y2
            doubleCoords[numCoords++] = x3
            doubleCoords[numCoords++] = y3
        }

        /**
         * {@inheritDoc}
         *
         * @since 1.6
         */
        override fun append(pi: PathIterator, connect: Boolean) {
            var connect = connect
            val coords = DoubleArray(6)
            while (!pi.isDone) {
                when (pi.currentSegment(coords).toByte()) {
                    SEG_MOVETO -> {
                        if (!connect || numTypes < 1 || numCoords < 1) {
                            moveTo(coords[0], coords[1])
                            break
                        }
                        if (pointTypes[numTypes - 1] != SEG_CLOSE && doubleCoords[numCoords - 2] == coords[0] && doubleCoords[numCoords - 1] == coords[1]) {
                            // Collapse out initial moveto/lineto
                            break
                        }
                        lineTo(coords[0], coords[1])
                    }
                    SEG_LINETO -> lineTo(coords[0], coords[1])
                    SEG_QUADTO -> quadTo(
                        coords[0], coords[1],
                        coords[2], coords[3]
                    )
                    SEG_CUBICTO -> curveTo(
                        coords[0], coords[1],
                        coords[2], coords[3],
                        coords[4], coords[5]
                    )
                    SEG_CLOSE -> closePath()
                }
                pi.next()
                connect = false
            }
        }

        fun transform(at: AffineTransform) {
            at.transform(doubleCoords, 0, doubleCoords, 0, numCoords / 2)
        }

        override val bounds2D: Rectangle2D
            get() = computeBounds2D()

        /**
         * {@inheritDoc}
         *
         * @since 1.6
         */
        @Synchronized
        fun computeBounds2D(): Rectangle2D {
            var x1: kotlin.Double
            var y1: kotlin.Double
            var x2: kotlin.Double
            var y2: kotlin.Double
            var i = numCoords
            if (i > 0) {
                y2 = doubleCoords[--i]
                y1 = y2
                x2 = doubleCoords[--i]
                x1 = x2
                while (i > 0) {
                    val y = doubleCoords[--i]
                    val x = doubleCoords[--i]
                    if (x < x1) x1 = x
                    if (y < y1) y1 = y
                    if (x > x2) x2 = x
                    if (y > y2) y2 = y
                }
            } else {
                y2 = 0.0
                x2 = y2
                y1 = x2
                x1 = y1
            }
            return Rectangle2D.Double(x1, y1, x2 - x1, y2 - y1)
        }

        override val pathIterator: PathIterator
            get() = getPathIterator(null)

        /**
         * {@inheritDoc}
         *
         *
         * The iterator for this class is not multi-threaded safe,
         * which means that the `Path2D` class does not
         * guarantee that modifications to the geometry of this
         * `Path2D` object do not affect any iterations of
         * that geometry that are already in process.
         *
         * @return a new `PathIterator` that iterates along the boundary
         * of this `Shape` and provides access to the geometry
         * of this `Shape`'s outline
         *
         * @since 1.6
         */
        override fun getPathIterator(at: AffineTransform?): PathIterator {
            return if (at == null) {
                CopyIterator(this)
            } else {
                TxIterator(this, at)
            }
        }

        override fun getFlattenPathIterator(flatness: kotlin.Double): PathIterator {
            return FlatteningPathIterator(pathIterator, flatness)
        }

        internal class CopyIterator(p2dd: Double) : Iterator(p2dd) {
            val doubleCoords: DoubleArray
            override fun currentSegment(coords: FloatArray): Int {
                val type = path.pointTypes[typeIdx].toInt()
                val numCoords = curvecoords[type]
                if (numCoords > 0) {
                    for (i in 0 until numCoords) {
                        coords[i] = doubleCoords[pointIdx + i].toFloat()
                    }
                }
                return type
            }

            override fun currentSegment(coords: DoubleArray): Int {
                val type = path.pointTypes[typeIdx].toInt()
                val numCoords = curvecoords[type]
                if (numCoords > 0) {
                    for (i in 0 until numCoords) {
                        coords[i] = doubleCoords[pointIdx + i]
                    }
                }
                return type
            }

            init {
                doubleCoords = p2dd.doubleCoords
            }
        }

        internal class TxIterator(p2dd: Double, at: AffineTransform) : Iterator(p2dd) {
            val doubleCoords: DoubleArray
            val affine: AffineTransform
            override fun currentSegment(coords: FloatArray): Int {
                val type = path.pointTypes[typeIdx].toInt()
                val numCoords = curvecoords[type]
                if (numCoords > 0) {
                    affine.transform(
                        doubleCoords, pointIdx,
                        coords, 0, numCoords / 2
                    )
                }
                return type
            }

            override fun currentSegment(coords: DoubleArray): Int {
                val type = path.pointTypes[typeIdx].toInt()
                val numCoords = curvecoords[type]
                if (numCoords > 0) {
                    affine.transform(
                        doubleCoords, pointIdx,
                        coords, 0, numCoords / 2
                    )
                }
                return type
            }

            init {
                doubleCoords = p2dd.doubleCoords
                affine = at
            }
        }
    }

    internal abstract class Iterator(val path: Path2D) : PathIterator {
        var typeIdx = 0
        var pointIdx = 0
        override val windingRule: Int
            get() = path.windingRule

        override val isDone: Boolean
            get() = typeIdx >= path.numTypes

        override operator fun next() {
            val type = path.pointTypes[typeIdx++].toInt()
            pointIdx += curvecoords[type]
        }

        companion object {
            val curvecoords = intArrayOf(2, 2, 4, 6, 0)
        }
    }

    companion object {
        /**
         * An even-odd winding rule for determining the interior of
         * a path.
         *
         * @see PathIterator.WIND_EVEN_ODD
         *
         * @since 1.6
         */
        val WIND_EVEN_ODD: Int = PathIterator.WIND_EVEN_ODD

        /**
         * A non-zero winding rule for determining the interior of a
         * path.
         *
         * @see PathIterator.WIND_NON_ZERO
         *
         * @since 1.6
         */
        val WIND_NON_ZERO: Int = PathIterator.WIND_NON_ZERO
        const val INIT_SIZE = 20
        const val EXPAND_MAX = 500

        // For code simplicity, copy these constants to our namespace
        // and cast them to byte constants for easy storage.
        private val SEG_MOVETO = PathIterator.SEG_MOVETO as Byte
        private val SEG_LINETO = PathIterator.SEG_LINETO as Byte
        private val SEG_QUADTO = PathIterator.SEG_QUADTO as Byte
        private val SEG_CUBICTO = PathIterator.SEG_CUBICTO as Byte
        private val SEG_CLOSE = PathIterator.SEG_CLOSE as Byte

        /**
         * Tests if the specified [Point2D] is inside the closed
         * boundary of the specified [PathIterator].
         *
         *
         * This method provides a basic facility for implementors of
         * the [Shape] interface to implement support for the
         * [Shape.contains] method.
         *
         * @param pi the specified `PathIterator`
         * @param p  the specified `Point2D`
         *
         * @return `true` if the specified coordinates are inside the
         * specified `PathIterator`; `false` otherwise
         *
         * @since 1.6
         */
        fun contains(pi: PathIterator, p: Point2D): Boolean {
            return contains(pi, p.x, p.y)
        }

        /**
         * Tests if the specified coordinates are inside the closed
         * boundary of the specified [PathIterator].
         *
         *
         * This method provides a basic facility for implementors of
         * the [Shape] interface to implement support for the
         * [Shape.contains] method.
         *
         * @param pi the specified `PathIterator`
         * @param x  the specified X coordinate
         * @param y  the specified Y coordinate
         *
         * @return `true` if the specified coordinates are inside the
         * specified `PathIterator`; `false` otherwise
         *
         * @since 1.6
         */
        fun contains(pi: PathIterator, x: kotlin.Double, y: kotlin.Double): Boolean {
            return if (x * 0.0 + y * 0.0 == 0.0) {
                /* N * 0.0 is 0.0 only if N is finite.
             * Here we know that both x and y are finite.
             */
                val mask = if (pi.windingRule === WIND_NON_ZERO) -1 else 1
                val cross: Int = Curve.pointCrossingsForPath(pi, x, y)
                cross and mask != 0
            } else {
                /* Either x or y was infinite or NaN.
             * A NaN always produces a negative response to any test
             * and Infinity values cannot be "inside" any path so
             * they should return false as well.
             */
                false
            }
        }

        /**
         * Tests if the specified [Rectangle2D] is entirely inside the
         * closed boundary of the specified [PathIterator].
         *
         *
         * This method provides a basic facility for implementors of
         * the [Shape] interface to implement support for the
         * [Shape.contains] method.
         *
         *
         * This method object may conservatively return false in
         * cases where the specified rectangular area intersects a
         * segment of the path, but that segment does not represent a
         * boundary between the interior and exterior of the path.
         * Such segments could lie entirely within the interior of the
         * path if they are part of a path with a [.WIND_NON_ZERO]
         * winding rule or if the segments are retraced in the reverse
         * direction such that the two sets of segments cancel each
         * other out without any exterior area falling between them.
         * To determine whether segments represent true boundaries of
         * the interior of the path would require extensive calculations
         * involving all of the segments of the path and the winding
         * rule and are thus beyond the scope of this implementation.
         *
         * @param pi the specified `PathIterator`
         * @param r  a specified `Rectangle2D`
         *
         * @return `true` if the specified `PathIterator` contains
         * the specified `Rectangle2D`; `false` otherwise.
         *
         * @since 1.6
         */
        fun contains(pi: PathIterator, r: Rectangle2D): Boolean {
            return contains(pi, r.x, r.y, r.width, r.height)
        }

        /**
         * Tests if the specified rectangular area is entirely inside the
         * closed boundary of the specified [PathIterator].
         *
         *
         * This method provides a basic facility for implementors of
         * the [Shape] interface to implement support for the
         * [Shape.contains] method.
         *
         *
         * This method object may conservatively return false in
         * cases where the specified rectangular area intersects a
         * segment of the path, but that segment does not represent a
         * boundary between the interior and exterior of the path.
         * Such segments could lie entirely within the interior of the
         * path if they are part of a path with a [.WIND_NON_ZERO]
         * winding rule or if the segments are retraced in the reverse
         * direction such that the two sets of segments cancel each
         * other out without any exterior area falling between them.
         * To determine whether segments represent true boundaries of
         * the interior of the path would require extensive calculations
         * involving all of the segments of the path and the winding
         * rule and are thus beyond the scope of this implementation.
         *
         * @param pi the specified `PathIterator`
         * @param x  the specified X coordinate
         * @param y  the specified Y coordinate
         * @param w  the width of the specified rectangular area
         * @param h  the height of the specified rectangular area
         *
         * @return `true` if the specified `PathIterator` contains
         * the specified rectangular area; `false` otherwise.
         *
         * @since 1.6
         */
        fun contains(
            pi: PathIterator,
            x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double
        ): Boolean {
            if ((x + w).isNaN() || (y + h).isNaN()) {
                /* [xy]+[wh] is NaN if any of those values are NaN,
             * or if adding the two together would produce NaN
             * by virtue of adding opposing Infinte values.
             * Since we need to add them below, their sum must
             * not be NaN.
             * We return false because NaN always produces a
             * negative response to tests
             */
                return false
            }
            if (w <= 0 || h <= 0) {
                return false
            }
            val mask = if (pi.windingRule === WIND_NON_ZERO) -1 else 2
            val crossings: Int = Curve.rectCrossingsForPath(pi, x, y, x + w, y + h)
            return crossings != Curve.RECT_INTERSECTS &&
                    crossings and mask != 0
        }

        /**
         * Tests if the interior of the specified [PathIterator]
         * intersects the interior of a specified [Rectangle2D].
         *
         *
         * This method provides a basic facility for implementors of
         * the [Shape] interface to implement support for the
         * [Shape.intersects] method.
         *
         *
         * This method object may conservatively return true in
         * cases where the specified rectangular area intersects a
         * segment of the path, but that segment does not represent a
         * boundary between the interior and exterior of the path.
         * Such a case may occur if some set of segments of the
         * path are retraced in the reverse direction such that the
         * two sets of segments cancel each other out without any
         * interior area between them.
         * To determine whether segments represent true boundaries of
         * the interior of the path would require extensive calculations
         * involving all of the segments of the path and the winding
         * rule and are thus beyond the scope of this implementation.
         *
         * @param pi the specified `PathIterator`
         * @param r  the specified `Rectangle2D`
         *
         * @return `true` if the specified `PathIterator` and
         * the interior of the specified `Rectangle2D`
         * intersect each other; `false` otherwise.
         *
         * @since 1.6
         */
        fun intersects(pi: PathIterator, r: Rectangle2D): Boolean {
            return intersects(pi, r.x, r.y, r.width, r.height)
        }

        /**
         * Tests if the interior of the specified [PathIterator]
         * intersects the interior of a specified set of rectangular
         * coordinates.
         *
         *
         * This method provides a basic facility for implementors of
         * the [Shape] interface to implement support for the
         * [Shape.intersects] method.
         *
         *
         * This method object may conservatively return true in
         * cases where the specified rectangular area intersects a
         * segment of the path, but that segment does not represent a
         * boundary between the interior and exterior of the path.
         * Such a case may occur if some set of segments of the
         * path are retraced in the reverse direction such that the
         * two sets of segments cancel each other out without any
         * interior area between them.
         * To determine whether segments represent true boundaries of
         * the interior of the path would require extensive calculations
         * involving all of the segments of the path and the winding
         * rule and are thus beyond the scope of this implementation.
         *
         * @param pi the specified `PathIterator`
         * @param x  the specified X coordinate
         * @param y  the specified Y coordinate
         * @param w  the width of the specified rectangular coordinates
         * @param h  the height of the specified rectangular coordinates
         *
         * @return `true` if the specified `PathIterator` and
         * the interior of the specified set of rectangular
         * coordinates intersect each other; `false` otherwise.
         *
         * @since 1.6
         */
        fun intersects(
            pi: PathIterator,
            x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double
        ): Boolean {
            if ((x + w).isNaN() || (y + h).isNaN()) {
                /* [xy]+[wh] is NaN if any of those values are NaN,
             * or if adding the two together would produce NaN
             * by virtue of adding opposing Infinte values.
             * Since we need to add them below, their sum must
             * not be NaN.
             * We return false because NaN always produces a
             * negative response to tests
             */
                return false
            }
            if (w <= 0 || h <= 0) {
                return false
            }
            val mask = if (pi.windingRule === WIND_NON_ZERO) -1 else 2
            val crossings: Int = Curve.rectCrossingsForPath(pi, x, y, x + w, y + h)
            return crossings == Curve.RECT_INTERSECTS ||
                    crossings and mask != 0
        }

        fun copyOf(original: ByteArray, newLength: Int): ByteArray {
            val copy = ByteArray(newLength)
            arraycopy(
                original, 0, copy, 0,
                min(original.size, newLength)
            )
            return copy
        }

        fun copyOf(original: IntArray, newLength: Int): IntArray {
            val copy = IntArray(newLength)
            arraycopy(
                original, 0, copy, 0,
                min(original.size, newLength)
            )
            return copy
        }

        /**
         * Copies the specified array, truncating or padding with zeros (if necessary)
         * so the copy has the specified length.  For all indices that are
         * valid in both the original array and the copy, the two arrays will
         * contain identical values.  For any indices that are valid in the
         * copy but not the original, the copy will contain <tt>0d</tt>.
         * Such indices will exist if and only if the specified length
         * is greater than that of the original array.
         *
         * @param original  the array to be copied
         * @param newLength the length of the copy to be returned
         *
         * @return a copy of the original array, truncated or padded with zeros
         * to obtain the specified length
         *
         * @throws NegativeArraySizeException if <tt>newLength</tt> is negative
         * @throws NullPointerException       if <tt>original</tt> is null
         * @since 1.6
         */
        fun copyOf(original: DoubleArray, newLength: Int): DoubleArray {
            val copy = DoubleArray(newLength)
            arraycopy(
                original, 0, copy, 0,
                min(original.size, newLength)
            )
            return copy
        }
    }
}