/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

import kotlin.math.max
import kotlin.math.min

interface Rectangle2D : Shape {
    val x: kotlin.Double
    val y: kotlin.Double
    val width: kotlin.Double
    val height: kotlin.Double
    val minX: kotlin.Double
    val minY: kotlin.Double
    val maxX: kotlin.Double
    val maxY: kotlin.Double
    val centerX: kotlin.Double
    val centerY: kotlin.Double
    fun intersectsLine(x1: kotlin.Double, y1: kotlin.Double, x2: kotlin.Double, y2: kotlin.Double): Boolean
    fun createIntersection(r: Rectangle2D): Rectangle2D
    fun createUnion(r: Rectangle2D): Rectangle2D
    fun normalize(): Rectangle2D
    fun add(x: kotlin.Double, y: kotlin.Double): Rectangle2D
    fun intersects(x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double): Boolean
    fun contains(x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double): Boolean
    fun contains(x: kotlin.Double, y: kotlin.Double): Boolean
    open class Double(
        override val x: kotlin.Double,
        override val y: kotlin.Double,
        override val width: kotlin.Double,
        override val height: kotlin.Double
    ) : AbstractRectangle2D() {

        /**
         * Adds a `Rectangle2D` object to this
         * `Rectangle2D`.  The resulting `Rectangle2D`
         * is the union of the two `Rectangle2D` objects.
         *
         * @param r the `Rectangle2D` to add to this
         * `Rectangle2D`.
         *
         * @since 1.2
         */
        fun add(r: Rectangle2D): Double {
            val x1: kotlin.Double = min(minX, r.minX)
            val x2: kotlin.Double = max(maxX, r.maxX)
            val y1: kotlin.Double = min(minY, r.minY)
            val y2: kotlin.Double = max(maxY, r.maxY)
            return Double(x1, y1, x2 - x1, y2 - y1)
        }

        override fun add(x: kotlin.Double, y: kotlin.Double): Rectangle2D {
            val x1: kotlin.Double = min(minX, x)
            val x2: kotlin.Double = max(maxX, x)
            val y1: kotlin.Double = min(minY, y)
            val y2: kotlin.Double = max(maxY, y)
            return Double(x1, y1, x2 - x1, y2 - y1)
        }

        override fun toString(): String {
            return "Rectangle2D.Double{" +
                    "x=" + x +
                    ", y=" + y +
                    ", width=" + width +
                    ", height=" + height +
                    '}'
        }

        /**
         * Normalizes this rectangle to have a positive width and height,
         * adjusting the reference position *P1* if necessary.
         *
         * @return a new rectangle or this rectangle
         */
        override fun normalize(): Double {
            // quick check if anything needs to be done
            if (isNormalized) return this
            var normalizedX = x
            var normalizedWidth = width
            if (normalizedWidth < 0) {
                normalizedWidth = -normalizedWidth
                normalizedX -= normalizedWidth
            }
            var normalizedHeight = height
            var normalizedY = y
            if (normalizedHeight < 0) {
                normalizedHeight = -normalizedHeight
                normalizedY -= normalizedHeight
            }
            return Double(
                normalizedX, normalizedY, normalizedWidth,
                normalizedHeight
            )
        }

        /**
         * Returns true if both width and height are positive.
         *
         * @return true if normalized
         */
        private val isNormalized: Boolean
            private get() = width >= 0 && height >= 0

        override fun intersects(x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double): Boolean {
            return intersects(Double(x, y, w, h))
        }

        override fun contains(x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double): Boolean {
            if (isEmpty || w <= 0 || h <= 0) {
                return false
            }
            val x0 = x
            val y0 = y
            return x >= x0 && y >= y0 && x + w <= x0 + width && y + h <= y0 + height
        }

        override fun contains(x: kotlin.Double, y: kotlin.Double): Boolean {
            return contains(Point2D.Double(x, y))
        }

        val isEmpty: Boolean
            get() = width <= 0.0 || height <= 0.0

        companion object {
            fun frameFromDiagonal(
                x1: kotlin.Double, y1: kotlin.Double,
                x2: kotlin.Double, y2: kotlin.Double
            ): Double {
                var x1 = x1
                var y1 = y1
                var x2 = x2
                var y2 = y2
                if (x2 < x1) {
                    val t = x1
                    x1 = x2
                    x2 = t
                }
                if (y2 < y1) {
                    val t = y1
                    y1 = y2
                    y2 = t
                }
                return Double(x1, y1, x2 - x1, y2 - y1)
            }
        }
    }
}