package org.kamaeleo.geom.curve

/**
 *
 * General implementation of the Non-Uniform Rational B-spline or NURB-Spline.  The main advantage
 * of the NURB-Spline over the B-Spline is the ability to represent conic sections.  To do this, a curve
 * with degree of 2 is used.  Figure 1 contains examples of conic arcs.
 *
 *
 *
 * <center><img align="center" src="doc-files/nurbs1.gif"></img></center>
 *
 *
 *
 * NURB-Splines can also be used to generate circles as shown in figure 2.
 *
 *
 *
 * <center><img align="center" src="doc-files/nurbs2.gif"></img></center>
 *
 *
 *
 * As seen in the figures, every control-point has an associated weight value.  The weight-values control
 * how much relative pull each control-point has on the curve.  If the weight-value is 0, then the associated
 * point will have no affect on the curve.  If a point has an associated weight of 0, but the curve is expected
 * to pass through that point, then it is likely that the result of evaluation will be the origin.  All weights
 * must be &gt;= 0.
 */
class NURBSpline(cp: ControlPath?, gi: GroupIterator?) : BSpline(cp, gi) {
    private var weightVector: ValueVector? = ValueVector(doubleArrayOf(1.0, 1.0, 1.0, 1.0), 4)
    /**
     * Returns the value of the useWeightVector flag.  The default value is true.
     *
     * @see .setUseWeightVector
     */
    /**
     * Sets the value of the useWeightVector flag.  If the flag is true, then the internal weightVector
     * will be used.  If the flag is false, then all weights will be assumed to be 1.
     *
     * @see .getUseWeightVector
     */
    var useWeightVector = true

    /**
     * Returns the weight-vector.
     *
     * @see .setWeightVector
     */
    fun getWeightVector(): ValueVector? {
        return weightVector
    }

    /**
     * Sets the weight-vector.
     *
     * @see .getWeightVector
     */
    fun setWeightVector(v: ValueVector?) {
        weightVector = v
    }

    /**
     * The requirements of the appendTo method include the requirements of the BSpline appendTo method, plus
     * a couple more.  The additional requirements only apply if the useWeightVector flag is true.  If so, then
     * the weight-vector must have size equal to the group-size of the GroupIterator and all weights must have a
     * value &gt;= 0.  This method returns quietly if these requirements are not met.
     *
     * @see BSpline.appendTo
     */
    override fun appendTo(mp: MultiPath) {
        if (!gi!!.isInRange(0, cp!!.numPoints())) return
        val numPts: Int = gi!!.groupSize
        if (nw.size < numPts) {
            nw = DoubleArray(2 * numPts)
            weight = DoubleArray(2 * numPts)
        }
        if (useWeightVector) {
            if (weightVector!!.size() !== numPts) return
            for (i in 0 until numPts) {
                weight[i] = weightVector!!.get(i)
                if (weight[i] < 0) return
            }
        } else {
            for (i in 0 until numPts) weight[i] = 1.0
        }
        super.appendTo(mp)
    }

    override fun eval(p: DoubleArray) {
        val dim = p.size - 1
        val t = p[dim]
        var sum2 = 0.0
        val numPts: Int = gi!!.groupSize
        for (i in 0 until numPts) {
            nw[i] = N(t, i) * weight[i]
            sum2 += nw[i]
        }
        if (sum2 == 0.0) sum2 = 1.0
        for (i in 0 until dim) {
            gi!!.set(0, 0)
            var sum1 = 0.0
            for (j in 0 until numPts) sum1 += nw[j] * cp!!.getPoint(
                gi!!.next()
            )!!.location.get(i)
            p[i] = sum1 / sum2
        }
    }

    override fun resetMemory() {
        super.resetMemory()
        if (nw.size > 0) {
            nw = DoubleArray(0)
            weight = DoubleArray(0)
        }
    }

    companion object {
        private var nw = DoubleArray(0) // (required length >= numPts)
        private var weight = DoubleArray(0) // (required length >= numPts)
    }
}