/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

import com.macrofocus.common.collection.arraycopy
import org.kamaeleo.geom.PathIterator.Companion.SEG_CLOSE
import org.kamaeleo.geom.PathIterator.Companion.SEG_CUBICTO
import org.kamaeleo.geom.PathIterator.Companion.SEG_LINETO
import org.kamaeleo.geom.PathIterator.Companion.SEG_MOVETO
import org.kamaeleo.geom.PathIterator.Companion.SEG_QUADTO
import kotlin.math.sqrt

/**
 * The `FlatteningPathIterator` class returns a flattened view of
 * another [PathIterator] object.  Other [Shape][java.awt.Shape]
 * classes can use this class to provide flattening behavior for their paths
 * without having to perform the interpolation calculations themselves.
 *
 * @author Jim Graham
 */
class FlatteningPathIterator(
    src: PathIterator, flatness: Double,
    limit: Int
) : PathIterator {
    var src // The source iterator
            : PathIterator
    var squareflat // Square of the flatness parameter
            : Double

    /**
     * Returns the recursion limit of this iterator.
     *
     * @return the recursion limit of this
     * `FlatteningPathIterator`.
     */
    // for testing against squared lengths
    var recursionLimit // Maximum number of recursion levels
            : Int
    var hold = DoubleArray(14) // The cache of interpolated coords

    // Note that this must be long enough
    // to store a full cubic segment and
    // a relative cubic segment to avoid
    // aliasing when copying the coords
    // of a curve to the end of the array.
    // This is also serendipitously equal
    // to the size of a full quad segment
    // and 2 relative quad segments.
    var curx = 0.0
    var cury // The ending x,y of the last segment
            = 0.0
    var movx = 0.0
    var movy // The x,y of the last move segment
            = 0.0
    var holdType // The type of the curve being held
            = 0

    // for interpolation
    var holdEnd // The index of the last curve segment
            = 0

    // being held for interpolation
    var holdIndex // The index of the curve segment
            = 0

    // that was last interpolated.  This
    // is the curve segment ready to be
    // returned in the next call to
    // currentSegment().
    var levels // The recursion level at which
            : IntArray

    // each curve being held in storage
    // was generated.
    var levelIndex // The index of the entry in the
            = 0

    /**
     * Tests if the iteration is complete.
     *
     * @return `true` if all the segments have
     * been read; `false` otherwise.
     */
    // levels array of the curve segment
    // at the holdIndex
    override var isDone // True when iteration is done
            = false

    /**
     * Constructs a new `FlatteningPathIterator` object that
     * flattens a path as it iterates over it.  The iterator does not
     * subdivide any curve read from the source iterator to more than
     * 10 levels of subdivision which yields a maximum of 1024 line
     * segments per curve.
     *
     * @param src      the original unflattened path being iterated over
     * @param flatness the maximum allowable distance between the
     * control points and the flattened curve
     */
    constructor(src: PathIterator, flatness: Double) : this(src, flatness, 10) {}

    private fun next(doNext: Boolean) {
        if (holdIndex >= holdEnd) {
            if (doNext) {
                src.next()
            }
            if (src.isDone) {
                isDone = true
                return
            }
            holdType = src.currentSegment(hold)
            levelIndex = 0
            levels[0] = 0
        }
        var level: Int
        when (holdType) {
            SEG_MOVETO, SEG_LINETO -> {
                curx = hold[0]
                cury = hold[1]
                if (holdType == SEG_MOVETO) {
                    movx = curx
                    movy = cury
                }
                holdIndex = 0
                holdEnd = 0
            }
            SEG_CLOSE -> {
                curx = movx
                cury = movy
                holdIndex = 0
                holdEnd = 0
            }
            SEG_QUADTO -> {
                if (holdIndex >= holdEnd) {
                    // Move the coordinates to the end of the array.
                    holdIndex = hold.size - 6
                    holdEnd = hold.size - 2
                    hold[holdIndex + 0] = curx
                    hold[holdIndex + 1] = cury
                    hold[holdIndex + 2] = hold[0]
                    hold[holdIndex + 3] = hold[1]
                    curx = hold[2]
                    hold[holdIndex + 4] = curx
                    cury = hold[3]
                    hold[holdIndex + 5] = cury
                }
                level = levels[levelIndex]
                while (level < recursionLimit) {
                    if (QuadCurve2D.getFlatnessSq(hold, holdIndex) < squareflat) {
                        break
                    }
                    ensureHoldCapacity(4)
                    QuadCurve2D.subdivide(
                        hold, holdIndex,
                        hold, holdIndex - 4,
                        hold, holdIndex
                    )
                    holdIndex -= 4

                    // Now that we have subdivided, we have constructed
                    // two curves of one depth lower than the original
                    // curve.  One of those curves is in the place of
                    // the former curve and one of them is in the next
                    // set of held coordinate slots.  We now set both
                    // curves level values to the next higher level.
                    level++
                    levels[levelIndex] = level
                    levelIndex++
                    levels[levelIndex] = level
                }

                // This curve segment is flat enough, or it is too deep
                // in recursion levels to try to flatten any more.  The
                // two coordinates at holdIndex+4 and holdIndex+5 now
                // contain the endpoint of the curve which can be the
                // endpoint of an approximating line segment.
                holdIndex += 4
                levelIndex--
            }
            SEG_CUBICTO -> {
                if (holdIndex >= holdEnd) {
                    // Move the coordinates to the end of the array.
                    holdIndex = hold.size - 8
                    holdEnd = hold.size - 2
                    hold[holdIndex + 0] = curx
                    hold[holdIndex + 1] = cury
                    hold[holdIndex + 2] = hold[0]
                    hold[holdIndex + 3] = hold[1]
                    hold[holdIndex + 4] = hold[2]
                    hold[holdIndex + 5] = hold[3]
                    curx = hold[4]
                    hold[holdIndex + 6] = curx
                    cury = hold[5]
                    hold[holdIndex + 7] = cury
                }
                level = levels[levelIndex]
                while (level < recursionLimit) {
                    if (CubicCurve2D.getFlatnessSq(hold, holdIndex) < squareflat) {
                        break
                    }
                    ensureHoldCapacity(6)
                    CubicCurve2D.subdivide(
                        hold, holdIndex,
                        hold, holdIndex - 6,
                        hold, holdIndex
                    )
                    holdIndex -= 6

                    // Now that we have subdivided, we have constructed
                    // two curves of one depth lower than the original
                    // curve.  One of those curves is in the place of
                    // the former curve and one of them is in the next
                    // set of held coordinate slots.  We now set both
                    // curves level values to the next higher level.
                    level++
                    levels[levelIndex] = level
                    levelIndex++
                    levels[levelIndex] = level
                }

                // This curve segment is flat enough, or it is too deep
                // in recursion levels to try to flatten any more.  The
                // two coordinates at holdIndex+6 and holdIndex+7 now
                // contain the endpoint of the curve which can be the
                // endpoint of an approximating line segment.
                holdIndex += 6
                levelIndex--
            }
        }
    }

    /*
     * Ensures that the hold array can hold up to (want) more values.
     * It is currently holding (hold.length - holdIndex) values.
     */
    fun ensureHoldCapacity(want: Int) {
        if (holdIndex - want < 0) {
            val have = hold.size - holdIndex
            val newsize = hold.size + GROW_SIZE
            val newhold = DoubleArray(newsize)
            arraycopy(
                hold, holdIndex,
                newhold, holdIndex + GROW_SIZE,
                have
            )
            hold = newhold
            holdIndex += GROW_SIZE
            holdEnd += GROW_SIZE
        }
    }

    /**
     * Returns the flatness of this iterator.
     *
     * @return the flatness of this `FlatteningPathIterator`.
     */
    val flatness: Double
        get() = sqrt(squareflat)

    /**
     * Returns the winding rule for determining the interior of the
     * path.
     *
     * @return the winding rule of the original unflattened path being
     * iterated over.
     *
     * @see PathIterator.WIND_EVEN_ODD
     *
     * @see PathIterator.WIND_NON_ZERO
     */
    override val windingRule: Int
        get() = src.windingRule

    /**
     * Moves the iterator to the next segment of the path forwards
     * along the primary direction of traversal as long as there are
     * more points in that direction.
     */
    override operator fun next() {
        next(true)
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, or SEG_CLOSE.
     * A float array of length 6 must be passed in and can be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of float x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types return one point,
     * and SEG_CLOSE does not return any points.
     *
     * @param coords an array that holds the data returned from
     * this method
     *
     * @return the path segment type of the current path segment.
     *
     * @throws NoSuchElementException if there
     * are no more elements in the flattening path to be
     * returned.
     * @see PathIterator.SEG_MOVETO
     *
     * @see PathIterator.SEG_LINETO
     *
     * @see PathIterator.SEG_CLOSE
     */
    override fun currentSegment(coords: FloatArray): Int {
        if (isDone) {
            throw NoSuchElementException("flattening iterator out of bounds")
        }
        var type = holdType
        if (type != SEG_CLOSE) {
            coords[0] = hold[holdIndex + 0].toFloat()
            coords[1] = hold[holdIndex + 1].toFloat()
            if (type != SEG_MOVETO) {
                type = SEG_LINETO
            }
        }
        return type
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, or SEG_CLOSE.
     * A double array of length 6 must be passed in and can be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of double x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types return one point,
     * and SEG_CLOSE does not return any points.
     *
     * @param coords an array that holds the data returned from
     * this method
     *
     * @return the path segment type of the current path segment.
     *
     * @throws NoSuchElementException if there
     * are no more elements in the flattening path to be
     * returned.
     * @see PathIterator.SEG_MOVETO
     *
     * @see PathIterator.SEG_LINETO
     *
     * @see PathIterator.SEG_CLOSE
     */
    override fun currentSegment(coords: DoubleArray): Int {
        if (isDone) {
            throw NoSuchElementException("flattening iterator out of bounds")
        }
        var type = holdType
        if (type != SEG_CLOSE) {
            coords[0] = hold[holdIndex + 0]
            coords[1] = hold[holdIndex + 1]
            if (type != SEG_MOVETO) {
                type = SEG_LINETO
            }
        }
        return type
    }

    companion object {
        const val GROW_SIZE = 24 // Multiple of cubic & quad curve size
    }

    /**
     * Constructs a new `FlatteningPathIterator` object
     * that flattens a path as it iterates over it.
     * The `limit` parameter allows you to control the
     * maximum number of recursive subdivisions that the iterator
     * can make before it assumes that the curve is flat enough
     * without measuring against the `flatness` parameter.
     * The flattened iteration therefore never generates more than
     * a maximum of `(2^limit)` line segments per curve.
     *
     * @param src      the original unflattened path being iterated over
     * @param flatness the maximum allowable distance between the
     * control points and the flattened curve
     * @param limit    the maximum number of recursive subdivisions
     * allowed for any curved segment
     *
     * @throws IllegalArgumentException if
     * `flatness` or `limit`
     * is less than zero
     */
    init {
        require(flatness >= 0.0) { "flatness must be >= 0" }
        require(limit >= 0) { "limit must be >= 0" }
        this.src = src
        squareflat = flatness * flatness
        recursionLimit = limit
        levels = IntArray(limit + 1)
        // prime the first path segment
        next(false)
    }
}