/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

import kotlin.math.sqrt

interface Point2D {
    val x: kotlin.Double
    val y: kotlin.Double
    fun distance(e: Point2D): kotlin.Double
    fun distanceSq(e: Point2D): kotlin.Double

    /**
     * Sets the location of this `Point2D` to the
     * specified `double` coordinates.
     *
     * @param x the new X coordinate of this `Point2D`
     * @param y the new Y coordinate of this `Point2D`
     *
     * @since 1.2
     */
    fun setLocation(x: kotlin.Double, y: kotlin.Double)

    class Double : Point2D {
        override var x = 0.0
        override var y = 0.0

        /**
         * Constructs and initializes a `Point2D` with
         * coordinates (0,&nbsp;0).
         *
         * @since 1.2
         */
        constructor() {}
        constructor(x: kotlin.Double, y: kotlin.Double) {
            this.x = x
            this.y = y
        }

        override fun toString(): String {
            return "Point.Double{" +
                    "x=" + x +
                    ", y=" + y +
                    '}'
        }

        /**
         * {@inheritDoc}
         *
         * @since 1.2
         */
        override fun setLocation(x: kotlin.Double, y: kotlin.Double) {
            this.x = x
            this.y = y
        }

        /**
         * Returns the distance from this `Point2D` to a
         * specified `Point2D`.
         *
         * @param pt the specified point to be measured
         * against this `Point2D`
         *
         * @return the distance between this `Point2D` and
         * the specified `Point2D`.
         */
        override fun distance(pt: Point2D): kotlin.Double {
            val px = pt.x - x
            val py = pt.y - y
            return sqrt(px * px + py * py)
        }

        override fun distanceSq(e: Point2D): kotlin.Double {
            val px = e.x - x
            val py = e.y - y
            return px * px + py * py
        }
    }

    class Float : Point2D {
        var fx = 0f
        var fy = 0f

        /**
         * Constructs and initializes a `Point2D` with
         * coordinates (0,&nbsp;0).
         *
         * @since 1.2
         */
        constructor() {}
        constructor(x: kotlin.Float, y: kotlin.Float) {
            this.fx = x
            this.fy = y
        }

        override val x: kotlin.Double
            get() = fx.toDouble()

        override val y: kotlin.Double
            get() = fy.toDouble()

        override fun toString(): String {
            return "Point.Float{" +
                    "x=" + fx +
                    ", y=" + fy +
                    '}'
        }

//        override fun getX(): kotlin.Double {
//            return x.toDouble()
//        }

//        override fun y: kotlin.Double {
//            return y.toDouble()
//        }

        /**
         * {@inheritDoc}
         *
         * @since 1.2
         */
        override fun setLocation(x: kotlin.Double, y: kotlin.Double) {
            this.fx = x.toFloat()
            this.fy = y.toFloat()
        }

        /**
         * Returns the distance from this `Point2D` to a
         * specified `Point2D`.
         *
         * @param pt the specified point to be measured
         * against this `Point2D`
         *
         * @return the distance between this `Point2D` and
         * the specified `Point2D`.
         */
        override fun distance(pt: Point2D): kotlin.Double {
            val px = pt.x - x
            val py = pt.y - y
            return sqrt(px * px + py * py)
        }

        override fun distanceSq(e: Point2D): kotlin.Double {
            val px = e.x - x
            val py = e.y - y
            return px * px + py * py
        }
    }

    companion object {
        /**
         * Computes the euclidean distance between (x1,y1) and (x2,y2)
         *
         * @param x1
         * @param y1
         * @param x2
         * @param y2
         *
         * @return euclidean distance between (x1,y1) and (x2,y2)
         */
        fun distance(
            x1: kotlin.Double, y1: kotlin.Double,
            x2: kotlin.Double, y2: kotlin.Double
        ): kotlin.Double {
            var x1 = x1
            var y1 = y1
            x1 -= x2
            y1 -= y2
            return sqrt(x1 * x1 + y1 * y1)
        }
    }
}