package org.kamaeleo.geom.curve

/**
 *
 * General n-point Bezier curve implementation.  The Bezier curve defines itself using all the points
 * from the control-path specified by the group-iterator.  To compute a single point on the curve requires
 * O(n) multiplications where n is the group-size of the group-iterator.  Thus, the Bezier curve is
 * considered to be expensive, but it has several mathematical properties (not discussed here) that
 * make it appealing.  Figure 1 shows an example of a Bezier curve.
 *
 *
 *
 * <center><img align="center" src="doc-files/bezier1.gif"></img></center>
 *
 *
 *
 * The maximum number of points that the Bezier curve can use is 1030 because the evaluation of a point
 * uses the nCr (n-choose-r) function.  The computation uses double precision, and double precision cannot
 * represent the result of 1031 choose i, where i = [500, 530].
 *
 * @see org.kamaeleo.geom.curve.Curve
 *
 * @see org.kamaeleo.geom.curve.PascalsTriangle
 */
class BezierCurve(cp: ControlPath?, gi: GroupIterator?) : ParametricCurve(cp, gi) {
    private var t_min = 0.0
    private var t_max = 1.0

    /**
     * Sets the sample-limit.  For more information on the sample-limit, see the
     * BinaryCurveApproximationAlgorithm class.  The default sample-limit is 1.
     *
     * @see org.kamaeleo.geom.curve.BinaryCurveApproximationAlgorithm
     *
     * @see .getSampleLimit
     */
    override var sampleLimit = 1
        set(limit) {
            field = limit
        }

    /**
     * Specifies the interval that the curve should define itself on.  The default interval is [0.0, 1.0].
     *
     * @see .t_min
     * @see .t_max
     */
    fun setInterval(t_min: Double, t_max: Double) {
        this.t_min = t_min
        this.t_max = t_max
    }

    /**
     * Returns the starting interval value.
     *
     * @see .setInterval
     * @see .t_max
     */
    fun t_min(): Double {
        return t_min
    }

    /**
     * Returns the finishing interval value.
     *
     * @see .setInterval
     * @see .t_min
     */
    fun t_max(): Double {
        return t_max
    }

    /**
     * The only requirement for this curve is the group-iterator must be in range or this method returns quietly.
     */
    override fun appendTo(mp: MultiPath) {
        if (!gi!!.isInRange(0, cp!!.numPoints())) return
        val n: Int = mp.dimension
        val d = DoubleArray(n + 1)
        d[n] = t_min
        eval(d)
        if (connect) mp.lineTo(d) else mp.moveTo(d)
        BinaryCurveApproximationAlgorithm.genPts(this, t_min, t_max, mp)
    }

    override fun eval(p: DoubleArray) {
        val t = p[p.size - 1]
        val numPts: Int = gi!!.groupSize
        if (numPts > a.size) a = DoubleArray(2 * numPts)
        a[numPts - 1] = 1.0
        val one_minus_t = 1.0 - t
        for (i in numPts - 2 downTo 0) a[i] = a[i + 1] * one_minus_t
        gi!!.set(0, 0)
        var i = 0
        var b = 1.0
        while (i < numPts) {
            val pt: Double = PascalsTriangle.nCr(numPts - 1, i)
            if (pt.isNaN() || pt.isNaN()) {
                // are there any techniques that can be used
                // to calculate past 1030 points?
                // 1031 choose 515 == infinity
            } else {
                val gravity = a[i] * b * pt
                val d: DoubleArray = cp!!.getPoint(gi!!.next())!!.location
                for (j in 0 until p.size - 1) p[j] += d[j] * gravity
            }
            b *= t
            i++
        }
    }

    override fun resetMemory() {
        if (a.size > 0) a = DoubleArray(0)
    }

    companion object {
        // a[] is required to compute (1 - t)^n starting from the last index.
        // The idea is that all Bezier curves can share the same array, which
        // is more memory efficient than each Bezier curve having its own array.
        private var a = DoubleArray(0)
    }
}