package org.kamaeleo.geom.curve

/**
 *
 * A Curve is an object that defines itself using mathematical equations and points of a
 * control-path in a given dimension.  Classes that extend Curve must define the appendTo method.
 *
 *
 *
 * A Curve object can be appended to a MultiPath object which creates a series of points that
 * represent the curve.  Most curves in the com.graphbuilder.curve package extend the
 * ParametricCurve class, with one exception being the Polyline class.  The ParametricCurve
 * class allows curves to use the BinaryCurveApproximationAlgorithm class to generate
 * the points.  The number of points created depends on the curve properties, the control points,
 * the sample limit and the flatness.  The flatness is the only parameter that is controlled
 * by the multi-path.  How the flatness and sample limit affect the number of points created
 * is described in the BinaryCurveApproximationAlgorithm class.
 *
 *
 *
 * The data representation of a point is a double array.  The double array was chosen because
 * it offers a high degree of precision.  An array based representation supports n-dimensional
 * curves, with index locations acting as dimensions.
 *
 *
 *
 * The ControlPath stores com.graphbuilder.curve.Point objects.  The reason the ControlPath
 * stores Point objects and not double arrays is to provide one level of indirection.  For example,
 * suppose two different objects had a reference to the same double array.  To change the dimension
 * of the array, a new array has to be created.  This would require updating the references in each
 * of the objects to the new array.  However, if a Point object stores the array, and the objects
 * refer to the Point object, then only the reference to the array in the Point object needs to be
 * updated.  However, multi-paths do not store Point objects, but instead references to double arrays.
 * There is no need for MultiPaths to reference Point objects because the points inside a multi-path
 * are generated.
 *
 *
 *
 * In many of the implementations, curve instances share static memory blocks which are used to
 * store the results of computations.  The reason this is done is to avoid using excessive memory.
 * However, none of the methods in the curve package are synchronized.  Thus, multiple threads should
 * not be appending curves to multi-paths, otherwise the results of computations will be overwritten.
 *
 * @see MultiPath
 *
 * @see ControlPath
 *
 * @see GroupIterator
 *
 * @see BinaryCurveApproximationAlgorithm
 *
 * @see ParametricCurve
 *
 * @see Point
 *
 * @see .appendTo
 */
abstract class Curve internal constructor(cp: ControlPath?, gi: GroupIterator?) {
    /**
     * @see .getControlPath
     */
    var cp: ControlPath? = null

    /**
     * @see .getGroupIterator
     */
    var gi: GroupIterator? = null
    /**
     * Connect is used in the appendTo method to determine if the first point appended should be of type
     * MOVE_TO or LINE_TO.  If connect is true then LINE_TO is used.  If connect is false then MOVE_TO is
     * used.
     *
     * @see .setConnect
     * @see .appendTo
     * @see MultiPath.MOVE_TO
     *
     * @see MultiPath.LINE_TO
     */
    /**
     * Sets the value of the boolean connect flag.
     *
     * @see .getConnect
     */
    /**
     * @see .getConnect
     */
    var connect = false
    /**
     * Returns the control-path this curve uses to define itself.
     *
     * @see .setControlPath
     * @see ControlPath
     */
    /**
     * Sets the control-path this curve uses to define itself.
     *
     * @see .getControlPath
     */
    var controlPath: org.kamaeleo.geom.curve.ControlPath?
        get() = cp
        set(cp) {
            this.cp = cp
        }
    /**
     * Returns the group-iterator associated with this curve.  The group-iterator determines
     * which points from the control-path, and the order of the points this curve uses to define
     * itself.
     *
     * @see .setGroupIterator
     * @see .getControlPath
     */
    /**
     * Sets the group-iterator this curve uses to define itself.
     *
     * @see .getGroupIterator
     */
    var groupIterator: org.kamaeleo.geom.curve.GroupIterator?
        get() = gi
        set(gi) {
            this.gi = gi
        }

    /**
     * Appends a sequence of points defined by this curve to the multi-path.  If the internal state of
     * the curve is such that it cannot be evaluated, then the specification is for this method to
     * return quietly.
     *
     * @see MultiPath
     *
     * @see BinaryCurveApproximationAlgorithm
     */
    abstract fun appendTo(mp: MultiPath)

    /**
     * Resets the shared memory to the initial state.
     */
    open fun resetMemory() {}

    /**
     * Constructs a curve with the specified control-path and group-iterator.
     */
    init {
        controlPath = cp
        groupIterator = gi
    }
}