package org.kamaeleo.geom.curve

/**
 *
 * A GroupIterator allows a curve to choose the points from a control-path that the curve uses to
 * define itself.  The subset of points to use is defined by a control-string.  The control-string is evaluated
 * to produce a series of integer groups that are iterated over.
 *
 *
 *
 * A common way to create a GroupIterator that uses all the points of a control-path is:
 *
 *
 * <pre>
 * GroupIterator gi = new GroupIterator("0:n-1", cp.getNumPoints());
</pre> *
 *
 *
 *
 * The "0:n-1" is the control-string and cp is a control-path.  Once created, the groups that are iterated
 * over cannot be changed.  Thus, if the number of points in the control-path changes, then a new group iterator
 * is required.
 *
 *
 *
 * Some blended curves do not evaluate over the first and last points.  A technique to get the curve to
 * connect to the endpoints is to create duplicate endpoints.  But instead of doing that, a control-string
 * such as "0,0:n-1,n-1" will do the same thing.
 *
 *
 *
 * The syntax of the control-string is fairly basic.  A control-string consists of one or more groups
 * separated by a comma and possibly additional whitespace.  Each group consists of either 1 or 2 expressions.
 * If there are two expressions, then a colon ':' separates them.  The expressions are parsed using the
 * ExpressionTree.parse(String) method.  Each expression can contain at most one variable and the value of that
 * variable is set to the value specified in the GroupIterator constructor.  An exception is thrown if there
 * are multiple variables in a single expression.  The result of evaluating the expressions is rounded.
 *
 *
 *
 * <center>**<font size="4">Detailed Example</font>**</center>
 *
 *
 *
 * Suppose the control-string is: "1:4,8:n/2,7:3,5"
 * <br></br>Suppose n = 21, then the internal group array is {1, 4, 8, 11, 7, 3, 5, 5}
 * <br></br>Notice that the length of the group array is twice the number groups.
 * <br></br>The group size is (|1 - 4| + 1) + (|8 - 11| + 1) + (|7 - 3| + 1) + (|5 - 5| + 1) = 14.
 *
 *
 * <pre>
 * GroupIterator gi = new GroupIterator("1:4,8:n/2,7:3,5", 21);
 *
 * while (gi.hasNext())
 * System.out.print(gi.next() + ", ");
 *
 * Output:  1, 2, 3, 4, 8, 9, 10, 11, 7, 6, 5, 4, 3, 5
 * index_i: 0, 0, 0, 0, 2, 2,  2,  2, 4, 4, 4, 4, 4, 6
 * count_j: 0, 1, 2, 3, 0, 1,  2,  3, 0, 1, 2, 3, 4, 0
 *
</pre> *
 *
 *
 *
 * Notice that the number of numbers outputted is 14, which is the group size.  At this point, calling next()
 * will result in an exception.  However, GroupIterator objects are reusable and can be reset by calling the
 * reset() method or by calling set(0, 0).
 *
 *
 *
 * The variables index_i and count_j represent the current state of the group-iterator.  The index_i
 * is an index location in the internal group-array that keeps track of the current group.  Since each
 * group always has a start index and a finish index, index_i refers to the first index and increments by 2.
 *
 *
 *
 * The count_j variable increments by one until group[index_i]  count_j == group[index_i + 1].
 * At this point, count_j is reset to 0 and index_i is incremented by 2.
 *
 * @see org.kamaeleo.geom.curve.Curve
 *
 * @see org.kamaeleo.geom.curve.ControlPath
 */
class GroupIterator(group: IntArray?) {
    private val group: IntArray?
    private var index_i = 0
    private var count_j = 0

    /**
     * Returns the total number of times next() can be called before hasNext() returns false starting from state 0, 0.
     */
    val groupSize: Int
        get() {
            var size = 0
            var i = 0
            while (i < group!!.size) {
                var dif = group[i] - group[i + 1]
                if (dif < 0) dif = -dif
                size += dif + 1
                i += 2
            }
            return size
        }

    /**
     * Returns true if the iterator is not finished.  True will be returned if the value of index_i is
     * less than the internal group array length, false otherwise.
     */
    operator fun hasNext(): Boolean {
        return index_i < group!!.size
    }

    /**
     * Returns the current index and advances the state to the next index.
     */
    operator fun next(): Int {
        var x = group!![index_i]
        val y = group[index_i + 1]
        if (x <= y) {
            x += count_j
            if (x >= y) {
                count_j = 0
                index_i += 2
            } else count_j++
        } else {
            x -= count_j
            if (x <= y) {
                count_j = 0
                index_i += 2
            } else count_j++
        }
        return x
    }

    /**
     * Sets the current state of the iterator.
     */
    operator fun set(index_i: Int, count_j: Int) {
        this.index_i = index_i
        this.count_j = count_j
    }

    /**
     * index_i is the index location into the internal group array of the current group.
     */
    fun index_i(): Int {
        return index_i
    }

    /**
     * count_j is the increment that keeps track of the position in the current group.
     */
    fun count_j(): Int {
        return count_j
    }

    /**
     * Returns true if all values returned by next() are &gt;= min and &lt; max, false otherwise.  This is useful
     * to determine if next() will generate an index value outside of a specified range.
     */
    fun isInRange(min: Int, max: Int): Boolean {
        for (aGroup in group!!) if (aGroup < min || aGroup >= max) return false
        return true
    }

    /**
     * Constructs a group-iterator by copying the specified group array into a new internal array.  A control-string
     * is created based on the values in the array.
     */
    init {
        this.group = group
    }
}