package org.kamaeleo.graphics

import org.kamaeleo.color.CPColor
import org.kamaeleo.font.CPFont
import org.kamaeleo.text.CPTextMetrics

interface GraphicsContext2D {
    fun getTextMetrics(text: String): CPTextMetrics

    fun drawGradient(
        color1: CPColor,
        color2: CPColor,
        x: Double,
        y: Double,
        w: Double,
        h: Double,
        isVertical: Boolean
    )

    /**
     * Saves the following attributes onto a stack.
     *
     *  * Global Alpha
     *  * Global Blend Operation
     *  * Transform
     *  * Fill Paint
     *  * Stroke Paint
     *  * Line Width
     *  * Line Cap
     *  * Line Join
     *  * Miter Limit
     *  * Clip
     *  * Font
     *  * Text Align
     *  * Text Baseline
     *  * Effect
     *  * Fill Rule
     *
     * This method does NOT alter the current state in any way. Also, note that
     * the current path is not saved.
     */
    fun save()

    /**
     * Pops the state off of the stack, setting the following attributes to their
     * value at the time when that state was pushed onto the stack. If the stack
     * is empty then nothing is changed.
     *
     *
     *  * Global Alpha
     *  * Global Blend Operation
     *  * Transform
     *  * Fill Paint
     *  * Stroke Paint
     *  * Line Width
     *  * Line Cap
     *  * Line Join
     *  * Miter Limit
     *  * Clip
     *  * Font
     *  * Text Align
     *  * Text Baseline
     *  * Effect
     *  * Fill Rule
     *
     * Note that the current path is not restored.
     */
    fun restore()

    /**
     * Translates the current transform by x, y.
     *
     * @param x value to translate along the x axis.
     * @param y value to translate along the y axis.
     */
    fun translate(x: Double, y: Double)

    /**
     * Scales the current transform by x, y.
     *
     * @param x value to scale in the x axis.
     * @param y value to scale in the y axis.
     */
    fun scale(x: Double, y: Double)

    /**
     * Rotates the current transform in degrees.
     *
     * @param degrees value in degrees to rotate the current transform.
     */
    fun rotate(degrees: Double)

    /**
     * Concatenates the input with the current transform.
     *
     * @param mxx - the X coordinate scaling element of the 3x4 matrix
     * @param myx - the Y coordinate shearing element of the 3x4 matrix
     * @param mxy - the X coordinate shearing element of the 3x4 matrix
     * @param myy - the Y coordinate scaling element of the 3x4 matrix
     * @param mxt - the X coordinate translation element of the 3x4 matrix
     * @param myt - the Y coordinate translation element of the 3x4 matrix
     */
    fun transform(
        mxx: Double, myx: Double,
        mxy: Double, myy: Double,
        mxt: Double, myt: Double
    )

    /**
     * Sets the current transform.
     *
     * @param mxx - the X coordinate scaling element of the 3x4 matrix
     * @param myx - the Y coordinate shearing element of the 3x4 matrix
     * @param mxy - the X coordinate shearing element of the 3x4 matrix
     * @param myy - the Y coordinate scaling element of the 3x4 matrix
     * @param mxt - the X coordinate translation element of the 3x4 matrix
     * @param myt - the Y coordinate translation element of the 3x4 matrix
     */
    fun setTransform(
        mxx: Double, myx: Double,
        mxy: Double, myy: Double,
        mxt: Double, myt: Double
    )

    /**
     * Sets the global alpha of the current state.
     * The default value is `1.0`.
     * Any valid double can be set, but only values in the range
     * `[0.0, 1.0]` are valid and the nearest value in that
     * range will be used for rendering.
     * The global alpha is a [common attribute](#comm-attr)
     * used for nearly all rendering methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @param alpha the new alpha value, clamped to `[0.0, 1.0]`
     * during actual use.
     */
    fun setGlobalAlpha(alpha: Double)

    /**
     * Gets the current global alpha.
     * The default value is {@code 1.0}.
     * The global alpha is a <a href="#comm-attr">common attribute</a>
     * used for nearly all rendering methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return the current global alpha.
     */
//    double getGlobalAlpha();

    /**
     * Sets the global blend mode.
     * The default value is {@link BlendMode#SRC_OVER SRC_OVER}.
     * A {@code null} value will be ignored and the current value will remain unchanged.
     * The blend mode is a <a href="#comm-attr">common attribute</a>
     * used for nearly all rendering methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @param op the {@code BlendMode} that will be set or null.
     */
//    public void setGlobalBlendMode(BlendMode op);

    /**
     * Gets the global blend mode.
     * The default value is {@link BlendMode#SRC_OVER SRC_OVER}.
     * The blend mode is a <a href="#comm-attr">common attribute</a>
     * used for nearly all rendering methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return the global {@code BlendMode} of the current state.
     */
//    public BlendMode getGlobalBlendMode();

    /**
     * Gets the current global alpha.
     * The default value is `1.0`.
     * The global alpha is a [common attribute](#comm-attr)
     * used for nearly all rendering methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return the current global alpha.
     */
    //    double getGlobalAlpha();
    /**
     * Sets the global blend mode.
     * The default value is [SRC_OVER][BlendMode.SRC_OVER].
     * A `null` value will be ignored and the current value will remain unchanged.
     * The blend mode is a [common attribute](#comm-attr)
     * used for nearly all rendering methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @param op the `BlendMode` that will be set or null.
     */
    //    public void setGlobalBlendMode(BlendMode op);
    /**
     * Gets the global blend mode.
     * The default value is [SRC_OVER][BlendMode.SRC_OVER].
     * The blend mode is a [common attribute](#comm-attr)
     * used for nearly all rendering methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return the global `BlendMode` of the current state.
     */
    //    public BlendMode getGlobalBlendMode();
    /**
     * Sets the current fill paint attribute.
     * The default value is [BLACK][Color.BLACK].
     * The fill paint is a [fill attribute](#fill-attr)
     * used for any of the fill methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * A `null` value will be ignored and the current value will remain unchanged.
     *
     * @param p The `Paint` to be used as the fill `Paint` or null.
     */
    fun setFill(p: CPColor)

    /**
     * Gets the current fill paint attribute.
     * The default value is {@link Color#BLACK BLACK}.
     * The fill paint is a <a href="#fill-attr">fill attribute</a>
     * used for any of the fill methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return p The {@code Paint} to be used as the fill {@code Paint}.
     */
//    public Color getFill();

    /**
     * Gets the current fill paint attribute.
     * The default value is [BLACK][Color.BLACK].
     * The fill paint is a [fill attribute](#fill-attr)
     * used for any of the fill methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return p The `Paint` to be used as the fill `Paint`.
     */
    //    public Color getFill();
    /**
     * Sets the current stroke paint attribute.
     * The default value is [BLACK][Color.BLACK].
     * The stroke paint is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * A `null` value will be ignored and the current value will remain unchanged.
     *
     * @param p The Paint to be used as the stroke Paint or null.
     */
    fun setStroke(p: CPColor)

    /**
     * Gets the current stroke.
     * The default value is {@link Color#BLACK BLACK}.
     * The stroke paint is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return the {@code Paint} to be used as the stroke {@code Paint}.
     */
//    public Color getStroke();

    /**
     * Gets the current stroke.
     * The default value is [BLACK][Color.BLACK].
     * The stroke paint is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return the `Paint` to be used as the stroke `Paint`.
     */
    //    public Color getStroke();
    /**
     * Sets the current line width.
     * The default value is `1.0`.
     * The line width is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * An infinite or non-positive value outside of the range `(0, +inf)`
     * will be ignored and the current value will remain unchanged.
     *
     * @param lw value in the range {0-positive infinity}, with any other value
     * being ignored and leaving the value unchanged.
     */
    fun setLineWidth(lw: Double)

    /**
     * Gets the current line width.
     * The default value is {@code 1.0}.
     * The line width is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return value between 0 and infinity.
     */
//    public double getLineWidth();

    /**
     * Sets the current stroke line cap.
     * The default value is {@link StrokeLineCap#SQUARE SQUARE}.
     * The line cap is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * A {@code null} value will be ignored and the current value will remain unchanged.
     *
     * @param cap {@code StrokeLineCap} with a value of Butt, Round, or Square or null.
     */
//    public void setLineCap(StrokeLineCap cap);

    /**
     * Gets the current stroke line cap.
     * The default value is {@link StrokeLineCap#SQUARE SQUARE}.
     * The line cap is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return {@code StrokeLineCap} with a value of Butt, Round, or Square.
     */
//    public StrokeLineCap getLineCap();

    /**
     * Sets the current stroke line join.
     * The default value is {@link StrokeLineJoin#MITER}.
     * The line join is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * A {@code null} value will be ignored and the current value will remain unchanged.
     *
     * @param join {@code StrokeLineJoin} with a value of Miter, Bevel, or Round or null.
     */
//    public void setLineJoin(StrokeLineJoin join);

    /**
     * Gets the current stroke line join.
     * The default value is {@link StrokeLineJoin#MITER}.
     * The line join is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return {@code StrokeLineJoin} with a value of Miter, Bevel, or Round.
     */
//    public StrokeLineJoin getLineJoin();

    /**
     * Sets the current miter limit.
     * The default value is {@code 10.0}.
     * The miter limit is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * An infinite or non-positive value outside of the range {@code (0, +inf)}
     * will be ignored and the current value will remain unchanged.
     *
     * @param ml miter limit value between 0 and positive infinity with
     * any other value being ignored and leaving the value unchanged.
     */
//    public void setMiterLimit(double ml);

    /**
     * Gets the current miter limit.
     * The default value is {@code 10.0}.
     * The miter limit is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return the miter limit value in the range {@code 0.0-positive infinity}
     */
//    public double getMiterLimit();

    /**
     * Sets the current stroke line dash pattern to a normalized copy of
     * the argument.
     * The default value is {@code null}.
     * The line dash array is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * If the array is {@code null} or empty or contains all {@code 0} elements
     * then dashing will be disabled and the current dash array will be set
     * to {@code null}.
     * If any of the elements of the array are a negative, infinite, or NaN
     * value outside the range {@code [0, +inf)} then the entire array will
     * be ignored and the current dash array will remain unchanged.
     * If the array is an odd length then it will be treated as if it
     * were two copies of the array appended to each other.
     *
     * @param dashes the array of finite non-negative dash lengths
     * @since JavaFX 8u40
     */
//    public void setLineDashes(double... dashes);

    /**
     * Gets a copy of the current line dash array.
     * The default value is {@code null}.
     * The array may be normalized by the validation tests in the
     * {@link #setLineDashes(double...)} method.
     * The line dash array is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return a copy of the current line dash array.
     * @since JavaFX 8u40
     */
//    public double[] getLineDashes();

    /**
     * Sets the line dash offset.
     * The default value is {@code 0.0}.
     * The line dash offset is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * An infinite or NaN value outside of the range {@code (-inf, +inf)}
     * will be ignored and the current value will remain unchanged.
     *
     * @param dashOffset the line dash offset in the range {@code (-inf, +inf)}
     * @since JavaFX 8u40
     */
//    public void setLineDashOffset(double dashOffset);

    /**
     * Gets the current line dash offset.
     * The default value is {@code 0.0}.
     * The line dash offset is a <a href="#strk-attr">stroke attribute</a>
     * used for any of the stroke methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return the line dash offset in the range {@code (-inf, +inf)}
     * @since JavaFX 8u40
     */
//    public double getLineDashOffset();

    /**
     * Gets the current line width.
     * The default value is `1.0`.
     * The line width is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return value between 0 and infinity.
     */
    //    public double getLineWidth();
    /**
     * Sets the current stroke line cap.
     * The default value is [SQUARE][StrokeLineCap.SQUARE].
     * The line cap is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * A `null` value will be ignored and the current value will remain unchanged.
     *
     * @param cap `StrokeLineCap` with a value of Butt, Round, or Square or null.
     */
    //    public void setLineCap(StrokeLineCap cap);
    /**
     * Gets the current stroke line cap.
     * The default value is [SQUARE][StrokeLineCap.SQUARE].
     * The line cap is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return `StrokeLineCap` with a value of Butt, Round, or Square.
     */
    //    public StrokeLineCap getLineCap();
    /**
     * Sets the current stroke line join.
     * The default value is [StrokeLineJoin.MITER].
     * The line join is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * A `null` value will be ignored and the current value will remain unchanged.
     *
     * @param join `StrokeLineJoin` with a value of Miter, Bevel, or Round or null.
     */
    //    public void setLineJoin(StrokeLineJoin join);
    /**
     * Gets the current stroke line join.
     * The default value is [StrokeLineJoin.MITER].
     * The line join is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return `StrokeLineJoin` with a value of Miter, Bevel, or Round.
     */
    //    public StrokeLineJoin getLineJoin();
    /**
     * Sets the current miter limit.
     * The default value is `10.0`.
     * The miter limit is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * An infinite or non-positive value outside of the range `(0, +inf)`
     * will be ignored and the current value will remain unchanged.
     *
     * @param ml miter limit value between 0 and positive infinity with
     * any other value being ignored and leaving the value unchanged.
     */
    //    public void setMiterLimit(double ml);
    /**
     * Gets the current miter limit.
     * The default value is `10.0`.
     * The miter limit is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return the miter limit value in the range `0.0-positive infinity`
     */
    //    public double getMiterLimit();
    /**
     * Sets the current stroke line dash pattern to a normalized copy of
     * the argument.
     * The default value is `null`.
     * The line dash array is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * If the array is `null` or empty or contains all `0` elements
     * then dashing will be disabled and the current dash array will be set
     * to `null`.
     * If any of the elements of the array are a negative, infinite, or NaN
     * value outside the range `[0, +inf)` then the entire array will
     * be ignored and the current dash array will remain unchanged.
     * If the array is an odd length then it will be treated as if it
     * were two copies of the array appended to each other.
     *
     * @param dashes the array of finite non-negative dash lengths
     * @since JavaFX 8u40
     */
    //    public void setLineDashes(double... dashes);
    /**
     * Gets a copy of the current line dash array.
     * The default value is `null`.
     * The array may be normalized by the validation tests in the
     * [.setLineDashes] method.
     * The line dash array is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return a copy of the current line dash array.
     * @since JavaFX 8u40
     */
    //    public double[] getLineDashes();
    /**
     * Sets the line dash offset.
     * The default value is `0.0`.
     * The line dash offset is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * An infinite or NaN value outside of the range `(-inf, +inf)`
     * will be ignored and the current value will remain unchanged.
     *
     * @param dashOffset the line dash offset in the range `(-inf, +inf)`
     * @since JavaFX 8u40
     */
    //    public void setLineDashOffset(double dashOffset);
    /**
     * Gets the current line dash offset.
     * The default value is `0.0`.
     * The line dash offset is a [stroke attribute](#strk-attr)
     * used for any of the stroke methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return the line dash offset in the range `(-inf, +inf)`
     * @since JavaFX 8u40
     */
    //    public double getLineDashOffset();
    /**
     * Sets the current Font.
     * The default value is specified by [Font.getDefault].
     * The font is a [text attribute](#text-attr)
     * used for any of the text methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * A `null` value will be ignored and the current value will remain unchanged.
     *
     * @param f the Font or null.
     */
    fun setFont(f: CPFont)

    /**
     * Gets the current Font.
     * The default value is specified by {@link Font#getDefault()}.
     * The font is a <a href="#text-attr">text attribute</a>
     * used for any of the text methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return the Font
     */
//    public Font getFont();

    /**
     * Sets the current Font Smoothing Type.
     * The default value is {@link FontSmoothingType#GRAY GRAY}.
     * The font smoothing type is a <a href="#text-attr">text attribute</a>
     * used for any of the text methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * A {@code null} value will be ignored and the current value will remain unchanged.
     * <p>
     * <b>Note</b> that the {@code FontSmoothingType} value of
     * {@link FontSmoothingType#LCD LCD} is only supported over an opaque
     * background.  {@code LCD} text will generally appear as {@code GRAY}
     * text over transparent or partially transparent pixels, and in some
     * implementations it may not be supported at all on a {@link Canvas}
     * because the required support does not exist for surfaces which contain
     * an alpha channel as all {@code Canvas} objects do.
     *
     * @param fontsmoothing the {@link FontSmoothingType} or null
     * @since JavaFX 8u40
     */
//    public void setFontSmoothingType(FontSmoothingType fontsmoothing);

    /**
     * Gets the current Font Smoothing Type.
     * The default value is {@link FontSmoothingType#GRAY GRAY}.
     * The font smoothing type is a <a href="#text-attr">text attribute</a>
     * used for any of the text methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return the {@link FontSmoothingType}
     * @since JavaFX 8u40
     */
//    public FontSmoothingType getFontSmoothingType();

    /**
     * Defines horizontal text alignment, relative to the text {@code x} origin.
     * The default value is {@link TextAlignment#LEFT LEFT}.
     * The text alignment is a <a href="#text-attr">text attribute</a>
     * used for any of the text methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * <p>
     * Let horizontal bounds represent the logical width of a single line of
     * text. Where each line of text has a separate horizontal bounds.
     * <p>
     * Then TextAlignment is specified as:
     * <ul>
     * <li>Left: the left edge of the horizontal bounds will be at {@code x}.
     * <li>Center: the center, halfway between left and right edge, of the
     * horizontal bounds will be at {@code x}.
     * <li>Right: the right edge of the horizontal bounds will be at {@code x}.
     * </ul>
     * <p>
     *
     * Note: Canvas does not support line wrapping, therefore the text
     * alignment Justify is identical to left aligned text.
     * <p>
     * A {@code null} value will be ignored and the current value will remain unchanged.
     *
     * @param align {@code TextAlignment} with values of Left, Center, Right or null.
     */
//    public void setTextAlign(TextAlignment align);

    /**
     * Gets the current {@code TextAlignment}.
     * The default value is {@link TextAlignment#LEFT LEFT}.
     * The text alignment is a <a href="#text-attr">text attribute</a>
     * used for any of the text methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return {@code TextAlignment} with values of Left, Center, Right, or
     * Justify.
     */
//    public TextAlignment getTextAlign();

    /**
     * Sets the current Text Baseline.
     * The default value is {@link VPos#BASELINE BASELINE}.
     * The text baseline is a <a href="#text-attr">text attribute</a>
     * used for any of the text methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * A {@code null} value will be ignored and the current value will remain unchanged.
     *
     * @param baseline {@code VPos} with values of Top, Center, Baseline, or Bottom or null.
     */
//    public void setTextBaseline(VPos baseline);

    /**
     * Gets the current Text Baseline.
     * The default value is {@link VPos#BASELINE BASELINE}.
     * The text baseline is a <a href="#text-attr">text attribute</a>
     * used for any of the text methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return {@code VPos} with values of Top, Center, Baseline, or Bottom
     */
//    public VPos getTextBaseline();

    /**
     * Gets the current Font.
     * The default value is specified by [Font.getDefault].
     * The font is a [text attribute](#text-attr)
     * used for any of the text methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return the Font
     */
    //    public Font getFont();
    /**
     * Sets the current Font Smoothing Type.
     * The default value is [GRAY][FontSmoothingType.GRAY].
     * The font smoothing type is a [text attribute](#text-attr)
     * used for any of the text methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * A `null` value will be ignored and the current value will remain unchanged.
     *
     *
     * **Note** that the `FontSmoothingType` value of
     * [LCD][FontSmoothingType.LCD] is only supported over an opaque
     * background.  `LCD` text will generally appear as `GRAY`
     * text over transparent or partially transparent pixels, and in some
     * implementations it may not be supported at all on a [Canvas]
     * because the required support does not exist for surfaces which contain
     * an alpha channel as all `Canvas` objects do.
     *
     * @param fontsmoothing the [FontSmoothingType] or null
     * @since JavaFX 8u40
     */
    //    public void setFontSmoothingType(FontSmoothingType fontsmoothing);
    /**
     * Gets the current Font Smoothing Type.
     * The default value is [GRAY][FontSmoothingType.GRAY].
     * The font smoothing type is a [text attribute](#text-attr)
     * used for any of the text methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return the [FontSmoothingType]
     * @since JavaFX 8u40
     */
    //    public FontSmoothingType getFontSmoothingType();
    /**
     * Defines horizontal text alignment, relative to the text `x` origin.
     * The default value is [LEFT][TextAlignment.LEFT].
     * The text alignment is a [text attribute](#text-attr)
     * used for any of the text methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * Let horizontal bounds represent the logical width of a single line of
     * text. Where each line of text has a separate horizontal bounds.
     *
     *
     * Then TextAlignment is specified as:
     *
     *  * Left: the left edge of the horizontal bounds will be at `x`.
     *  * Center: the center, halfway between left and right edge, of the
     * horizontal bounds will be at `x`.
     *  * Right: the right edge of the horizontal bounds will be at `x`.
     *
     *
     *
     *
     * Note: Canvas does not support line wrapping, therefore the text
     * alignment Justify is identical to left aligned text.
     *
     *
     * A `null` value will be ignored and the current value will remain unchanged.
     *
     * @param align `TextAlignment` with values of Left, Center, Right or null.
     */
    //    public void setTextAlign(TextAlignment align);
    /**
     * Gets the current `TextAlignment`.
     * The default value is [LEFT][TextAlignment.LEFT].
     * The text alignment is a [text attribute](#text-attr)
     * used for any of the text methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return `TextAlignment` with values of Left, Center, Right, or
     * Justify.
     */
    //    public TextAlignment getTextAlign();
    /**
     * Sets the current Text Baseline.
     * The default value is [BASELINE][VPos.BASELINE].
     * The text baseline is a [text attribute](#text-attr)
     * used for any of the text methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * A `null` value will be ignored and the current value will remain unchanged.
     *
     * @param baseline `VPos` with values of Top, Center, Baseline, or Bottom or null.
     */
    //    public void setTextBaseline(VPos baseline);
    /**
     * Gets the current Text Baseline.
     * The default value is [BASELINE][VPos.BASELINE].
     * The text baseline is a [text attribute](#text-attr)
     * used for any of the text methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return `VPos` with values of Top, Center, Baseline, or Bottom
     */
    //    public VPos getTextBaseline();
    /**
     * Fills the given string of text at position x, y
     * with the current fill paint attribute.
     * A `null` text value will be ignored.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr),
     * [fill](#fill-attr),
     * or [text](#text-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param text the string of text or null.
     * @param x    position on the x axis.
     * @param y    position on the y axis.
     */
    fun fillText(text: String, x: Double, y: Double)

    /**
     * Draws the given string of text at position x, y
     * with the current stroke paint attribute.
     * A {@code null} text value will be ignored.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>,
     * <a href="#strk-attr">stroke</a>,
     * or <a href="#text-attr">text</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param text the string of text or null.
     * @param x position on the x axis.
     * @param y position on the y axis.
     */
//    public void strokeText(String text, double x, double y);

    /**
     * Fills text and includes a maximum width of the string.
     * If the width of the text extends past max width, then it will be sized
     * to fit.
     * A {@code null} text value will be ignored.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>,
     * <a href="#fill-attr">fill</a>,
     * or <a href="#text-attr">text</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param text the string of text or null.
     * @param x position on the x axis.
     * @param y position on the y axis.
     * @param maxWidth  maximum width the text string can have.
     */
//    public void fillText(String text, double x, double y, double maxWidth);

    /**
     * Draws text with stroke paint and includes a maximum width of the string.
     * If the width of the text extends past max width, then it will be sized
     * to fit.
     * A {@code null} text value will be ignored.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>,
     * <a href="#strk-attr">stroke</a>,
     * or <a href="#text-attr">text</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param text the string of text or null.
     * @param x position on the x axis.
     * @param y position on the y axis.
     * @param maxWidth  maximum width the text string can have.
     */
//    public void strokeText(String text, double x, double y, double maxWidth);

    /**
     * Set the filling rule attribute for determining the interior of paths
     * in fill or clip operations.
     * The default value is {@code FillRule.NON_ZERO}.
     * A {@code null} value will be ignored and the current value will remain unchanged.
     * The fill rule is a <a href="#path-attr">path attribute</a>
     * used for any of the fill or clip path methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @param fillRule {@code FillRule} with a value of  Even_odd or Non_zero or null.
     */
//    public void setFillRule(FillRule fillRule);

    /**
     * Get the filling rule attribute for determining the interior of paths
     * in fill and clip operations.
     * The default value is {@code FillRule.NON_ZERO}.
     * The fill rule is a <a href="#path-attr">path attribute</a>
     * used for any of the fill or clip path methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     *
     * @return current fill rule.
     */
//    public FillRule getFillRule();

    /**
     * Draws the given string of text at position x, y
     * with the current stroke paint attribute.
     * A `null` text value will be ignored.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr),
     * [stroke](#strk-attr),
     * or [text](#text-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param text the string of text or null.
     * @param x position on the x axis.
     * @param y position on the y axis.
     */
    //    public void strokeText(String text, double x, double y);
    /**
     * Fills text and includes a maximum width of the string.
     * If the width of the text extends past max width, then it will be sized
     * to fit.
     * A `null` text value will be ignored.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr),
     * [fill](#fill-attr),
     * or [text](#text-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param text the string of text or null.
     * @param x position on the x axis.
     * @param y position on the y axis.
     * @param maxWidth  maximum width the text string can have.
     */
    //    public void fillText(String text, double x, double y, double maxWidth);
    /**
     * Draws text with stroke paint and includes a maximum width of the string.
     * If the width of the text extends past max width, then it will be sized
     * to fit.
     * A `null` text value will be ignored.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr),
     * [stroke](#strk-attr),
     * or [text](#text-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param text the string of text or null.
     * @param x position on the x axis.
     * @param y position on the y axis.
     * @param maxWidth  maximum width the text string can have.
     */
    //    public void strokeText(String text, double x, double y, double maxWidth);
    /**
     * Set the filling rule attribute for determining the interior of paths
     * in fill or clip operations.
     * The default value is `FillRule.NON_ZERO`.
     * A `null` value will be ignored and the current value will remain unchanged.
     * The fill rule is a [path attribute](#path-attr)
     * used for any of the fill or clip path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @param fillRule `FillRule` with a value of  Even_odd or Non_zero or null.
     */
    //    public void setFillRule(FillRule fillRule);
    /**
     * Get the filling rule attribute for determining the interior of paths
     * in fill and clip operations.
     * The default value is `FillRule.NON_ZERO`.
     * The fill rule is a [path attribute](#path-attr)
     * used for any of the fill or clip path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     * @return current fill rule.
     */
    //    public FillRule getFillRule();
    /**
     * Resets the current path to empty.
     * The default path is empty.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     */
    fun beginPath()

    /**
     * Issues a move command for the current path to the given x,y coordinate.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     *
     * @param x0 the X position for the move to command.
     * @param y0 the Y position for the move to command.
     */
    fun moveTo(x0: Double, y0: Double)

    /**
     * Adds segments to the current path to make a line to the given x,y
     * coordinate.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     *
     * @param x1 the X coordinate of the ending point of the line.
     * @param y1 the Y coordinate of the ending point of the line.
     */
    fun lineTo(x1: Double, y1: Double)

    /**
     * Adds segments to the current path to make a quadratic Bezier curve.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     *
     * @param xc the X coordinate of the control point
     * @param yc the Y coordinate of the control point
     * @param x1 the X coordinate of the end point
     * @param y1 the Y coordinate of the end point
     */
    fun quadraticCurveTo(xc: Double, yc: Double, x1: Double, y1: Double)

    /**
     * Adds segments to the current path to make a cubic Bezier curve.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     *
     * @param xc1 the X coordinate of first Bezier control point.
     * @param yc1 the Y coordinate of the first Bezier control point.
     * @param xc2 the X coordinate of the second Bezier control point.
     * @param yc2 the Y coordinate of the second Bezier control point.
     * @param x1  the X coordinate of the end point.
     * @param y1  the Y coordinate of the end point.
     */
    fun bezierCurveTo(
        xc1: Double,
        yc1: Double,
        xc2: Double,
        yc2: Double,
        x1: Double,
        y1: Double
    )

    /**
     * Adds segments to the current path to make an arc.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a <a href="#path-attr">path attribute</a>
     * used for any of the path methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>
     * and <b>is not affected</b> by the {@link #save()} and
     * {@link #restore()} operations.
     * <p>
     * If {@code p0} is the current point in the path and {@code p1} is the
     * point specified by {@code (x1, y1)} and {@code p2} is the point
     * specified by {@code (x2, y2)}, then the arc segments appended will
     * be segments along the circumference of a circle of the specified
     * radius touching and inscribed into the convex (interior) side of
     * {@code p0->p1->p2}.  The path will contain a line segment (if
     * needed) to the tangent point between that circle and {@code p0->p1}
     * followed by circular arc segments to reach the tangent point between
     * the circle and {@code p1->p2} and will end with the current point at
     * that tangent point (not at {@code p2}).
     * Note that the radius and circularity of the arc segments will be
     * measured or considered relative to the current transform, but the
     * resulting segments that are computed from those untransformed
     * points will then be transformed when they are added to the path.
     * Since all computation is done in untransformed space, but the
     * pre-existing path segments are all transformed, the ability to
     * correctly perform the computation may implicitly depend on being
     * able to inverse transform the current end of the current path back
     * into untransformed coordinates.
     * </p>
     * <p>
     * If there is no way to compute and inscribe the indicated circle
     * for any reason then the entire operation will simply append segments
     * to force a line to point {@code p1}.  Possible reasons that the
     * computation may fail include:
     * <ul>
     * <li>The current path is empty.</li>
     * <li>The segments {@code p0->p1->p2} are colinear.</li>
     * <li>the current transform is non-invertible so that the current end
     * point of the current path cannot be untransformed for computation.</li>
     * </ul>
     *
     * @param x1 the X coordinate of the first point of the arc.
     * @param y1 the Y coordinate of the first point of the arc.
     * @param x2 the X coordinate of the second point of the arc.
     * @param y2 the Y coordinate of the second point of the arc.
     * @param radius the radius of the arc in the range {0.0-positive infinity}.
     */
//    public void arcTo(double x1, double y1, double x2, double y2, double radius);

    /**
     * Adds path elements to the current path to make an arc that uses Euclidean
     * degrees. This Euclidean orientation sweeps from East to North, then West,
     * then South, then back to East.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a <a href="#path-attr">path attribute</a>
     * used for any of the path methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>
     * and <b>is not affected</b> by the {@link #save()} and
     * {@link #restore()} operations.
     *
     * @param centerX the center x position of the arc.
     * @param centerY the center y position of the arc.
     * @param radiusX the x radius of the arc.
     * @param radiusY the y radius of the arc.
     * @param startAngle the starting angle of the arc in the range {@code 0-360.0}
     * @param length  the length of the baseline of the arc.
     */
//    public void arc(double centerX, double centerY,
//                    double radiusX, double radiusY,
//                    double startAngle, double length);

    /**
     * Adds segments to the current path to make an arc.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     *
     *
     * If `p0` is the current point in the path and `p1` is the
     * point specified by `(x1, y1)` and `p2` is the point
     * specified by `(x2, y2)`, then the arc segments appended will
     * be segments along the circumference of a circle of the specified
     * radius touching and inscribed into the convex (interior) side of
     * `p0->p1->p2`.  The path will contain a line segment (if
     * needed) to the tangent point between that circle and `p0->p1`
     * followed by circular arc segments to reach the tangent point between
     * the circle and `p1->p2` and will end with the current point at
     * that tangent point (not at `p2`).
     * Note that the radius and circularity of the arc segments will be
     * measured or considered relative to the current transform, but the
     * resulting segments that are computed from those untransformed
     * points will then be transformed when they are added to the path.
     * Since all computation is done in untransformed space, but the
     * pre-existing path segments are all transformed, the ability to
     * correctly perform the computation may implicitly depend on being
     * able to inverse transform the current end of the current path back
     * into untransformed coordinates.
     *
     *
     *
     * If there is no way to compute and inscribe the indicated circle
     * for any reason then the entire operation will simply append segments
     * to force a line to point `p1`.  Possible reasons that the
     * computation may fail include:
     *
     *  * The current path is empty.
     *  * The segments `p0->p1->p2` are colinear.
     *  * the current transform is non-invertible so that the current end
     * point of the current path cannot be untransformed for computation.
     *
     *
     * @param x1 the X coordinate of the first point of the arc.
     * @param y1 the Y coordinate of the first point of the arc.
     * @param x2 the X coordinate of the second point of the arc.
     * @param y2 the Y coordinate of the second point of the arc.
     * @param radius the radius of the arc in the range {0.0-positive infinity}.
     */
    //    public void arcTo(double x1, double y1, double x2, double y2, double radius);
    /**
     * Adds path elements to the current path to make an arc that uses Euclidean
     * degrees. This Euclidean orientation sweeps from East to North, then West,
     * then South, then back to East.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     *
     * @param centerX the center x position of the arc.
     * @param centerY the center y position of the arc.
     * @param radiusX the x radius of the arc.
     * @param radiusY the y radius of the arc.
     * @param startAngle the starting angle of the arc in the range `0-360.0`
     * @param length  the length of the baseline of the arc.
     */
    //    public void arc(double centerX, double centerY,
    //                    double radiusX, double radiusY,
    //                    double startAngle, double length);
    /**
     * Adds path elements to the current path to make a rectangle.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     *
     * @param x x position of the upper left corner of the rectangle.
     * @param y y position of the upper left corner of the rectangle.
     * @param w width of the rectangle.
     * @param h height of the rectangle.
     */
    fun rect(x: Double, y: Double, w: Double, h: Double)

    /**
     * Appends an SVG Path string to the current path. If there is no current
     * path the string must then start with either type of move command.
     * A {@code null} value or incorrect SVG path will be ignored.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a <a href="#path-attr">path attribute</a>
     * used for any of the path methods as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>
     * and <b>is not affected</b> by the {@link #save()} and
     * {@link #restore()} operations.
     *
     * @param svgpath the SVG Path string.
     */
//    public void appendSVGPath(String svgpath);

    /**
     * Appends an SVG Path string to the current path. If there is no current
     * path the string must then start with either type of move command.
     * A `null` value or incorrect SVG path will be ignored.
     * The coordinates are transformed by the current transform as they are
     * added to the path and unaffected by subsequent changes to the transform.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     *
     * @param svgpath the SVG Path string.
     */
    //    public void appendSVGPath(String svgpath);
    /**
     * Closes the path.
     * The current path is a [path attribute](#path-attr)
     * used for any of the path methods as specified in the
     * [Rendering Attributes Table](#attr-ops-table)
     * and **is not affected** by the [.save] and
     * [.restore] operations.
     */
    fun closePath()

    /**
     * Fills the path with the current fill paint.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr),
     * [fill](#fill-attr),
     * or [path](#path-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * Note that the path segments were transformed as they were originally
     * added to the current path so the current transform will not affect
     * those path segments again, but it may affect other attributes in
     * affect at the time of the `fill()` operation.
     *
     */
    fun fill()

    /**
     * Strokes the path with the current stroke paint.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr),
     * [stroke](#strk-attr),
     * or [path](#path-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * Note that the path segments were transformed as they were originally
     * added to the current path so the current transform will not affect
     * those path segments again, but it may affect other attributes in
     * affect at the time of the `stroke()` operation.
     *
     */
    fun stroke()

    /**
     * Intersects the current clip with the current path and applies it to
     * subsequent rendering operation as an anti-aliased mask.
     * The current clip is a [common attribute](#comm-attr)
     * used for nearly all rendering operations as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * This method will itself be affected only by the
     * [path](#path-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     * Note that the path segments were transformed as they were originally
     * added to the current path so the current transform will not affect
     * those path segments again, but it may affect other attributes in
     * affect at the time of the `stroke()` operation.
     *
     */
    fun clip()

    /**
     * Returns true if the the given x,y point is inside the path.
     *
     * @param x the X coordinate to use for the check.
     * @param y the Y coordinate to use for the check.
     * @return true if the point given is inside the path, false
     * otherwise.
     */
//    public boolean isPointInPath(double x, double y);

    /**
     * Clears a portion of the canvas with a transparent color value.
     * <p>
     * This method will be affected only by the current transform, clip,
     * and effect.
     * </p>
     *
     * @param x X position of the upper left corner of the rectangle.
     * @param y Y position of the upper left corner of the rectangle.
     * @param w width of the rectangle.
     * @param h height of the rectangle.
     */
//    public void clearRect(double x, double y, double w, double h);

    /**
     * Returns true if the the given x,y point is inside the path.
     *
     * @param x the X coordinate to use for the check.
     * @param y the Y coordinate to use for the check.
     * @return true if the point given is inside the path, false
     * otherwise.
     */
    //    public boolean isPointInPath(double x, double y);
    /**
     * Clears a portion of the canvas with a transparent color value.
     *
     *
     * This method will be affected only by the current transform, clip,
     * and effect.
     *
     *
     * @param x X position of the upper left corner of the rectangle.
     * @param y Y position of the upper left corner of the rectangle.
     * @param w width of the rectangle.
     * @param h height of the rectangle.
     */
    //    public void clearRect(double x, double y, double w, double h);
    /**
     * Fills a rectangle using the current fill paint.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr)
     * or [fill](#fill-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param x the X position of the upper left corner of the rectangle.
     * @param y the Y position of the upper left corner of the rectangle.
     * @param w the width of the rectangle.
     * @param h the height of the rectangle.
     */
    fun fillRect(x: Double, y: Double, w: Double, h: Double)

    /**
     * Strokes a rectangle using the current stroke paint.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr)
     * or [stroke](#strk-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param x the X position of the upper left corner of the rectangle.
     * @param y the Y position of the upper left corner of the rectangle.
     * @param w the width of the rectangle.
     * @param h the height of the rectangle.
     */
    fun strokeRect(x: Double, y: Double, w: Double, h: Double)

    /**
     * Fills an oval using the current fill paint.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * or <a href="#fill-attr">fill</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param x the X coordinate of the upper left bound of the oval.
     * @param y the Y coordinate of the upper left bound of the oval.
     * @param w the width at the center of the oval.
     * @param h the height at the center of the oval.
     */
//    public void fillOval(double x, double y, double w, double h);

    /**
     * Strokes an oval using the current stroke paint.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * or <a href="#strk-attr">stroke</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param x the X coordinate of the upper left bound of the oval.
     * @param y the Y coordinate of the upper left bound of the oval.
     * @param w the width at the center of the oval.
     * @param h the height at the center of the oval.
     */
//    public void strokeOval(double x, double y, double w, double h);

    /**
     * Fills an arc using the current fill paint. A {@code null} ArcType or
     * non positive width or height will cause the render command to be ignored.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * or <a href="#fill-attr">fill</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param x the X coordinate of the arc.
     * @param y the Y coordinate of the arc.
     * @param w the width of the arc.
     * @param h the height of the arc.
     * @param startAngle the starting angle of the arc in degrees.
     * @param arcExtent the angular extent of the arc in degrees.
     * @param closure closure type (Round, Chord, Open) or null.
     */
//    public void fillArc(double x, double y, double w, double h,
//                        double startAngle, double arcExtent, ArcType closure);

    /**
     * Strokes an Arc using the current stroke paint. A {@code null} ArcType or
     * non positive width or height will cause the render command to be ignored.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * or <a href="#strk-attr">stroke</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param x the X coordinate of the arc.
     * @param y the Y coordinate of the arc.
     * @param w the width of the arc.
     * @param h the height of the arc.
     * @param startAngle the starting angle of the arc in degrees.
     * @param arcExtent arcExtent the angular extent of the arc in degrees.
     * @param closure closure type (Round, Chord, Open) or null
     */
//    public void strokeArc(double x, double y, double w, double h,
//                          double startAngle, double arcExtent, ArcType closure);
    /**
     * Fills a rounded rectangle using the current fill paint.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * or <a href="#fill-attr">fill</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param x the X coordinate of the upper left bound of the oval.
     * @param y the Y coordinate of the upper left bound of the oval.
     * @param w the width at the center of the oval.
     * @param h the height at the center of the oval.
     * @param arcWidth the arc width of the rectangle corners.
     * @param arcHeight the arc height of the rectangle corners.
     */
//    public void fillRoundRect(double x, double y, double w, double h,
//                              double arcWidth, double arcHeight);

    /**
     * Strokes a rounded rectangle using the current stroke paint.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * or <a href="#strk-attr">stroke</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param x the X coordinate of the upper left bound of the oval.
     * @param y the Y coordinate of the upper left bound of the oval.
     * @param w the width at the center of the oval.
     * @param h the height at the center of the oval.
     * @param arcWidth the arc width of the rectangle corners.
     * @param arcHeight the arc height of the rectangle corners.
     */
//    public void strokeRoundRect(double x, double y, double w, double h,
//                                double arcWidth, double arcHeight);

    /**
     * Fills an oval using the current fill paint.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr)
     * or [fill](#fill-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param x the X coordinate of the upper left bound of the oval.
     * @param y the Y coordinate of the upper left bound of the oval.
     * @param w the width at the center of the oval.
     * @param h the height at the center of the oval.
     */
    //    public void fillOval(double x, double y, double w, double h);
    /**
     * Strokes an oval using the current stroke paint.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr)
     * or [stroke](#strk-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param x the X coordinate of the upper left bound of the oval.
     * @param y the Y coordinate of the upper left bound of the oval.
     * @param w the width at the center of the oval.
     * @param h the height at the center of the oval.
     */
    //    public void strokeOval(double x, double y, double w, double h);
    /**
     * Fills an arc using the current fill paint. A `null` ArcType or
     * non positive width or height will cause the render command to be ignored.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr)
     * or [fill](#fill-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param x the X coordinate of the arc.
     * @param y the Y coordinate of the arc.
     * @param w the width of the arc.
     * @param h the height of the arc.
     * @param startAngle the starting angle of the arc in degrees.
     * @param arcExtent the angular extent of the arc in degrees.
     * @param closure closure type (Round, Chord, Open) or null.
     */
    //    public void fillArc(double x, double y, double w, double h,
    //                        double startAngle, double arcExtent, ArcType closure);
    /**
     * Strokes an Arc using the current stroke paint. A `null` ArcType or
     * non positive width or height will cause the render command to be ignored.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr)
     * or [stroke](#strk-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param x the X coordinate of the arc.
     * @param y the Y coordinate of the arc.
     * @param w the width of the arc.
     * @param h the height of the arc.
     * @param startAngle the starting angle of the arc in degrees.
     * @param arcExtent arcExtent the angular extent of the arc in degrees.
     * @param closure closure type (Round, Chord, Open) or null
     */
    //    public void strokeArc(double x, double y, double w, double h,
    //                          double startAngle, double arcExtent, ArcType closure);
    /**
     * Fills a rounded rectangle using the current fill paint.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr)
     * or [fill](#fill-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param x the X coordinate of the upper left bound of the oval.
     * @param y the Y coordinate of the upper left bound of the oval.
     * @param w the width at the center of the oval.
     * @param h the height at the center of the oval.
     * @param arcWidth the arc width of the rectangle corners.
     * @param arcHeight the arc height of the rectangle corners.
     */
    //    public void fillRoundRect(double x, double y, double w, double h,
    //                              double arcWidth, double arcHeight);
    /**
     * Strokes a rounded rectangle using the current stroke paint.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr)
     * or [stroke](#strk-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param x the X coordinate of the upper left bound of the oval.
     * @param y the Y coordinate of the upper left bound of the oval.
     * @param w the width at the center of the oval.
     * @param h the height at the center of the oval.
     * @param arcWidth the arc width of the rectangle corners.
     * @param arcHeight the arc height of the rectangle corners.
     */
    //    public void strokeRoundRect(double x, double y, double w, double h,
    //                                double arcWidth, double arcHeight);
    /**
     * Strokes a line using the current stroke paint.
     *
     *
     * This method will be affected by any of the
     * [global common](#comm-attr)
     * or [stroke](#strk-attr)
     * attributes as specified in the
     * [Rendering Attributes Table](#attr-ops-table).
     *
     *
     * @param x1 the X coordinate of the starting point of the line.
     * @param y1 the Y coordinate of the starting point of the line.
     * @param x2 the X coordinate of the ending point of the line.
     * @param y2 the Y coordinate of the ending point of the line.
     */
    fun strokeLine(x1: Double, y1: Double, x2: Double, y2: Double)

    /**
     * Fills a polygon with the given points using the currently set fill paint.
     * A {@code null} value for any of the arrays will be ignored and nothing will be drawn.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>,
     * <a href="#fill-attr">fill</a>,
     * or <a href="#path-attr">Fill Rule</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param xPoints array containing the x coordinates of the polygon's points or null.
     * @param yPoints array containing the y coordinates of the polygon's points or null.
     * @param nPoints the number of points that make the polygon.
     */
//    public void fillPolygon(double xPoints[], double yPoints[], int nPoints);

    /**
     * Strokes a polygon with the given points using the currently set stroke paint.
     * A {@code null} value for any of the arrays will be ignored and nothing will be drawn.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * or <a href="#strk-attr">stroke</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param xPoints array containing the x coordinates of the polygon's points or null.
     * @param yPoints array containing the y coordinates of the polygon's points or null.
     * @param nPoints the number of points that make the polygon.
     */
//    public void strokePolygon(double xPoints[], double yPoints[], int nPoints);

    /**
     * Strokes a polyline with the given points using the currently set stroke
     * paint attribute.
     * A {@code null} value for any of the arrays will be ignored and nothing will be drawn.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * or <a href="#strk-attr">stroke</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param xPoints array containing the x coordinates of the polyline's points or null.
     * @param yPoints array containing the y coordinates of the polyline's points or null.
     * @param nPoints the number of points that make the polyline.
     */
//    public void strokePolyline(double xPoints[], double yPoints[], int nPoints);

    /**
     * Draws an image at the given x, y position using the width
     * and height of the given image.
     * A {@code null} image value or an image still in progress will be ignored.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param img the image to be drawn or null.
     * @param x the X coordinate on the destination for the upper left of the image.
     * @param y the Y coordinate on the destination for the upper left of the image.
     */
//    public void drawImage(Image img, double x, double y);

    /**
     * Draws an image into the given destination rectangle of the canvas. The
     * Image is scaled to fit into the destination rectagnle.
     * A {@code null} image value or an image still in progress will be ignored.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param img the image to be drawn or null.
     * @param x the X coordinate on the destination for the upper left of the image.
     * @param y the Y coordinate on the destination for the upper left of the image.
     * @param w the width of the destination rectangle.
     * @param h the height of the destination rectangle.
     */
//    public void drawImage(Image img, double x, double y, double w, double h);

    /**
     * Draws the specified source rectangle of the given image to the given
     * destination rectangle of the Canvas.
     * A {@code null} image value or an image still in progress will be ignored.
     * <p>
     * This method will be affected by any of the
     * <a href="#comm-attr">global common</a>
     * attributes as specified in the
     * <a href="#attr-ops-table">Rendering Attributes Table</a>.
     * </p>
     *
     * @param img the image to be drawn or null.
     * @param sx the source rectangle's X coordinate position.
     * @param sy the source rectangle's Y coordinate position.
     * @param sw the source rectangle's width.
     * @param sh the source rectangle's height.
     * @param dx the destination rectangle's X coordinate position.
     * @param dy the destination rectangle's Y coordinate position.
     * @param dw the destination rectangle's width.
     * @param dh the destination rectangle's height.
     */
//    public void drawImage(Image img,
//                          double sx, double sy, double sw, double sh,
//                          double dx, double dy, double dw, double dh);
}