/*
 * Copyright (c) 2011 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.colormap

import com.macrofocus.common.collection.TreeSet
import com.macrofocus.common.interval.MutableInterval
import com.macrofocus.common.interval.PropertyClosedInterval
import org.kamaeleo.color.CPColorFactory
import org.kamaeleo.palette.CustomPalette
import org.kamaeleo.palette.MutablePalette
import org.kamaeleo.palette.PaletteFactory
import kotlin.math.max

/**
 * Factory class for creating Colormap of different types.
 */
class ColorMapFactory(paletteFactory: PaletteFactory) {
    private val colorFactory = CPColorFactory.instance
    private val paletteFactory: PaletteFactory
    fun getPaletteFactory(): PaletteFactory {
        return paletteFactory
    }

    fun createContinuousSequentialColorMap(min: Double, max: Double): MutableColorMap {
        return createContinuousColorMap(min, max, paletteFactory.createDefaultSequentialPalette())
    }

    fun createContinuousDivergingColorMap(min: Double, max: Double): MutableColorMap {
        return createContinuousColorMap(min, max, paletteFactory.createDefaultDivergingPalette())
    }

    fun createCategoricalSequentialColorMap(values: Set<*>?): MutableColorMap {
        return createCategoricalColorMap(values, false, paletteFactory.createDefaultSequentialPalette())
    }

    fun createCategoricalDivergingColorMap(values: Set<*>?): MutableColorMap {
        return createCategoricalColorMap(values, false, paletteFactory.createDefaultDivergingPalette())
    }

    fun createQualitativeColorMap(values: Set<*>?): MutableColorMap {
        return createCategoricalColorMap(values, true, paletteFactory.createDefaultQualititativePalette())
    }

    fun createContinuousColorMap(min: Double, max: Double, palette: MutablePalette): MutableColorMap {
        return SimpleColorMap(PropertyClosedInterval(min, max - min), palette)
    }

    fun createCategoricalColorMap(values: Set<*>?, cyclic: Boolean, palette: MutablePalette): MutableColorMap {
        return SimpleColorMap(values, cyclic, palette)
    }

    fun createColorMap(interval: MutableInterval, palette: MutablePalette): MutableColorMap {
        return SimpleColorMap(interval, palette)
    }

    fun createAlphabeticalColorMap(values: Set<*>?): MutableColorMap {
        val set = TreeSet<String>(String.CASE_INSENSITIVE_ORDER)
        set.add("A")
        set.add("B")
        set.add("C")
        return SimpleColorMap(set, false, paletteFactory.createDefaultSequentialPalette())
    }

    fun createAutoContinuousColorMap(mininum: Number?, maximum: Number?): MutableColorMap {
        return if (mininum != null && maximum != null) {
            if (mininum.toDouble() < 0 && maximum.toDouble() > 0) {
                val diffMin = 0 - mininum.toDouble()
                val diffMax = maximum.toDouble()
                val maxDiff: Double = max(diffMin, diffMax)
                createContinuousDivergingColorMap(0 - maxDiff, maxDiff)
            } else {
                createContinuousSequentialColorMap(mininum.toDouble(), maximum.toDouble())
            }
        } else {
            createContinuousSequentialColorMap(0.0, 0.0)
        }
    }

    fun createAutoSegmentedColorMap(mininum: Number?, maximum: Number?): MutableColorMap {
        return if (mininum != null && maximum != null) {
            if (mininum.toDouble() < 0 && maximum.toDouble() > 0) {
                val diffMin = 0 - mininum.toDouble()
                val diffMax = maximum.toDouble()
                val maxDiff: Double = max(diffMin, diffMax)
                val min = 0 - maxDiff
                val minF = (min - mininum.toDouble()) / (maximum.toDouble() - mininum.toDouble())
                val maxF = (maxDiff - mininum.toDouble()) / (maximum.toDouble() - mininum.toDouble())
                val meanF = (maxF + minF) / 2.0
                createContinuousColorMap(
                    mininum.toDouble(),
                    maximum.toDouble(),
                    CustomPalette(
                        CustomPalette.Entry(minF, colorFactory.createRGBColor(255, 0, 0)),
                        CustomPalette.Entry(meanF, colorFactory.white),
                        CustomPalette.Entry(maxF, colorFactory.createRGBColor(0, 128, 0))
                    )
                )
            } else {
                createContinuousColorMap(
                    mininum.toDouble(),
                    maximum.toDouble(),
                    CustomPalette(
                        CustomPalette.Entry(0.0, colorFactory.createRGBColor(255, 0, 0)),
                        CustomPalette.Entry(0.5, colorFactory.white),
                        CustomPalette.Entry(1.0, colorFactory.createRGBColor(0, 128, 0))
                    )
                )
            }
        } else {
            createContinuousColorMap(0.0, 0.0, CustomPalette(
                CustomPalette.Entry(0.0, colorFactory.white)
            ))
        }
    }

    init {
        this.paletteFactory = paletteFactory
    }
}