/*
 * Copyright (c) 2011 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.colormap

import com.macrofocus.common.collection.TreeMap
import com.macrofocus.common.collection.TreeSet
import com.macrofocus.common.collection.UniversalComparator
import com.macrofocus.common.interval.IntervalEvent
import com.macrofocus.common.interval.IntervalListener
import com.macrofocus.common.interval.MutableInterval
import com.macrofocus.common.properties.*
import org.kamaeleo.color.CPColor
import org.kamaeleo.color.CPColorFactory
import org.kamaeleo.color.alpha
import org.kamaeleo.color.brightenAndSaturate
import org.kamaeleo.colormap.ColorMap.Assignments
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_ALPHA
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_ASSIGNMENTS
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_BRIGHTNESS
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_COLORCOUNT
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_INTERVAL
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_INVERTED
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_MATCHING
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_NULLCOLOR
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_OVERCOLOR
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_PALETTE
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_SATURATION
import org.kamaeleo.colormap.ColorMap.Companion.PROPERTY_UNDERCOLOR
import org.kamaeleo.palette.*

/**
 * Default data model for color maps.
 */
class SimpleColorMap private constructor() : AbstractColorMap(),
    MutableColorMap {
    enum class PropertyType {
        Inverval, Palette, Matching, Assignments, MissingValuesColor, UnderflowColor, OverflowColor, Inverted, NumberOfSteps, Brightness, Saturation, Alpha
    }

    protected val properties: MutableProperties<PropertyType> = EnumProperties(
        PropertyType.values()
    )
    override var interval: MutableInterval? = null
        set(interval) {
            if (field !== interval) {
                val old: MutableInterval? = this.interval
                if (field != null) {
                    field!!.removeIntervalListener(intervalListener)
                }
                field = interval
                if (field != null) {
                    field!!.addIntervalListener(intervalListener)
                }
                notifyColorMapChanged(ColorMapEvent(this, PROPERTY_INTERVAL, old, interval))
            }
        }
    override var paletteProperty: MutableProperty<MutablePalette?> =
        properties.createProperty<MutablePalette?>(PropertyType.Palette, null)
    override var palette by PropertyDelegates.mutable(paletteProperty)

    private val dictionary: TreeMap<Any?, CPColor>
    private var _matching: ColorMap.Matching = ColorMap.Matching.EXACT
    private var _assignments: Assignments = Assignments.STATIC
    override val nullColorProperty: MutableProperty<CPColor?> = properties.createProperty(PropertyType.MissingValuesColor, null)
    override var nullColor by PropertyDelegates.mutable(nullColorProperty)
    override var underColorSetProperty: MutableProperty<Boolean> = SimpleProperty(false)
    override var underColorStoreProperty: MutableProperty<CPColor?> = SimpleProperty(null)
    override val underColorProperty: MutableProperty<CPColor?> = properties.addProperty<CPColor?>(
        PropertyType.UnderflowColor,
        SetStoreMutableProperty(
            underColorSetProperty,
            underColorStoreProperty, paletteProperty, 0.0
        )
    )
    override var overColorSetProperty: MutableProperty<Boolean> = SimpleProperty(false)
    override var overColorStoreProperty: MutableProperty<CPColor?> = SimpleProperty(null)
    override val overColorProperty: MutableProperty<CPColor?> = properties.addProperty<CPColor?>(
        PropertyType.OverflowColor,
        SetStoreMutableProperty(
            overColorSetProperty,
            overColorStoreProperty, paletteProperty, 1.0
        )
    )
    override var isInvertedProperty: MutableProperty<Boolean> = properties.createProperty(PropertyType.Inverted, false)
    override var isInverted by PropertyDelegates.mutable(isInvertedProperty)

    override var colorCountSetProperty: MutableProperty<Boolean> = SimpleProperty(false)

    override var colorCountProperty: MutableProperty<Int> = properties.createProperty(PropertyType.NumberOfSteps, 3)
    override var colorCount by PropertyDelegates.mutable(colorCountProperty)

    override var brightnessProperty: MutableProperty<Int> = properties.createProperty(PropertyType.Brightness, 0)
    override var brightness by PropertyDelegates.mutable(brightnessProperty)

    override var saturationProperty: MutableProperty<Int> = properties.createProperty(PropertyType.Saturation, 0)
    override var saturation by PropertyDelegates.mutable(saturationProperty)

    override var alpha = 1f
        set(alpha) {
            if (field != alpha) {
                val old = field
                field = alpha
                notifyColorMapChanged(ColorMapEvent(this, PROPERTY_ALPHA, old, alpha))
            }
        }
    private val factory: CPColorFactory = CPColorFactory.instance
    private val intervalListener: IntervalListener = object : IntervalListener {
        override fun intervalChanged(event: IntervalEvent?) {
            notifyColorMapChanged(ColorMapEvent(this@SimpleColorMap, PROPERTY_INTERVAL, null, null))
        }
    }
    private val paletteListener: PaletteListener = object : PaletteListener {
        override fun paletteChanged(event: PaletteEvent) {
            notifyColorMapChanged(ColorMapEvent(this@SimpleColorMap, PROPERTY_PALETTE, null, null))
        }
    }

    constructor(interval: MutableInterval, palette: MutablePalette) : this(
    ) {
        this.interval = interval
        this.palette = palette
    }

    constructor(
        values: Set<*>?,
        cyclic: Boolean,
        palette: MutablePalette
    ) : this() {
        this.palette = palette
        values?.let { assignColors(it, cyclic) }
    }

    override fun getColor(value: Any?): CPColor? {
        var color: CPColor?
        if (value != null) {
            color = factory.createNativeColor(value)
            if (color != null) {
                return color
            } else if (value is CPColor) {
                return value as CPColor?
            } else if (value is Comparable<*> && dictionary.containsKey(value)) {
                color = dictionary.get(value)

                // ToDo: Uncomment as soon as this is supported by GWT (see https://gwt-review.googlesource.com/#/c/3650/)
//            } else if (value instanceof Comparable && _matching == Matching.CLOSEST && dictionary.lowerKey(value) != null) {
//                color = dictionary.lowerEntry(value).getValue();
            } else {
                if (value is Number && interval != null) {
                    val v = value.toDouble()
                    var fraction: Double
                    fraction = if (!interval!!.isDegenerate) {
                        (v - interval!!.start) / interval!!.extent
                    } else {
                        0.5
                    }
                    if (isInverted) {
                        fraction = 1.0 - fraction
                    }
                    if (fraction < lowestFraction) {
                        return underflowColor
                    } else if (fraction > highestFraction) {
                        return overflowColor
                    } else {
                        val colorCount = if (colorCountSetProperty!!.value) colorCount else 0
                        if (colorCount > 0) {
                            fraction = if (colorCount > 1) {
                                (fraction * colorCount).toInt() / (colorCount - 1).toDouble()
                            } else {
                                0.5
                            }
                        }
                        color = paletteProperty!!.value!!.getColor(fraction)
                    }
                    // ToDo: reimplement!
//                } else if (value is java.util.Date && interval != null) {
//                    val v: Long = (value as java.util.Date).getTime()
//                    var fraction: Double = (v - interval!!.start) / interval!!.extent
//                    if (isInverted()) {
//                        fraction = 1.0 - fraction
//                    }
//                    if (fraction < lowestFraction) {
//                        return underflowColor
//                    } else if (fraction > highestFraction) {
//                        return overflowColor
//                    } else {
//                        val colorCount = if (colorCountSet!!.value) getColorCount() else 0
//                        if (colorCount > 0) {
//                            if (colorCount > 1) {
//                                fraction = (fraction * colorCount).toInt() / (colorCount - 1).toDouble()
//                            }
//                        } else {
//                            fraction = 0.5
//                        }
//                        color = paletteProperty.value!!.getColor(fraction)!!
//                    }
                } else if (value is Array<*>) {
                    val str = arrayToString(value)
                    color = if (dictionary.containsKey(str)) {
                        dictionary.get(str)
                    } else {
                        nullColor
                    }
                } else {
                    color = if (_assignments === Assignments.DYNAMIC) {
                        val c: CPColor =
                            paletteProperty.value!!.getColorAt(dictionary.size % paletteProperty.value!!.colorCount)
                        dictionary.put(value, c)
                        c
                    } else {
//                    return palette.getColor(value.hashCode() % palette.getColorCount());
                        nullColor
                    }
                }
            }
        } else {
            color = nullColor
        }
        if (color != null && (brightnessProperty!!.value.toInt() != 0 || saturationProperty!!.value.toInt() != 0)) {
            color = color.brightenAndSaturate(
                brightnessProperty!!.value.toInt() / 300f,
                saturationProperty!!.value.toInt() / 300f
            )
        }
        if (color != null && alpha != 1f) {
            color = color.alpha(alpha)
        }
        return color
    }

    private fun arrayToString(array: Array<*>): String {
        var str = ""
        for (i in array.indices) {
            val v = array[i]
            if (v != null) {
                str += v.toString()
            }
            if (i < array.size - 1) {
                str += "/"
            }
        }
        return str
    }

    protected val lowestFraction: Double
        protected get() {
            if (paletteProperty!!.value is InterpolatedPalette) {
                val entries: TreeSet<InterpolatedPalette.Entry> =
                    (paletteProperty!!.value as InterpolatedPalette).getEntries() as TreeSet
                if (!entries.isEmpty()) {
                    return entries.first().fraction
                }
            } else if (paletteProperty!!.value is CustomPalette) {
                val entries: TreeSet<CustomPalette.Entry> = (palette as CustomPalette).getEntries() as TreeSet
                if (!entries.isEmpty()) {
                    return entries.first().fraction
                }
            }
            return 0.0
        }
    protected val highestFraction: Double
        protected get() {
            if (paletteProperty!!.value is InterpolatedPalette) {
                val entries: TreeSet<InterpolatedPalette.Entry> =
                    (paletteProperty!!.value as InterpolatedPalette).getEntries() as TreeSet
                if (!entries.isEmpty()) {
                    return entries.last().fraction
                }
            } else if (paletteProperty!!.value is CustomPalette) {
                val entries: TreeSet<CustomPalette.Entry> = (palette as CustomPalette).getEntries() as TreeSet
                if (!entries.isEmpty()) {
                    return entries.last().fraction
                }
            }
            return 1.0
        }

    override fun setMatching(matching: ColorMap.Matching?) {
        if (_matching !== matching) {
            val old: ColorMap.Matching = _matching
            _matching = matching!!
            notifyColorMapChanged(ColorMapEvent(this, PROPERTY_MATCHING, old, matching))
        }
    }

    override fun setAssignments(assignments: Assignments?) {
        if (_assignments !== assignments) {
            val old: Assignments = _assignments
            _assignments = assignments!!
            notifyColorMapChanged(ColorMapEvent(this, PROPERTY_ASSIGNMENTS, old, assignments))
        }
    }

    override val underflowColor: CPColor
        get() = underColorProperty.value!!

    override fun setUnderColor(underColor: CPColor?) {
        if (underflowColor !== underColor && (underflowColor == null || !underflowColor.equals(underColor))) {
            this.underColorProperty.value = underColor!!
        }
    }

    override val overflowColor: CPColor
        get() = overColorStoreProperty!!.value!!

    override fun setOverColor(overColor: CPColor?) {
        if (overflowColor !== overColor && (overflowColor == null || !overflowColor.equals(overColor))) {
            this.overColorStoreProperty.value = overColor
        }
    }

    override val isOverflowColorSet: Boolean
        get() = overColorSetProperty!!.value
    override val isUnderflowColorSet: Boolean
        get() = underColorSetProperty!!.value

    // ToDo: Reimplement
//    fun getColorCount(): Int {
//        return if (colorCountSetProperty!!.value) {
//            val colorCount = colorCountProperty!!.value
//            if (colorCount > 0 && colorCount < 256) {
//                colorCount
//            } else {
//                0
//            }
//        } else {
//            0
//        }
//    }

    // ToDo: Reimplement
//    override fun setColorCount(colorCount: Int) {
//        val set = colorCount > 0 && colorCount < 256
//        colorCountSet!!.value = set
//        if (set) {
//            this.colorCountProperty!!.value = colorCount
//        }
//    }

    override fun assignColors(values: Set<*>, cyclic: Boolean) {
        dictionary.clear()
        var i = 0
        for (value in values) {
            val color: CPColor?
            color = if (!cyclic) {
                paletteProperty.value!!.getColor(i / values.size.toDouble())
            } else {
                paletteProperty.value!!.getColorAt(i % paletteProperty.value!!.colorCount)
            }
            var v: Any?
            v = if (value is Array<*>) {
                arrayToString(value)
            } else {
                value
            }
            dictionary.put(v, color!!)
            i++
        }
        notifyColorMapChanged(ColorMapEvent(this, PROPERTY_ASSIGNMENTS, null, null))
    }

    override fun assignColors(vararg values: Any?) {
        dictionary.clear()
        var i = 0
        for (value in values) {
            if (!dictionary.containsKey(value)) {
                dictionary.put(value, paletteProperty!!.value!!.getColorAt(i % paletteProperty!!.value!!.colorCount))
                i++
            }
        }
        notifyColorMapChanged(ColorMapEvent(this, PROPERTY_ASSIGNMENTS, null, null))
    }

    override fun assignColors(values: Iterable<Any?>) {
        dictionary.clear()
        var i = 0
        for (value in values) {
            if (!dictionary.containsKey(value)) {
                dictionary.put(value, paletteProperty!!.value!!.getColorAt(i % paletteProperty!!.value!!.colorCount))
                i++
            }
        }
        notifyColorMapChanged(ColorMapEvent(this, PROPERTY_ASSIGNMENTS, null, null))
    }

    override fun setColor(value: Any?, color: CPColor?) {
        dictionary.put(value, color!!)
        notifyColorMapChanged(ColorMapEvent(this, PROPERTY_ASSIGNMENTS, null, null))
    }

    override fun clearAssignedColor() {
        dictionary.clear()
        notifyColorMapChanged(ColorMapEvent(this, PROPERTY_ASSIGNMENTS, null, null))
    }

    override val assignedValues: Set<Any?>?
        get() = dictionary.keys

    override fun resetAssignedValues() {
        val set: MutableSet<Any?> = TreeSet(UniversalComparator())
        set.addAll(dictionary.keys)
        assignColors(set, true)
    }

    override fun setProperty(property: String?, value: Any?) {
        if (property == PROPERTY_INTERVAL) {
            interval = value as MutableInterval
        } else if (property == PROPERTY_PALETTE) {
            palette = value as MutablePalette
        } else if (property == PROPERTY_MATCHING) {
            setMatching(value as ColorMap.Matching)
        } else if (property == PROPERTY_ASSIGNMENTS) {
            setAssignments(value as Assignments)
        } else if (property == PROPERTY_NULLCOLOR) {
            nullColor = value as CPColor
        } else if (property == PROPERTY_UNDERCOLOR) {
            setUnderColor(value as CPColor)
        } else if (property == PROPERTY_OVERCOLOR) {
            setOverColor(value as CPColor)
        } else if (property == PROPERTY_INVERTED) {
            isInverted = value as Boolean
        } else if (property == PROPERTY_COLORCOUNT) {
            colorCount = value as Int
        } else if (property == PROPERTY_BRIGHTNESS) {
            brightness = value as Int
        } else if (property == PROPERTY_SATURATION) {
            saturation = value as Int
        } else if (property == PROPERTY_ALPHA) {
            alpha = value as Float
        } else {
            println("Unknonw property $property")
        }
    }

    override fun toString(): String {
        return "SimpleColorMap{" +
                "interval=" + interval +
                ", palette=" + palette +
                ", nullColor=" + nullColor +
                ", underColor=" + underColorProperty +
//                ", overColor=" + overColor +
                '}'
    }

    init {
        properties.createProperty<Any?>(PropertyType.Inverval, null)
        properties.createProperty<Any>(PropertyType.Matching, ColorMap.Matching.EXACT)
        properties.createProperty<Any>(PropertyType.Assignments, Assignments.STATIC)

    }

    init {
        paletteProperty.addPropertyListener(object : PropertyListener<MutablePalette?> {
            override fun propertyChanged(event: PropertyEvent<MutablePalette?>) {
                if (event.oldValue != null) {
                    event.oldValue!!.removePaletteListener(paletteListener)
                }
                if (event.newValue != null) {
                    event.newValue!!.addPaletteListener(paletteListener)
                }
                notifyColorMapChanged(
                    ColorMapEvent(
                        this@SimpleColorMap,
                        PROPERTY_PALETTE,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
        isInvertedProperty.addPropertyListener(object : PropertyListener<Boolean> {
            override fun propertyChanged(event: PropertyEvent<Boolean>) {
                notifyColorMapChanged(
                    ColorMapEvent(
                        this@SimpleColorMap,
                        PROPERTY_INVERTED,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
        brightnessProperty.addPropertyListener(object : PropertyListener<Int> {
            override fun propertyChanged(event: PropertyEvent<Int>) {
                notifyColorMapChanged(
                    ColorMapEvent(
                        this@SimpleColorMap,
                        PROPERTY_BRIGHTNESS,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
        saturationProperty.addPropertyListener(object : PropertyListener<Int> {
            override fun propertyChanged(event: PropertyEvent<Int>) {
                notifyColorMapChanged(
                    ColorMapEvent(
                        this@SimpleColorMap,
                        PROPERTY_SATURATION,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
        overColorProperty.addPropertyListener(object : PropertyListener<CPColor?> {
            override fun propertyChanged(event: PropertyEvent<CPColor?>) {
                notifyColorMapChanged(
                    ColorMapEvent(
                        this@SimpleColorMap,
                        PROPERTY_OVERCOLOR,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
        underColorProperty.addPropertyListener(object : PropertyListener<CPColor?> {
            override fun propertyChanged(event: PropertyEvent<CPColor?>) {
                notifyColorMapChanged(
                    ColorMapEvent(
                        this@SimpleColorMap,
                        PROPERTY_UNDERCOLOR,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
        nullColorProperty.addPropertyListener(object : PropertyListener<CPColor?> {
            override fun propertyChanged(event: PropertyEvent<CPColor?>) {
                notifyColorMapChanged(
                    ColorMapEvent(
                        this@SimpleColorMap,
                        PROPERTY_NULLCOLOR,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
        colorCountSetProperty.addPropertyListener(object : PropertyListener<Boolean> {
            override fun propertyChanged(event: PropertyEvent<Boolean>) {
                notifyColorMapChanged(
                    ColorMapEvent(
                        this@SimpleColorMap,
                        PROPERTY_COLORCOUNT,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
        colorCountProperty.addPropertyListener(object : PropertyListener<Int> {
            override fun propertyChanged(event: PropertyEvent<Int>) {
                notifyColorMapChanged(
                    ColorMapEvent(
                        this@SimpleColorMap,
                        PROPERTY_COLORCOUNT,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
        nullColorProperty.value = factory.gray
        dictionary = TreeMap<Any?, CPColor>()
    }
}

