/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

import com.macrofocus.common.math.convertDegreesToRadians
import com.macrofocus.common.math.convertRadiansToDegrees
import kotlin.math.*

interface Arc2D : Shape {
    val arcType: Int
    val x: kotlin.Double
    val y: kotlin.Double
    val width: kotlin.Double
    val height: kotlin.Double

    //    public double minX;
    //
    //    public double minY;
    //
    //    public double maxX;
    //
    //    public double maxY;
    //
    val centerX: kotlin.Double
    val centerY: kotlin.Double
    val angleStart: kotlin.Double
    val angleExtent: kotlin.Double
    val startPoint: Point2D
    val endPoint: Point2D

    class Double : Arc2D {
        /**
         * The X coordinate of the upper-left corner of the framing
         * rectangle of the arc.
         */
        override val x: kotlin.Double

        /**
         * The Y coordinate of the upper-left corner of the framing
         * rectangle of the arc.
         */
        override val y: kotlin.Double

        /**
         * The overall width of the full ellipse of which this arc is
         * a partial section (not considering the
         * angular extents).
         */
        override val width: kotlin.Double

        /**
         * The overall height of the full ellipse of which this arc is
         * a partial section (not considering the
         * angular extents).
         */
        override val height: kotlin.Double

        /** The starting angle of the arc in degrees.  */
        private val start: kotlin.Double

        /** The angular extent of the arc in degrees.  */
        private val extent: kotlin.Double
        override val arcType: Int

        constructor(bounds: Rectangle2D, start: kotlin.Double, extent: kotlin.Double, type: Int) {
            x = bounds.x
            y = bounds.y
            width = bounds.width
            height = bounds.height
            this.start = start
            this.extent = extent
            this.arcType = type
        }

        constructor(
            x: kotlin.Double,
            y: kotlin.Double,
            width: kotlin.Double,
            height: kotlin.Double,
            start: kotlin.Double,
            extent: kotlin.Double,
            type: Int
        ) {
            this.x = x
            this.y = y
            this.width = width
            this.height = height
            this.start = start
            this.extent = extent
            this.arcType = type
        }

        override val bounds2D: Rectangle2D
            get() = computeBounds2D()

        /**
         * Returns the high-precision framing rectangle of the arc.  The framing
         * rectangle contains only the part of this `Arc2D` that is
         * in between the starting and ending angles and contains the pie
         * wedge, if this `Arc2D` has a `PIE` closure type.
         *
         *
         * This method differs from the
         * [getBounds][RectangularShape.getBounds] in that the
         * `getBounds` method only returns the bounds of the
         * enclosing ellipse of this `Arc2D` without considering
         * the starting and ending angles of this `Arc2D`.
         *
         * @return the <CODE>Rectangle2D</CODE> that represents the arc's
         * framing rectangle.
         *
         * @since 1.2
         */
        fun computeBounds2D(): Rectangle2D {
            if (isEmpty) {
                return Rectangle2D.Double(x, y, width, height)
            }
            var x1: kotlin.Double
            var y1: kotlin.Double
            var x2: kotlin.Double
            var y2: kotlin.Double
            if (arcType == PIE) {
                y2 = 0.0
                x2 = y2
                y1 = x2
                x1 = y1
            } else {
                y1 = 1.0
                x1 = y1
                y2 = -1.0
                x2 = y2
            }
            var angle = 0.0
            for (i in 0..5) {
                if (i < 4) {
                    // 0-3 are the four quadrants
                    angle += 90.0
                    if (!containsAngle(angle)) {
                        continue
                    }
                } else if (i == 4) {
                    // 4 is start angle
                    angle = angleStart
                } else {
                    // 5 is end angle
                    angle += angleExtent
                }
                val rads: kotlin.Double = convertDegreesToRadians(-angle)
                val xe: kotlin.Double = cos(rads)
                x1 = min(x1, xe)
                val ye: kotlin.Double = sin(rads)
                y1 = min(y1, ye)
                x2 = max(x2, xe)
                y2 = max(y2, ye)
            }
            val w = width
            val h = height
            x2 = (x2 - x1) * 0.5 * w
            y2 = (y2 - y1) * 0.5 * h
            x1 = x + (x1 * 0.5 + 0.5) * w
            y1 = y + (y1 * 0.5 + 0.5) * h
            return Rectangle2D.Double(x1, y1, x2, y2)
        }

        val isEmpty: Boolean
            get() = width <= 0.0 || height <= 0.0

        /**
         * Determines whether or not the specified point is inside the boundary
         * of the arc.
         *
         * @param x The X coordinate of the point to test.
         * @param y The Y coordinate of the point to test.
         *
         * @return <CODE>true</CODE> if the point lies within the bound of
         * the arc, <CODE>false</CODE> if the point lies outside of the
         * arc's bounds.
         */
        override operator fun contains(p: Point2D): Boolean {
            val x: kotlin.Double = p.x
            val y: kotlin.Double = p.y

            // Normalize the coordinates compared to the ellipse
            // having a center at 0,0 and a radius of 0.5.
            val ellw = width
            if (ellw <= 0.0) {
                return false
            }
            val normx = (x - x) / ellw - 0.5
            val ellh = height
            if (ellh <= 0.0) {
                return false
            }
            val normy = (y - y) / ellh - 0.5
            val distSq = normx * normx + normy * normy
            if (distSq >= 0.25) {
                return false
            }
            val angExt: kotlin.Double = abs(angleExtent)
            if (angExt >= 360.0) {
                return true
            }
            val inarc = containsAngle(
                -convertRadiansToDegrees(
                    atan2(
                        normy,
                        normx
                    )
                )
            )
            if (arcType == PIE) {
                return inarc
            }
            // CHORD and OPEN behave the same way
            if (inarc) {
                if (angExt >= 180.0) {
                    return true
                }
                // point must be outside the "pie triangle"
            } else {
                if (angExt <= 180.0) {
                    return false
                }
                // point must be inside the "pie triangle"
            }
            // The point is inside the pie triangle iff it is on the same
            // side of the line connecting the ends of the arc as the center.
            var angle: kotlin.Double = convertDegreesToRadians(-angleStart)
            val x1: kotlin.Double = cos(angle)
            val y1: kotlin.Double = sin(angle)
            angle += convertDegreesToRadians(-angleExtent)
            val x2: kotlin.Double = cos(angle)
            val y2: kotlin.Double = sin(angle)
            val inside: Boolean = Line2D.relativeCCW(x1, y1, x2, y2, 2 * normx, 2 * normy) *
                    Line2D.relativeCCW(x1, y1, x2, y2, 0.0, 0.0) >= 0
            return inarc != inside
        }

        /**
         * Determines whether or not the interior of the arc intersects
         * the interior of the specified rectangle.
         *
         * @param x The X coordinate of the rectangle's upper-left corner.
         * @param y The Y coordinate of the rectangle's upper-left corner.
         * @param w The width of the rectangle.
         * @param h The height of the rectangle.
         *
         * @return <CODE>true</CODE> if the arc intersects the rectangle,
         * <CODE>false</CODE> if the arc doesn't intersect the rectangle.
         */
        override fun intersects(r: Rectangle2D): Boolean {
            val x: kotlin.Double = r.x
            val y: kotlin.Double = r.y
            val w: kotlin.Double = r.width
            val h: kotlin.Double = r.height
            val aw = width
            val ah = height
            if (w <= 0 || h <= 0 || aw <= 0 || ah <= 0) {
                return false
            }
            val ext = angleExtent
            if (ext == 0.0) {
                return false
            }
            val ax = x
            val ay = y
            val axw = ax + aw
            val ayh = ay + ah
            val xw = x + w
            val yh = y + h

            // check bbox
            if (x >= axw || y >= ayh || xw <= ax || yh <= ay) {
                return false
            }

            // extract necessary data
            val axc = centerX
            val ayc = centerY
            val sp: Point2D = startPoint
            val ep: Point2D = endPoint
            val sx: kotlin.Double = sp.x
            val sy: kotlin.Double = sp.y
            val ex: kotlin.Double = ep.x
            val ey: kotlin.Double = ep.y

            /*
             * Try to catch rectangles that intersect arc in areas
             * outside of rectagle with left top corner coordinates
             * (min(center x, start point x, end point x),
             *  min(center y, start point y, end point y))
             * and rigth bottom corner coordinates
             * (max(center x, start point x, end point x),
             *  max(center y, start point y, end point y)).
             * So we'll check axis segments outside of rectangle above.
             */if (ayc >= y && ayc <= yh) { // 0 and 180
                if (sx < xw && ex < xw && axc < xw && axw > x && containsAngle(0.0) ||
                    sx > x && ex > x && axc > x && ax < xw && containsAngle(180.0)
                ) {
                    return true
                }
            }
            if (axc >= x && axc <= xw) { // 90 and 270
                if (sy > y && ey > y && ayc > y && ay < yh && containsAngle(90.0) ||
                    sy < yh && ey < yh && ayc < yh && ayh > y && containsAngle(270.0)
                ) {
                    return true
                }
            }

            /*
             * For PIE we should check intersection with pie slices;
             * also we should do the same for arcs with extent is greater
             * than 180, because we should cover case of rectangle, which
             * situated between center of arc and chord, but does not
             * intersect the chord.
             */
            val rect: Rectangle2D = Rectangle2D.Double(x, y, w, h)
            if (arcType == PIE || abs(ext) > 180) {
                // for PIE: try to find intersections with pie slices
                if (rect.intersectsLine(axc, ayc, sx, sy) ||
                    rect.intersectsLine(axc, ayc, ex, ey)
                ) {
                    return true
                }
            } else {
                // for CHORD and OPEN: try to find intersections with chord
                if (rect.intersectsLine(sx, sy, ex, ey)) {
                    return true
                }
            }

            // finally check the rectangle corners inside the arc
            return contains(Point2D.Double(x, y)) || contains(Point2D.Double(x + w, y)) ||
                    contains(Point2D.Double(x, y + h)) || contains(Point2D.Double(x + w, y + h))
        }

        override val pathIterator: PathIterator
            get() = getPathIterator(null)

        /**
         * Returns an iteration object that defines the boundary of the
         * arc.
         * This iterator is multithread safe.
         * `Arc2D` guarantees that
         * modifications to the geometry of the arc
         * do not affect any iterations of that geometry that
         * are already in process.
         *
         * @param at an optional <CODE>AffineTransform</CODE> to be applied
         * to the coordinates as they are returned in the iteration, or null
         * if the untransformed coordinates are desired.
         *
         * @return A <CODE>PathIterator</CODE> that defines the arc's boundary.
         *
         * @since 1.2
         */
        override fun getPathIterator(at: AffineTransform?): PathIterator {
            return ArcIterator(this, at)
        }

        override fun getFlattenPathIterator(flatness: kotlin.Double): PathIterator {
            return FlatteningPathIterator(pathIterator, flatness)
        }

        override val centerX: kotlin.Double
            get() = x + width / 2.0
        override val centerY: kotlin.Double
            get() = y + height / 2.0

        override val angleStart: kotlin.Double
            get() = start

        override val angleExtent: kotlin.Double
            get() = extent

        override val startPoint: Point2D
            get() = computeStartPoint()
        /**
         * Returns the starting point of the arc.  This point is the
         * intersection of the ray from the center defined by the
         * starting angle and the elliptical boundary of the arc.
         *
         * @return A <CODE>Point2D</CODE> object representing the
         * x,y coordinates of the starting point of the arc.
         */
        fun computeStartPoint(): Point2D {
            val angle: kotlin.Double = convertDegreesToRadians(-angleStart)
            val x: kotlin.Double = x + (cos(angle) * 0.5 + 0.5) * width
            val y: kotlin.Double = y + (sin(angle) * 0.5 + 0.5) * height
            return Point2D.Double(x, y)
        }

        override val endPoint: Point2D
            get() = computeEndPoint()

        /**
         * Returns the ending point of the arc.  This point is the
         * intersection of the ray from the center defined by the
         * starting angle plus the angular extent of the arc and the
         * elliptical boundary of the arc.
         *
         * @return A <CODE>Point2D</CODE> object representing the
         * x,y coordinates  of the ending point of the arc.
         */
        fun computeEndPoint(): Point2D {
            val angle: kotlin.Double = convertDegreesToRadians(-angleStart - angleExtent)
            val x: kotlin.Double = x + (cos(angle) * 0.5 + 0.5) * width
            val y: kotlin.Double = y + (sin(angle) * 0.5 + 0.5) * height
            return Point2D.Double(x, y)
        }

        /**
         * Determines whether or not the specified angle is within the
         * angular extents of the arc.
         *
         * @param angle The angle to test.
         *
         * @return <CODE>true</CODE> if the arc contains the angle,
         * <CODE>false</CODE> if the arc doesn't contain the angle.
         */
        fun containsAngle(angle: kotlin.Double): Boolean {
            var angle = angle
            var angExt = angleExtent
            val backwards = angExt < 0.0
            if (backwards) {
                angExt = -angExt
            }
            if (angExt >= 360.0) {
                return true
            }
            angle = normalizeDegrees(angle) - normalizeDegrees(angleStart)
            if (backwards) {
                angle = -angle
            }
            if (angle < 0.0) {
                angle += 360.0
            }
            return angle >= 0.0 && angle < angExt
        }

        companion object {
            fun getArcByCenter(
                x: kotlin.Double, y: kotlin.Double, radius: kotlin.Double,
                angSt: kotlin.Double, angExt: kotlin.Double, closure: Int
            ): Double {
                return Double(
                    x - radius, y - radius, radius * 2.0, radius * 2.0,
                    angSt, angExt, closure
                )
            }

            /*
         * Normalizes the specified angle into the range -180 to 180.
         */
            fun normalizeDegrees(angle: kotlin.Double): kotlin.Double {
                var angle = angle
                if (angle > 180.0) {
                    if (angle <= 180.0 + 360.0) {
                        angle = angle - 360.0
                    } else {
                        angle = IEEEremainder(angle, 360.0)
                        // IEEEremainder can return -180 here for some input values...
                        if (angle == -180.0) {
                            angle = 180.0
                        }
                    }
                } else if (angle <= -180.0) {
                    if (angle > -180.0 - 360.0) {
                        angle = angle + 360.0
                    } else {
                        angle = IEEEremainder(angle, 360.0)
                        // IEEEremainder can return -180 here for some input values...
                        if (angle == -180.0) {
                            angle = 180.0
                        }
                    }
                }
                return angle
            }

            fun IEEEremainder(f1: kotlin.Double, f2: kotlin.Double): kotlin.Double {
                val r: kotlin.Double = abs(f1 % f2)
                return if (r.isNaN() || r == f2 || r <= abs(f2) / 2.0) {
                    r
                } else {
                    sign(f1) * (r - f2)
                }
            }
        }
    }

    companion object {
        /**
         * The closure type for an open arc with no path segments
         * connecting the two ends of the arc segment.
         */
        const val OPEN = 0

        /**
         * The closure type for an arc closed by drawing a straight
         * line segment from the start of the arc segment to the end of the
         * arc segment.
         */
        const val CHORD = 1

        /**
         * The closure type for an arc closed by drawing straight line
         * segments from the start of the arc segment to the center
         * of the full ellipse and from that point to the end of the arc segment.
         */
        const val PIE = 2
    }
}