/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

import org.kamaeleo.geom.PathIterator.Companion.SEG_CLOSE
import org.kamaeleo.geom.PathIterator.Companion.SEG_CUBICTO
import org.kamaeleo.geom.PathIterator.Companion.SEG_LINETO
import org.kamaeleo.geom.PathIterator.Companion.SEG_QUADTO
import org.kamaeleo.geom.PathIterator.Companion.WIND_NON_ZERO

/**
 * An `Area` object stores and manipulates a
 * resolution-independent description of an enclosed area of
 * 2-dimensional space.
 * `Area` objects can be transformed and can perform
 * various Constructive Area Geometry (CAG) operations when combined
 * with other `Area` objects.
 * The CAG operations include area
 * [addition][.add], [subtraction][.subtract],
 * [intersection][.intersect], and [exclusive or][.exclusiveOr].
 * See the linked method documentation for examples of the various
 * operations.
 *
 *
 * The `Area` class implements the `Shape`
 * interface and provides full support for all of its hit-testing
 * and path iteration facilities, but an `Area` is more
 * specific than a generalized path in a number of ways:
 *
 *  * Only closed paths and sub-paths are stored.
 * `Area` objects constructed from unclosed paths
 * are implicitly closed during construction as if those paths
 * had been filled by the `Graphics2D.fill` method.
 *  * The interiors of the individual stored sub-paths are all
 * non-empty and non-overlapping.  Paths are decomposed during
 * construction into separate component non-overlapping parts,
 * empty pieces of the path are discarded, and then these
 * non-empty and non-overlapping properties are maintained
 * through all subsequent CAG operations.  Outlines of different
 * component sub-paths may touch each other, as long as they
 * do not cross so that their enclosed areas overlap.
 *  * The geometry of the path describing the outline of the
 * `Area` resembles the path from which it was
 * constructed only in that it describes the same enclosed
 * 2-dimensional area, but may use entirely different types
 * and ordering of the path segments to do so.
 *
 * Interesting issues which are not always obvious when using
 * the `Area` include:
 *
 *  * Creating an `Area` from an unclosed (open)
 * `Shape` results in a closed outline in the
 * `Area` object.
 *  * Creating an `Area` from a `Shape`
 * which encloses no area (even when "closed") produces an
 * empty `Area`.  A common example of this issue
 * is that producing an `Area` from a line will
 * be empty since the line encloses no area.  An empty
 * `Area` will iterate no geometry in its
 * `PathIterator` objects.
 *  * A self-intersecting `Shape` may be split into
 * two (or more) sub-paths each enclosing one of the
 * non-intersecting portions of the original path.
 *  * An `Area` may take more path segments to
 * describe the same geometry even when the original
 * outline is simple and obvious.  The analysis that the
 * `Area` class must perform on the path may
 * not reflect the same concepts of "simple and obvious"
 * as a human being perceives.
 *
 *
 * @since 1.2
 */
class Area : Shape {
    private var curves: MutableList<Curve>? = null
    private var cachedBounds: Rectangle2D? = null

    /**
     * Default constructor which creates an empty area.
     *
     * @since 1.2
     */
    constructor() {
        curves = EmptyCurves
    }

    /**
     * The `Area` class creates an area geometry from the
     * specified [java.awt.Shape] object.  The geometry is explicitly
     * closed, if the `Shape` is not already closed.  The
     * fill rule (even-odd or winding) specified by the geometry of the
     * `Shape` is used to determine the resulting enclosed area.
     *
     * @param s the `Shape` from which the area is constructed
     *
     * @throws NullPointerException if `s` is null
     * @since 1.2
     */
    constructor(s: Shape) {
        curves = if (s is Area) {
            s.curves
        } else {
            pathToCurves(s.pathIterator)
        }
    }

    /**
     * Adds the shape of the specified `Area` to the
     * shape of this `Area`.
     * The resulting shape of this `Area` will include
     * the union of both shapes, or all areas that were contained
     * in either this or the specified `Area`.
     * <pre>
     * // Example:
     * Area a1 = new Area([triangle 0,0 =&gt; 8,0 =&gt; 0,8]);
     * Area a2 = new Area([triangle 0,0 =&gt; 8,0 =&gt; 8,8]);
     * a1.add(a2);
     *
     * a1(before)     +         a2         =     a1(after)
     *
     * ################     ################     ################
     * ##############         ##############     ################
     * ############             ############     ################
     * ##########                 ##########     ################
     * ########                     ########     ################
     * ######                         ######     ######    ######
     * ####                             ####     ####        ####
     * ##                                 ##     ##            ##
    </pre> *
     *
     * @param rhs the `Area` to be added to the
     * current shape
     *
     * @throws NullPointerException if `rhs` is null
     * @since 1.2
     */
    fun add(rhs: Area) {
        curves = AreaOp.AddOp().calculate(curves!!, rhs.curves!!)
        invalidateBounds()
    }

    private fun invalidateBounds() {
        cachedBounds = null
    }

    /**
     * Subtracts the shape of the specified `Area` from the
     * shape of this `Area`.
     * The resulting shape of this `Area` will include
     * areas that were contained only in this `Area`
     * and not in the specified `Area`.
     * <pre>
     * // Example:
     * Area a1 = new Area([triangle 0,0 =&gt; 8,0 =&gt; 0,8]);
     * Area a2 = new Area([triangle 0,0 =&gt; 8,0 =&gt; 8,8]);
     * a1.subtract(a2);
     *
     * a1(before)     -         a2         =     a1(after)
     *
     * ################     ################
     * ##############         ##############     ##
     * ############             ############     ####
     * ##########                 ##########     ######
     * ########                     ########     ########
     * ######                         ######     ######
     * ####                             ####     ####
     * ##                                 ##     ##
    </pre> *
     *
     * @param rhs the `Area` to be subtracted from the
     * current shape
     *
     * @throws NullPointerException if `rhs` is null
     * @since 1.2
     */
    fun subtract(rhs: Area) {
        curves = AreaOp.SubOp().calculate(curves!!, rhs.curves!!)
        invalidateBounds()
    }

    /**
     * Sets the shape of this `Area` to the intersection of
     * its current shape and the shape of the specified `Area`.
     * The resulting shape of this `Area` will include
     * only areas that were contained in both this `Area`
     * and also in the specified `Area`.
     * <pre>
     * // Example:
     * Area a1 = new Area([triangle 0,0 =&gt; 8,0 =&gt; 0,8]);
     * Area a2 = new Area([triangle 0,0 =&gt; 8,0 =&gt; 8,8]);
     * a1.intersect(a2);
     *
     * a1(before)   intersect     a2         =     a1(after)
     *
     * ################     ################     ################
     * ##############         ##############       ############
     * ############             ############         ########
     * ##########                 ##########           ####
     * ########                     ########
     * ######                         ######
     * ####                             ####
     * ##                                 ##
    </pre> *
     *
     * @param rhs the `Area` to be intersected with this
     * `Area`
     *
     * @throws NullPointerException if `rhs` is null
     * @since 1.2
     */
    fun intersect(rhs: Area) {
        curves = AreaOp.IntOp().calculate(curves!!, rhs.curves!!)
        invalidateBounds()
    }

    /**
     * Sets the shape of this `Area` to be the combined area
     * of its current shape and the shape of the specified `Area`,
     * minus their intersection.
     * The resulting shape of this `Area` will include
     * only areas that were contained in either this `Area`
     * or in the specified `Area`, but not in both.
     * <pre>
     * // Example:
     * Area a1 = new Area([triangle 0,0 =&gt; 8,0 =&gt; 0,8]);
     * Area a2 = new Area([triangle 0,0 =&gt; 8,0 =&gt; 8,8]);
     * a1.exclusiveOr(a2);
     *
     * a1(before)    xor        a2         =     a1(after)
     *
     * ################     ################
     * ##############         ##############     ##            ##
     * ############             ############     ####        ####
     * ##########                 ##########     ######    ######
     * ########                     ########     ################
     * ######                         ######     ######    ######
     * ####                             ####     ####        ####
     * ##                                 ##     ##            ##
    </pre> *
     *
     * @param rhs the `Area` to be exclusive ORed with this
     * `Area`.
     *
     * @throws NullPointerException if `rhs` is null
     * @since 1.2
     */
    fun exclusiveOr(rhs: Area) {
        curves = AreaOp.XorOp().calculate(curves!!, rhs.curves!!)
        invalidateBounds()
    }

    /**
     * Removes all of the geometry from this `Area` and
     * restores it to an empty area.
     *
     * @since 1.2
     */
    fun reset() {
        curves = ArrayList()
        invalidateBounds()
    }

    /**
     * Tests whether this `Area` object encloses any area.
     *
     * @return `true` if this `Area` object
     * represents an empty area; `false` otherwise.
     *
     * @since 1.2
     */
    val isEmpty: Boolean
        get() = curves!!.size == 0

    /**
     * Tests whether this `Area` consists entirely of
     * straight edged polygonal geometry.
     *
     * @return `true` if the geometry of this
     * `Area` consists entirely of line segments;
     * `false` otherwise.
     *
     * @since 1.2
     */
    val isPolygonal: Boolean
        get() {
            for (c in curves!!) {
                if(c.order > 1) {
                    return false
                }
            }
            return true
        }// One might be able to prove that this is impossible...

    /**
     * Tests whether this `Area` is rectangular in shape.
     *
     * @return `true` if the geometry of this
     * `Area` is rectangular in shape; `false`
     * otherwise.
     *
     * @since 1.2
     */
    val isRectangular: Boolean
        get() {
            val size: Int = curves!!.size
            if (size == 0) {
                return true
            }
            if (size > 3) {
                return false
            }
            val c1: Curve = curves!!.get(1)!!
            val c2: Curve = curves!!.get(2)!!
            if (c1.order != 1 || c2.order != 1) {
                return false
            }
            return if (c1.xTop !== c1.xBot || c2.xTop !== c2.xBot) {
                false
            } else c1.yTop === c2.yTop && c1.yBot === c2.yBot
            // One might be able to prove that this is impossible...
        }// First Order0 "moveto"

    /**
     * Tests whether this `Area` is comprised of a single
     * closed subpath.  This method returns `true` if the
     * path contains 0 or 1 subpaths, or `false` if the path
     * contains more than 1 subpath.  The subpaths are counted by the
     * number of [SEG_MOVETO][PathIterator.SEG_MOVETO]  segments
     * that appear in the path.
     *
     * @return `true` if the `Area` is comprised
     * of a single basic geometry; `false` otherwise.
     *
     * @since 1.2
     */
    val isSingular: Boolean
        get() {
            if (curves!!.size < 3) {
                return true
            }
            for (c in curves!!) {
                if (c!!.order == 0) {
                    return false
                }
            }
            return true
        }

    /**
     * Returns a high precision bounding [Rectangle2D] that
     * completely encloses this `Area`.
     *
     *
     * The Area class will attempt to return the tightest bounding
     * box possible for the Shape.  The bounding box will not be
     * padded to include the control points of curves in the outline
     * of the Shape, but should tightly fit the actual geometry of
     * the outline itself.
     *
     * @return the bounding `Rectangle2D` for the
     * `Area`.
     *
     * @since 1.2
     */
    override val bounds2D: Rectangle2D
        get() = getCachedBounds().bounds2D

    private fun getCachedBounds(): Rectangle2D {
        if (cachedBounds != null) {
            return cachedBounds!!
        }
        var r: Rectangle2D = Rectangle2D.Double(0.0, 0.0, 0.0, 0.0)
        if (curves!!.size > 0) {
            val c: Curve = curves!!.get(0)
            // First point is always an order 0 curve (moveto)
            r = Rectangle2D.Double(c.x0, c.y0, 0.0, 0.0)
            for (i in 1 until curves!!.size) {
                r = (curves!!.get(i) as Curve).enlarge(Rectangle2D.Double(r.x, r.y, r.width, r.height))
            }
        }
        return Rectangle2D.Double(r.x, r.y, r.width, r.height).also { cachedBounds = it }
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    override operator fun contains(p: Point2D): Boolean {
        return contains(p.x, p.y)
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    fun contains(x: Double, y: Double): Boolean {
        if (!getCachedBounds().contains(x, y)) {
            return false
        }

        var crossings = 0
        for (c in curves!!) {
            crossings += c.crossingsFor(x, y)
        }
        return crossings and 1 == 1
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    override fun intersects(r: Rectangle2D): Boolean {
        return intersects(r.x, r.y, r.width, r.height)
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    fun intersects(x: Double, y: Double, w: Double, h: Double): Boolean {
        if (w < 0 || h < 0) {
            return false
        }
        if (!getCachedBounds().intersects(x, y, w, h)) {
            return false
        }
        val c: Crossings? = Crossings.findCrossings(curves!!, x, y, x + w, y + h)
        return c == null || !c.isEmpty
    }

    /**
     * Creates a [PathIterator] for the outline of this
     * `Area` object.  This `Area` object is unchanged.
     *
     * @param at an optional `AffineTransform` to be applied to
     * the coordinates as they are returned in the iteration, or
     * `null` if untransformed coordinates are desired
     *
     * @return the `PathIterator` object that returns the
     * geometry of the outline of this `Area`, one
     * segment at a time.
     *
     * @since 1.2
     */
    override val pathIterator: org.kamaeleo.geom.PathIterator
        get() = AreaIterator(curves!!, null)

    override fun getPathIterator(at: AffineTransform?): PathIterator {
        return AreaIterator(curves!!, at)
    }

    /**
     * Creates a `PathIterator` for the flattened outline of
     * this `Area` object.  Only uncurved path segments
     * represented by the SEG_MOVETO, SEG_LINETO, and SEG_CLOSE point
     * types are returned by the iterator.  This `Area`
     * object is unchanged.
     *
     * @param at       an optional `AffineTransform` to be
     * applied to the coordinates as they are returned in the
     * iteration, or `null` if untransformed coordinates
     * are desired
     * @param flatness the maximum amount that the control points
     * for a given curve can vary from colinear before a subdivided
     * curve is replaced by a straight line connecting the end points
     *
     * @return the `PathIterator` object that returns the
     * geometry of the outline of this `Area`, one segment
     * at a time.
     *
     * @since 1.2
     */
    override fun getFlattenPathIterator(flatness: Double): PathIterator {
        return FlatteningPathIterator(pathIterator, flatness)
    }

    /**
     * Returns an exact copy of this `Area` object.
     *
     * @return Created clone object
     *
     * @since 1.2
     */
    fun clone(): Any {
        return Area(this)
    }

    /**
     * Tests whether the geometries of the two `Area` objects
     * are equal.
     * This method will return false if the argument is null.
     *
     * @param other the `Area` to be compared to this
     * `Area`
     *
     * @return `true` if the two geometries are equal;
     * `false` otherwise.
     *
     * @since 1.2
     */
    fun equals(other: Area?): Boolean {
        // REMIND: A *much* simpler operation should be possible...
        // Should be able to do a curve-wise comparison since all Areas
        // should evaluate their curves in the same top-down order.
        if (other === this) {
            return true
        }
        if (other == null) {
            return false
        }
        val c: MutableList<Curve> = AreaOp.XorOp().calculate(curves!!, other.curves!!)
        return c.isEmpty()
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    operator fun contains(r: Rectangle2D): Boolean {
        return contains(r.x, r.y, r.width, r.height)
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    fun contains(x: Double, y: Double, w: Double, h: Double): Boolean {
        if (w < 0 || h < 0) {
            return false
        }
        if (!getCachedBounds().contains(x, y, w, h)) {
            return false
        }
        val c: Crossings? = Crossings.findCrossings(curves!!, x, y, x + w, y + h)
        return c != null && c.covers(y, y + h)
    }

    internal class AreaIterator(curves: MutableList<Curve>, at: AffineTransform?) : PathIterator {
        private val transform: AffineTransform?
        private val curves: MutableList<Curve>
        private var index = 0
        private var prevcurve: Curve? = null
        private var thiscurve: Curve? = null

        // REMIND: Which is better, EVEN_ODD or NON_ZERO?
        //         The paths calculated could be classified either way.
        //return WIND_EVEN_ODD;
        override val windingRule: Int
            get() =// REMIND: Which is better, EVEN_ODD or NON_ZERO?
            //         The paths calculated could be classified either way.
                //return WIND_EVEN_ODD;
                WIND_NON_ZERO
        override val isDone: Boolean
            get() = prevcurve == null && thiscurve == null

        override operator fun next() {
            if (prevcurve != null) {
                prevcurve = null
            } else {
                prevcurve = thiscurve
                index++
                if (index < curves.size) {
                    thiscurve = curves.get(index)
                    if (thiscurve!!.order !== 0 && prevcurve!!.x1 == thiscurve!!.x0 && prevcurve!!.y1 == thiscurve!!.y0) {
                        prevcurve = null
                    }
                } else {
                    thiscurve = null
                }
            }
        }

        override fun currentSegment(coords: FloatArray): Int {
            val dcoords = DoubleArray(6)
            val segtype = currentSegment(dcoords)
            val numpoints =
                if (segtype == SEG_CLOSE) 0 else if (segtype == SEG_QUADTO) 2 else if (segtype == SEG_CUBICTO) 3 else 1
            for (i in 0 until numpoints * 2) {
                coords[i] = dcoords[i].toFloat()
            }
            return segtype
        }

        override fun currentSegment(coords: DoubleArray): Int {
            val segtype: Int
            var numpoints: Int
            if (prevcurve != null) {
                // Need to finish off junction between curves
                if (thiscurve == null || thiscurve!!.order === 0) {
                    return SEG_CLOSE
                }
                coords[0] = thiscurve!!.x0
                coords[1] = thiscurve!!.y0
                segtype = SEG_LINETO
                numpoints = 1
            } else if (thiscurve == null) {
                throw NoSuchElementException("area iterator out of bounds")
            } else {
                segtype = thiscurve!!.getSegment(coords)
                numpoints = thiscurve!!.order
                if (numpoints == 0) {
                    numpoints = 1
                }
            }
            if (transform != null) {
                transform.transform(coords, 0, coords, 0, numpoints)
            }
            return segtype
        }

        init {
            this.curves = curves
            transform = at
            if (curves.size >= 1) {
                thiscurve = curves.get(0)
            }
        }
    }

    companion object {
        private val EmptyCurves: MutableList<Curve> = ArrayList()
        private fun pathToCurves(pi: PathIterator): MutableList<Curve> {
            val curves = ArrayList<Curve>()
            val windingRule: Int = pi.windingRule
            // coords array is big enough for holding:
            //     coordinates returned from currentSegment (6)
            //     OR
            //         two subdivided quadratic curves (2+4+4=10)
            //         AND
            //             0-1 horizontal splitting parameters
            //             OR
            //             2 parametric equation derivative coefficients
            //     OR
            //         three subdivided cubic curves (2+6+6+6=20)
            //         AND
            //             0-2 horizontal splitting parameters
            //             OR
            //             3 parametric equation derivative coefficients
            val coords = DoubleArray(23)
            var movx = 0.0
            var movy = 0.0
            var curx = 0.0
            var cury = 0.0
            while (!pi.isDone) {
                var newy: Double
                var newx: Double
                when (pi.currentSegment(coords)) {
                    PathIterator.SEG_MOVETO -> {
                        Curve.insertLine(curves, curx, cury, movx, movy)
                        run {
                            movx = coords[0]
                            curx = movx
                        }
                        run {
                            movy = coords[1]
                            cury = movy
                        }
                        Curve.insertMove(curves, movx, movy)
                    }
                    PathIterator.SEG_LINETO -> {
                        newx = coords[0]
                        newy = coords[1]
                        Curve.insertLine(curves, curx, cury, newx, newy)
                        curx = newx
                        cury = newy
                    }
                    PathIterator.SEG_QUADTO -> {
                        newx = coords[2]
                        newy = coords[3]
                        Curve.insertQuad(curves, curx, cury, coords)
                        curx = newx
                        cury = newy
                    }
                    PathIterator.SEG_CUBICTO -> {
                        newx = coords[4]
                        newy = coords[5]
                        Curve.insertCubic(curves, curx, cury, coords)
                        curx = newx
                        cury = newy
                    }
                    PathIterator.SEG_CLOSE -> {
                        Curve.insertLine(curves, curx, cury, movx, movy)
                        curx = movx
                        cury = movy
                    }
                }
                pi.next()
            }
            Curve.insertLine(curves, curx, cury, movx, movy)
            val operator: AreaOp
            if (windingRule == PathIterator.WIND_EVEN_ODD) {
                operator = AreaOp.EOWindOp()
            } else {
                operator = AreaOp.NZWindOp()
            }
            return operator.calculate(curves, EmptyCurves)
        }
    }
}