/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

import org.kamaeleo.geom.PathIterator.Companion.SEG_CLOSE
import org.kamaeleo.geom.PathIterator.Companion.SEG_CUBICTO
import org.kamaeleo.geom.PathIterator.Companion.SEG_MOVETO
import org.kamaeleo.geom.PathIterator.Companion.WIND_NON_ZERO

/**
 * A utility class to iterate over the path segments of an ellipse
 * through the PathIterator interface.
 *
 * @author Jim Graham
 */
internal class EllipseIterator(e: Ellipse2D, at: AffineTransform?) : PathIterator {
    val x: Double
    val y: Double
    val w: Double
    val h: Double
    val affine: AffineTransform?
    var index = 0

    /**
     * Return the winding rule for determining the insideness of the
     * path.
     *
     * @see .WIND_EVEN_ODD
     *
     * @see .WIND_NON_ZERO
     */
    override val windingRule: Int
        get() = WIND_NON_ZERO

    /**
     * Tests if there are more points to read.
     *
     * @return true if there are more points to read
     */
    override val isDone: Boolean
        get() = index > 5

    /**
     * Moves the iterator to the next segment of the path forwards
     * along the primary direction of traversal as long as there are
     * more points in that direction.
     */
    override operator fun next() {
        index++
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, SEG_QUADTO, SEG_CUBICTO, or SEG_CLOSE.
     * A float array of length 6 must be passed in and may be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of float x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types will return one point,
     * SEG_QUADTO will return two points,
     * SEG_CUBICTO will return 3 points
     * and SEG_CLOSE will not return any points.
     *
     * @see .SEG_MOVETO
     *
     * @see .SEG_LINETO
     *
     * @see .SEG_QUADTO
     *
     * @see .SEG_CUBICTO
     *
     * @see .SEG_CLOSE
     */
    override fun currentSegment(coords: FloatArray): Int {
        if (isDone) {
            throw NoSuchElementException("ellipse iterator out of bounds")
        }
        if (index == 5) {
            return SEG_CLOSE
        }
        if (index == 0) {
            val ctrls = ctrlpts[3]
            coords[0] = (x + ctrls[4] * w).toFloat()
            coords[1] = (y + ctrls[5] * h).toFloat()
            if (affine != null) {
                affine.transform(coords, 0, coords, 0, 1)
            }
            return SEG_MOVETO
        }
        val ctrls = ctrlpts[index - 1]
        coords[0] = (x + ctrls[0] * w).toFloat()
        coords[1] = (y + ctrls[1] * h).toFloat()
        coords[2] = (x + ctrls[2] * w).toFloat()
        coords[3] = (y + ctrls[3] * h).toFloat()
        coords[4] = (x + ctrls[4] * w).toFloat()
        coords[5] = (y + ctrls[5] * h).toFloat()
        if (affine != null) {
            affine.transform(coords, 0, coords, 0, 3)
        }
        return SEG_CUBICTO
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, SEG_QUADTO, SEG_CUBICTO, or SEG_CLOSE.
     * A double array of length 6 must be passed in and may be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of double x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types will return one point,
     * SEG_QUADTO will return two points,
     * SEG_CUBICTO will return 3 points
     * and SEG_CLOSE will not return any points.
     *
     * @see .SEG_MOVETO
     *
     * @see .SEG_LINETO
     *
     * @see .SEG_QUADTO
     *
     * @see .SEG_CUBICTO
     *
     * @see .SEG_CLOSE
     */
    override fun currentSegment(coords: DoubleArray): Int {
        if (isDone) {
            throw NoSuchElementException("ellipse iterator out of bounds")
        }
        if (index == 5) {
            return SEG_CLOSE
        }
        if (index == 0) {
            val ctrls = ctrlpts[3]
            coords[0] = x + ctrls[4] * w
            coords[1] = y + ctrls[5] * h
            if (affine != null) {
                affine.transform(coords, 0, coords, 0, 1)
            }
            return SEG_MOVETO
        }
        val ctrls = ctrlpts[index - 1]
        coords[0] = x + ctrls[0] * w
        coords[1] = y + ctrls[1] * h
        coords[2] = x + ctrls[2] * w
        coords[3] = y + ctrls[3] * h
        coords[4] = x + ctrls[4] * w
        coords[5] = y + ctrls[5] * h
        if (affine != null) {
            affine.transform(coords, 0, coords, 0, 3)
        }
        return SEG_CUBICTO
    }

    companion object {
        // ArcIterator.btan(Math.PI/2)
        const val CtrlVal = 0.5522847498307933

        /*
     * ctrlpts contains the control points for a set of 4 cubic
     * bezier curves that approximate a circle of radius 0.5
     * centered at 0.5, 0.5
     */
        private const val pcv = 0.5 + CtrlVal * 0.5
        private const val ncv = 0.5 - CtrlVal * 0.5
        private val ctrlpts = arrayOf(
            doubleArrayOf(1.0, pcv, pcv, 1.0, 0.5, 1.0),
            doubleArrayOf(ncv, 1.0, 0.0, pcv, 0.0, 0.5),
            doubleArrayOf(0.0, ncv, ncv, 0.0, 0.5, 0.0),
            doubleArrayOf(
                pcv, 0.0, 1.0, ncv, 1.0, 0.5
            )
        )
    }

    init {
        x = e.x
        y = e.y
        w = e.width
        h = e.height
        affine = at
        if (w < 0 || h < 0) {
            index = 6
        }
    }
}