/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.kamaeleo.geom

import kotlin.math.abs
import kotlin.math.absoluteValue
import kotlin.math.max
import kotlin.math.min

/**
 * The `RoundRectangle2D` class defines a rectangle with
 * rounded corners defined by a location `(x,y)`, a
 * dimension `(w x h)`, and the width and height of an arc
 * with which to round the corners.
 *
 *
 * This class is the abstract superclass for all objects that
 * store a 2D rounded rectangle.
 * The actual storage representation of the coordinates is left to
 * the subclass.
 *
 * @author Jim Graham
 * @since 1.2
 */
abstract class RoundRectangle2D
/**
 * This is an abstract class that cannot be instantiated directly.
 * Type-specific implementation subclasses are available for
 * instantiation and provide a number of formats for storing
 * the information necessary to satisfy the various accessor
 * methods below.
 *
 * @see java.awt.geom.RoundRectangle2D.Float
 *
 * @see java.awt.geom.RoundRectangle2D.Double
 *
 * @since 1.2
 */
protected constructor() : AbstractRectangle2D() {
    /**
     * Sets this `RoundRectangle2D` to be the same as the
     * specified `RoundRectangle2D`.
     *
     * @param rr the specified `RoundRectangle2D`
     *
     * @since 1.2
     */
    open fun setRoundRect(rr: RoundRectangle2D) {
        setRoundRect(
            rr.x, rr.y, rr.width, rr.height,
            rr.arcWidth, rr.arcHeight
        )
    }

    /**
     * Gets the width of the arc that rounds off the corners.
     *
     * @return the width of the arc that rounds off the corners
     * of this `RoundRectangle2D`.
     *
     * @since 1.2
     */
    abstract val arcWidth: kotlin.Double

    /**
     * Gets the height of the arc that rounds off the corners.
     *
     * @return the height of the arc that rounds off the corners
     * of this `RoundRectangle2D`.
     *
     * @since 1.2
     */
    abstract val arcHeight: kotlin.Double

    /**
     * Sets the location, size, and corner radii of this
     * `RoundRectangle2D` to the specified
     * `double` values.
     *
     * @param x         the X coordinate to which to set the
     * location of this `RoundRectangle2D`
     * @param y         the Y coordinate to which to set the
     * location of this `RoundRectangle2D`
     * @param w         the width to which to set this
     * `RoundRectangle2D`
     * @param h         the height to which to set this
     * `RoundRectangle2D`
     * @param arcWidth  the width to which to set the arc of this
     * `RoundRectangle2D`
     * @param arcHeight the height to which to set the arc of this
     * `RoundRectangle2D`
     *
     * @since 1.2
     */
    abstract fun setRoundRect(
        x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double,
        arcWidth: kotlin.Double, arcHeight: kotlin.Double
    )

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    fun setFrame(x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double) {
        setRoundRect(x, y, w, h, arcWidth, arcHeight)
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    override fun intersects(x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double): Boolean {
        var x = x
        var y = y
        if (isEmpty || w <= 0 || h <= 0) {
            return false
        }
        val rrx0: kotlin.Double = x
        val rry0: kotlin.Double = y
        val rrx1: kotlin.Double = rrx0 + width
        val rry1: kotlin.Double = rry0 + height
        // Check for trivial rejection - bounding rectangles do not intersect
        if (x + w <= rrx0 || x >= rrx1 || y + h <= rry0 || y >= rry1) {
            return false
        }
        val aw: kotlin.Double = min(width, abs(arcWidth)) / 2.0
        val ah: kotlin.Double = min(height, abs(arcHeight)) / 2.0
        val x0class = classify(x, rrx0, rrx1, aw)
        val x1class = classify(x + w, rrx0, rrx1, aw)
        val y0class = classify(y, rry0, rry1, ah)
        val y1class = classify(y + h, rry0, rry1, ah)
        // Trivially accept if any point is inside inner rectangle
        if (x0class == 2 || x1class == 2 || y0class == 2 || y1class == 2) {
            return true
        }
        // Trivially accept if either edge spans inner rectangle
        if (x0class < 2 && x1class > 2 || y0class < 2 && y1class > 2) {
            return true
        }
        // Since neither edge spans the center, then one of the corners
        // must be in one of the rounded edges.  We detect this case if
        // a [xy]0class is 3 or a [xy]1class is 1.  One of those two cases
        // must be true for each direction.
        // We now find a "nearest point" to test for being inside a rounded
        // corner.
        x = if (x1class == 1) x + w - (rrx0 + aw) else x - (rrx1 - aw)
        y = if (y1class == 1) y + h - (rry0 + ah) else y - (rry1 - ah)
        x = x / aw
        y = y / ah
        return x * x + y * y <= 1.0
    }

    private fun classify(
        coord: kotlin.Double, left: kotlin.Double, right: kotlin.Double,
        arcsize: kotlin.Double
    ): Int {
        return if (coord < left) {
            0
        } else if (coord < left + arcsize) {
            1
        } else if (coord < right - arcsize) {
            2
        } else if (coord < right) {
            3
        } else {
            4
        }
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    override fun contains(x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double): Boolean {
        return if (isEmpty || w <= 0 || h <= 0) {
            false
        } else contains(x, y) &&
                contains(x + w, y) &&
                contains(x, y + h) &&
                contains(x + w, y + h)
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    override fun contains(x: kotlin.Double, y: kotlin.Double): Boolean {
        var x = x
        var y = y
        if (isEmpty) {
            return false
        }
        var rrx0: kotlin.Double = x
        var rry0: kotlin.Double = y
        val rrx1: kotlin.Double = rrx0 + width
        val rry1: kotlin.Double = rry0 + height
        // Check for trivial rejection - point is outside bounding rectangle
        if (x < rrx0 || y < rry0 || x >= rrx1 || y >= rry1) {
            return false
        }
        val aw: kotlin.Double = min(width, arcWidth.absoluteValue) / 2.0
        val ah: kotlin.Double = min(height, arcHeight.absoluteValue) / 2.0
        // Check which corner point is in and do circular containment
        // test - otherwise simple acceptance
        if (x >= aw.let { rrx0 += it; rrx0 } && x < rrx1 - aw.also { rrx0 = it }) {
            return true
        }
        if (y >= ah.let { rry0 += it; rry0 } && y < rry1 - ah.also { rry0 = it }) {
            return true
        }
        x = (x - rrx0) / aw
        y = (y - rry0) / ah
        return x * x + y * y <= 1.0
    }

    abstract val isEmpty: Boolean

    /**
     * Returns an iteration object that defines the boundary of this
     * `RoundRectangle2D`.
     * The iterator for this class is multi-threaded safe, which means
     * that this `RoundRectangle2D` class guarantees that
     * modifications to the geometry of this `RoundRectangle2D`
     * object do not affect any iterations of that geometry that
     * are already in process.
     *
     * @param at an optional `AffineTransform` to be applied to
     * the coordinates as they are returned in the iteration, or
     * `null` if untransformed coordinates are desired
     *
     * @return the `PathIterator` object that returns the
     * geometry of the outline of this
     * `RoundRectangle2D`, one segment at a time.
     *
     * @since 1.2
     */
    override fun getPathIterator(at: AffineTransform?): PathIterator {
        return RoundRectIterator(this, at)
    }

    /**
     * Returns the hashcode for this `RoundRectangle2D`.
     *
     * @return the hashcode for this `RoundRectangle2D`.
     *
     * @since 1.6
     */
    override fun hashCode(): Int {
        var bits: Long = x.toRawBits()
        bits += y.toRawBits() * 37
        bits += width.toRawBits() * 43
        bits += height.toRawBits() * 47
        bits += arcWidth.toRawBits() * 53
        bits += arcHeight.toRawBits() * 59
        return bits.toInt() xor (bits shr 32) as Int
    }

    /**
     * Determines whether or not the specified `Object` is
     * equal to this `RoundRectangle2D`.  The specified
     * `Object` is equal to this `RoundRectangle2D`
     * if it is an instance of `RoundRectangle2D` and if its
     * location, size, and corner arc dimensions are the same as this
     * `RoundRectangle2D`.
     *
     * @param obj an `Object` to be compared with this
     * `RoundRectangle2D`.
     *
     * @return `true` if `obj` is an instance
     * of `RoundRectangle2D` and has the same values;
     * `false` otherwise.
     *
     * @since 1.6
     */
    override fun equals(obj: Any?): Boolean {
        if (obj === this) {
            return true
        }
        if (obj is RoundRectangle2D) {
            val rr2d = obj
            return x === rr2d.x &&
                    y === rr2d.y &&
                    width === rr2d.width &&
                    height === rr2d.height &&
                    arcWidth == rr2d.arcWidth &&
                    arcHeight == rr2d.arcHeight
        }
        return false
    }

    /**
     * The `Double` class defines a rectangle with rounded
     * corners all specified in `double` coordinates.
     *
     * @since 1.2
     */
    class Double : RoundRectangle2D {
        /**
         * {@inheritDoc}
         *
         * @since 1.2
         */
        /**
         * The X coordinate of this `RoundRectangle2D`.
         *
         * @serial
         * @since 1.2
         */
        override var x = 0.0
        /**
         * {@inheritDoc}
         *
         * @since 1.2
         */
        /**
         * The Y coordinate of this `RoundRectangle2D`.
         *
         * @serial
         * @since 1.2
         */
        override var y = 0.0
        /**
         * {@inheritDoc}
         *
         * @since 1.2
         */
        /**
         * The width of this `RoundRectangle2D`.
         *
         * @serial
         * @since 1.2
         */
        override var width = 0.0
        /**
         * {@inheritDoc}
         *
         * @since 1.2
         */
        /**
         * The height of this `RoundRectangle2D`.
         *
         * @serial
         * @since 1.2
         */
        override var height = 0.0

        /**
         * The width of the arc that rounds off the corners.
         *
         * @serial
         * @since 1.2
         */
        override var arcWidth = 0.0

        /**
         * The height of the arc that rounds off the corners.
         *
         * @serial
         * @since 1.2
         */
        override var arcHeight = 0.0

        /**
         * Constructs a new `RoundRectangle2D`, initialized to
         * location (0.0,&nbsp;0.0), size (0.0,&nbsp;0.0), and corner arcs
         * of radius 0.0.
         *
         * @since 1.2
         */
        constructor() {}

        /**
         * Constructs and initializes a `RoundRectangle2D`
         * from the specified `double` coordinates.
         *
         * @param x    the X coordinate of the newly
         * constructed `RoundRectangle2D`
         * @param y    the Y coordinate of the newly
         * constructed `RoundRectangle2D`
         * @param w    the width to which to set the newly
         * constructed `RoundRectangle2D`
         * @param h    the height to which to set the newly
         * constructed `RoundRectangle2D`
         * @param arcw the width of the arc to use to round off the
         * corners of the newly constructed
         * `RoundRectangle2D`
         * @param arch the height of the arc to use to round off the
         * corners of the newly constructed
         * `RoundRectangle2D`
         *
         * @since 1.2
         */
        constructor(
            x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double,
            arcw: kotlin.Double, arch: kotlin.Double
        ) {
            setRoundRect(x, y, w, h, arcw, arch)
        }

        override fun add(x: kotlin.Double, y: kotlin.Double): Rectangle2D {
            val x1: kotlin.Double = min(minX, x)
            val x2: kotlin.Double = max(maxX, x)
            val y1: kotlin.Double = min(minY, y)
            val y2: kotlin.Double = max(maxY, y)
            return Double(x1, y1, x2 - x1, y2 - y1, arcWidth, arcHeight)
        }

        override fun normalize(): Rectangle2D {
            // quick check if anything needs to be done
            if (isNormalized) return this
            var normalizedX = x
            var normalizedY = y
            var normalizedWidth = width
            var normalizedHeight = height
            if (normalizedWidth < 0) {
                normalizedWidth = -normalizedWidth
                normalizedX -= normalizedWidth
            }
            if (normalizedHeight < 0) {
                normalizedHeight = -normalizedHeight
                normalizedY -= normalizedHeight
            }
            return Double(
                normalizedX, normalizedY, normalizedWidth,
                normalizedHeight, arcWidth, arcHeight
            )
        }

        /**
         * Returns true if both width and height are positive.
         *
         * @return true if normalized
         */
        private val isNormalized: Boolean
            private get() = width >= 0 && height >= 0

        /**
         * {@inheritDoc}
         *
         * @since 1.2
         */
        override fun setRoundRect(
            x: kotlin.Double, y: kotlin.Double, w: kotlin.Double, h: kotlin.Double,
            arcw: kotlin.Double, arch: kotlin.Double
        ) {
            this.x = x
            this.y = y
            width = w
            height = h
            arcWidth = arcw
            arcHeight = arch
        }

        /**
         * {@inheritDoc}
         *
         * @since 1.2
         */
        override fun setRoundRect(rr: RoundRectangle2D) {
            x = rr.x
            y = rr.y
            width = rr.width
            height = rr.height
            arcWidth = rr.arcWidth
            arcHeight = rr.arcHeight
        }

        override val isEmpty: Boolean
            get() = width <= 0.0f || height <= 0.0f

        /**
         * {@inheritDoc}
         *
         * @since 1.2
         */
        fun computeBounds2D(): Rectangle2D {
            return Rectangle2D.Double(x, y, width, height)
        }
    }
}