package org.kamaeleo.geom.curve

/**
 *
 * General non-rational B-Spline implementation where the degree can be specified.
 *
 *
 *
 * For the B-Spline, there are 3 types of knot-vectors, uniform clamped, uniform unclamped,
 * and non-uniform.  A uniform knot-vector means that the knots are equally spaced.  A clamped
 * knot-vector means that the first k-knots and last k-knots are repeated, where k is the degree + 1.
 * Non-uniform means that the knot-values have no specific properties.  For all 3 types, the
 * knot-values must be non-decreasing.
 *
 *
 *
 * Here are some examples of uniform clamped knot vectors for degree 3:
 *
 *
 * <pre>
 * number of control points = 4: [0, 0, 0, 0, 1, 1, 1, 1]
 * number of control points = 7: [0, 0, 0, 0, 0.25, 0.5, 0.75, 1, 1, 1, 1]
</pre> *
 *
 *
 *
 * The following is a figure of a B-Spline generated using a uniform clamped knot vector:
 *
 *
 *
 * <center><img align="center" src="doc-files/bspline1.gif"></img></center>
 *
 *
 *
 * Here are some examples of uniform unclamped knot vectors for degree 3:
 *
 *
 * <pre>
 * number of control points = 4: [0, 0.14, 0.29, 0.43, 0.57, 0.71, 0.86, 1] (about)
 * number of control points = 7: [0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1]
</pre> *
 *
 *
 *
 * The following is a figure of a B-Spline generated using a uniform unclamped knot vector:
 *
 *
 *
 * <center><img align="center" src="doc-files/bspline2.gif"></img></center>
 *
 *
 *
 * Note: Although the knot-values in the examples are between 0 and 1, this is not a requirement.
 *
 *
 *
 * When the knot-vector is uniform clamped, the default interval is [0, 1].  When the knot-vector
 * is uniform unclamped, the default interval is [grad * degree, 1 - grad * degree], where grad is the
 * gradient or the knot-span.  Specifying the knotVectorType as UNIFORM_CLAMPED or UNIFORM_UNCLAMPED
 * means that the internal knot-vector will not be used.
 *
 *
 *
 * Note: The computation required is O(2^degree) or exponential.  Increasing the degree by 1 means
 * that twice as many computations are done.
 */
open class BSpline internal constructor(cp: ControlPath?, gi: GroupIterator?) : ParametricCurve(cp, gi) {
    private val knotVector: ValueVector = ValueVector(doubleArrayOf(0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0), 8)
    private var t_min = 0.0
    private var t_max = 1.0

    /**
     * Sets the sample-limit.  For more information on the sample-limit, see the
     * BinaryCurveApproximationAlgorithm class.  The default sample-limit is 1.
     *
     * @see org.kamaeleo.geom.curve.BinaryCurveApproximationAlgorithm
     *
     * @see .getSampleLimit
     */
    override var sampleLimit = 1
        set(limit) {
            field = limit
        }
    private var degree = 4 // the internal degree variable is always 1 plus the specified degree
    private var knotVectorType = 0
    /**
     * Returns the value of the useDefaultInterval flag.
     *
     * @see .setUseDefaultInterval
     */
    /**
     * Sets the value of the useDefaultInterval flag.  When the knot-vector type is one of UNIFORM_CLAMPED or
     * UNIFORM_UNCLAMPED and the useDefaultInterval flag is true, then default values will be computed for
     * t_min and t_max.  Otherwise t_min and t_max are used as the interval.
     *
     * @see .getUseDefaultInterval
     */
    var useDefaultInterval = true

    /**
     * Specifies the interval that the curve should define itself on.  The default interval is [0.0, 1.0].
     * When the knot-vector type is one of UNIFORM_CLAMPED or UNIFORM_UNCLAMPED and the useDefaultInterval
     * flag is true, then these values will not be used.
     *
     * @see .t_min
     * @see .t_max
     */
    fun setInterval(t_min: Double, t_max: Double) {
        this.t_min = t_min
        this.t_max = t_max
    }

    /**
     * Returns the starting interval value.
     *
     * @see .setInterval
     * @see .t_max
     */
    fun t_min(): Double {
        return t_min
    }

    /**
     * Returns the finishing interval value.
     *
     * @see .setInterval
     * @see .t_min
     */
    fun t_max(): Double {
        return t_max
    }

    /**
     * Returns the degree of the curve.
     *
     * @see .setDegree
     */
    fun getDegree(): Int {
        return degree - 1
    }

    /**
     * Sets the degree of the curve.  The degree specifies how many controls points have influence
     * when computing a single point on the curve.  Specifically, degree + 1 control points are used.
     * The degree must be greater than 0.  A degree of 1 is linear, 2 is quadratic, 3 is cubic, etc.
     * Warning: Increasing the degree by 1 doubles the number of computations required.  The default
     * degree is 3 (cubic).
     *
     * @see .getDegree
     */
    fun setDegree(d: Int) {
        degree = d + 1
    }

    /**
     * Returns the type of knot-vector to use.
     *
     * @see .setKnotVectorType
     */
    fun getKnotVectorType(): Int {
        return knotVectorType
    }

    /**
     * Sets the type of knot-vector to use.  There are 3 types, UNIFORM_CLAMPED, UNIFORM_UNCLAMPED and NON_UNIFORM.
     * NON_UNIFORM can be thought of as user specified.  UNIFORM_CLAMPED and UNIFORM_UNCLAMPED are standard
     * knot-vectors for the B-Spline.
     *
     * @see .getKnotVectorType
     */
    fun setKnotVectorType(type: Int) {
        knotVectorType = type
    }

    /**
     * There are two types of requirements for this curve, common requirements and requirements that depend on the
     * knotVectorType.  The common requirements are that the group-iterator must be in range and the number of
     * points (group size) must be greater than the degree.  If the knot-vector type is NON_UNIFORM (user specified)
     * then there are additional requirements, otherwise there are no additional requirements.
     *
     *
     * The additional requirements when the knotVectorType is NON_UNIFORM are that the internal-knot vector must have
     * an exact size of degree + numPts + 1, where degree is specified by the setDegree method and numPts is the
     * group size.  Also, the knot-vector values must be non-decreasing.
     *
     *
     * If any of these requirements are not met, then this method returns quietly.
     */
    override fun appendTo(mp: MultiPath) {
        if (!gi!!.isInRange(0, cp!!.numPoints())) return
        val numPts: Int = gi!!.groupSize
        val f = numPts - degree
        if (f < 0) return
        val x = numPts + degree
        if (knot.size < x) knot = DoubleArray(2 * x)
        var t1 = t_min
        var t2 = t_max
        if (knotVectorType == NON_UNIFORM) {
            if (knotVector.size() !== x) return
            knot[0] = knotVector.get(0)
            for (i in 1 until x) {
                knot[i] = knotVector.get(i)
                if (knot[i] < knot[i - 1]) return
            }
        } else if (knotVectorType == UNIFORM_UNCLAMPED) {
            val grad = 1.0 / (x - 1)
            for (i in 0 until x) knot[i] = i * grad
            if (useDefaultInterval) {
                t1 = (degree - 1) * grad
                t2 = 1.0 - (degree - 1) * grad
            }
        } else if (knotVectorType == UNIFORM_CLAMPED) {
            for (i in 0 until degree) knot[i] = 0.0
            var j = degree
            val grad = 1.0 / (f + 1)
            for (i in 1..f) knot[j++] = i * grad
            for (i in j until x) knot[i] = 1.0
            if (useDefaultInterval) {
                t1 = 0.0
                t2 = 1.0
            }
        }
        if (a.size < degree) {
            a = IntArray(2 * degree)
            c = IntArray(2 * degree)
        }
        val p = DoubleArray(mp.dimension + 1)
        p[mp.dimension] = t1
        eval(p)
        if (connect) mp.lineTo(p) else mp.moveTo(p)
        BinaryCurveApproximationAlgorithm.genPts(this, t1, t2, mp)
    }

    override fun eval(p: DoubleArray) {
        val dim = p.size - 1
        val t = p[dim]
        val numPts: Int = gi!!.groupSize
        gi!!.set(0, 0)
        for (i in 0 until numPts) {
            val w = N(t, i)
            //double w = N(t, i, degree);
            val loc: DoubleArray = cp!!.getPoint(gi!!.next())!!.location
            for (j in 0 until dim) p[j] += loc[j] * w //pt[i][j] * w);
        }
    }

    /**
     * Non-recursive implementation of the N-function.
     */
    fun N(t: Double, i: Int): Double {
        var d = 0.0
        for (j in 0 until degree) {
            val t1 = knot[i + j]
            val t2 = knot[i + j + 1]
            if (t >= t1 && t <= t2 && t1 != t2) {
                val dm2 = degree - 2
                for (k in degree - j - 1 downTo 0) a[k] = 0
                if (j > 0) {
                    for (k in 0 until j) c[k] = k
                    c[j] = Int.MAX_VALUE
                } else {
                    c[0] = dm2
                    c[1] = degree
                }
                var z = 0
                while (true) {
                    if (c[z] < c[z + 1] - 1) {
                        var e = 1.0
                        var bc = 0
                        var y = dm2 - j
                        var p = j - 1
                        var m = dm2
                        var n = degree
                        while (m >= 0) {
                            if (p >= 0 && c[p] == m) {
                                val w = i + bc
                                val kd = knot[w + n]
                                e *= (kd - t) / (kd - knot[w + 1])
                                bc++
                                p--
                            } else {
                                val w = i + a[y]
                                val kw = knot[w]
                                e *= (t - kw) / (knot[w + n - 1] - kw)
                                y--
                            }
                            m--
                            n--
                        }

                        // this code updates the a-counter
                        if (j > 0) {
                            var g = 0
                            var reset = false
                            while (true) {
                                a[g]++
                                if (a[g] > j) {
                                    g++
                                    reset = true
                                } else {
                                    if (reset) {
                                        for (h in g - 1 downTo 0) a[h] = a[g]
                                    }
                                    break
                                }
                            }
                        }
                        d += e

                        // this code updates the bit-counter
                        c[z]++
                        if (c[z] > dm2) break
                        for (k in 0 until z) c[k] = k
                        z = 0
                    } else {
                        z++
                    }
                }
                break // required to prevent spikes
            }
        }
        return d
    }

    /*
     The recursive implementation of the N-function (not used) is below.  In addition to being
     slower, the recursive implementation of the N-function has another problem which relates to
     the base case.  Note: the reason the recursive implementation is slower is because there
     are a lot of repetitive calculations.

     Some definitions of the N-function give the base case as t >= knot[i] && t < knot[i+1] or
     t > knot[i] && t <= knot[i+1].  To see why this is a problem, consider evaluating t on the
     range [0, 1] with the Bezier Curve knot vector [0,0,0,...,1,1,1].  Then, trying to evaluate
     t == 1 or t == 0 won't work.  Changing the base case to the one below (with equal signs on
     both comparisons) leads to a problem known as spikes.  A curve will have spikes	at places
     where t falls on a knot because when equality is used on both comparisons, the value of t
     will have 2 regions of influence.
     */
    /*private double N(double t, int i, int k) {
         if (k == 1) {
             if (t >= knot[i] && t <= knot[i+1] && knot[i] != knot[i+1]) return 1.0;
             return 0.0;
         }

         double n1 = N(t, i, k-1);
         double n2 = N(t, i+1, k-1);

         double a = 0.0;
         double b = 0.0;

         if (n1 != 0) a = (t - knot[i]) / (knot[i+k-1] - knot[i]);
         if (n2 != 0) b = (knot[i+k] - t) / (knot[i+k] - knot[i+1]);

         return a * n1 + b * n2;
     }*/
    override fun resetMemory() {
        if (a.size > 0) {
            a = IntArray(0)
            c = IntArray(0)
        }
        if (knot.size > 0) knot = DoubleArray(0)
    }

    companion object {
        private const val UNIFORM_CLAMPED = 0
        private const val UNIFORM_UNCLAMPED = 1
        private const val NON_UNIFORM = 2
        private var a = IntArray(0) // counter used for the a-function values (required length >= degree)
        private var c = IntArray(0) // counter used for bit patterns (required length >= degree)
        private var knot = DoubleArray(0) // (required length >= numPts + degree)
    }
}