package org.kamaeleo.geom.curve

import com.macrofocus.common.collection.arraycopy

/**
 * A multi-path is a series of paths (sequence of connected points) in n-dimensions.  The points and
 * move types (MOVE_TO or LINE_TO) are stored using arrays.
 *
 * @see org.kamaeleo.geom.curve.Curve
 *
 * @see ShapeMultiPath
 */
open class MultiPath(dimension: Int) {
    /**
     * Returns the dimension. The dimension is used by the BinaryCurveApproximationAlgorithm
     * to know what dimension of points to create.  If the dimension of the multi-path is greater
     * than the dimension of the control points for a curve, then an ArrayIndexOutOfBoundsException
     * will occur when the curve is appended to the MultiPath.
     */
    val dimension: Int
    private var point = Array<DoubleArray?>(2) { DoubleArray(0) }
    private var type = arrayOfNulls<Any>(point.size)

    /**
     * Returns the size counter.
     */
    var numPoints = 0
        private set
    /**
     * Returns the flatness.  The flatness is used by the BinaryCurveApproximationAlgorithm to
     * determine how closely the line segements formed by the points of this multi-path should
     * approximate a given curve.  The default flatness value is 1.0.  When using curves in a
     * graphics environment, the flatness usually inversely proportional to the scale.
     *
     * @see .setFlatness
     */
    /**
     * Sets the flatness.  As the flatness value gets closer to zero, the BinaryCurveApproximationAlgorithm
     * generates more points.
     *
     * @see .getFlatness
     */
    var flatness = 1.0
        set(f) {
            field = f
        }

    /**
     * Returns a reference to the point at the specified index.
     */
    operator fun get(index: Int): DoubleArray {
        return point[index]!!
    }

    /**
     * Returns the type of the point at the specified index.  The type can other be MultiPath.MOVE_TO
     * or MultiPath.LINE_TO.
     */
    fun getType(index: Int): Any? {
        return type[index]
    }

    /**
     * Appends a point of type LINE_TO.  If the size counter is 0 then the request is interpretted as a
     * MOVE_TO request.
     *
     * @see .moveTo
     */
    fun lineTo(p: DoubleArray?) {
        append(p, LINE_TO)
    }

    private fun append(p: DoubleArray?, t: Any) {
        var t: Any? = t
        if (numPoints == 0) t = MOVE_TO
        ensureCapacity(numPoints + 1)
        point[numPoints] = p
        type[numPoints] = t
        numPoints++
    }

    /**
     * Checks that the point array has the specified capacity, otherwise the capacity of the point
     * array is increased to be the maximum between twice the current capacity and the specified capacity.
     */
    fun ensureCapacity(capacity: Int) {
        if (point.size < capacity) {
            var x = 2 * point.size
            if (x < capacity) x = capacity
            val p2 = arrayOfNulls<DoubleArray>(x)
            arraycopy(point, 0, p2, 0, numPoints)
            val t2 = arrayOfNulls<Any>(x)
            arraycopy(type, 0, t2, 0, numPoints)
            point = p2
            type = t2
        }
    }

    /**
     * Appends a point of type MOVE_TO.
     *
     * @see .lineTo
     */
    fun moveTo(p: DoubleArray?) {
        append(p, MOVE_TO)
    }

    companion object {
        val MOVE_TO = Any()
        val LINE_TO = Any()
    }

    /**
     * Constructs a multi-path specifying the minimum required dimension of each point appended
     * to this multi-path.
     */
    init {
        this.dimension = dimension
    }
}