package org.kamaeleo.geom.curve

import org.kamaeleo.geom.*

/**
 * The ShapeMultiPath is-a MultiPath and implements the java.awt.Shape interface.
 * Here is an example of how to use a ShapeMultiPath:
 *
 *
 * <pre>
 * ControlPath cp = new ControlPath();
 * cp.addPoint(...); // add points
 * Curve c = new BezierCurve(cp, new GroupIterator("0:n-1"));
 *
 * ShapeMultiPath smp = new ShapeMultiPath();
 * c.appendTo(smp);
 *
 * Graphics2D g = ...;
 * g.draw(smp);
 *
</pre> *
 */
class ShapeMultiPath(dimension: Int) : MultiPath(dimension), Shape {
    /**
     * Returns the value of the winding rule. The default value is PathIterator.WIND_EVEN_ODD.
     */
    val windingRule: Int = org.kamaeleo.geom.PathIterator.WIND_EVEN_ODD
    private val ai0 = 0
    private val ai1 = 1

    /**
     * Returns a new integer array with the basis vectors.  The default basis vectors are {0, 1}.
     */
    val basisVectors: IntArray
        get() = intArrayOf(ai0, ai1)
    //---------------------------------------------------------------
    //---------------------------------------------------------------
    /**
     * Computes the bounding box of the points.  When computing the bounding box, a point is considered if it is
     * of type LINE_TO or it is of type MOVE_TO and the next point is of type LINE_TO.  A value of null is
     * returned if there is not enough data to define a bounding box.
     */
    override val bounds2D: org.kamaeleo.geom.Rectangle2D
        get() {
            val n: Int = numPoints
            var x1 = Double.MAX_VALUE
            var y1 = Double.MAX_VALUE
            var x2 = -Double.MAX_VALUE
            var y2 = -Double.MAX_VALUE
            var defined = false
            for (i in 0 until n) {
                val p: DoubleArray = get(i)
                var b = false
                if (getType(i) === MultiPath.MOVE_TO) {
                    if (i < n - 1 && getType(i + 1) === MultiPath.LINE_TO) b = true
                } else {
                    b = true
                }
                if (b) {
                    defined = true
                    if (p[ai0] < x1) x1 = p[ai0]
                    if (p[ai1] < y1) y1 = p[ai1]
                    if (p[ai0] > x2) x2 = p[ai0]
                    if (p[ai1] > y2) y2 = p[ai1]
                }
            }
            return Rectangle2D.Double(x1, y1, x2 - x1, y2 - y1)
        }

    /**
     * See the contains(x, y) method.
     *
     * @see .contains
     */
    override operator fun contains(p: Point2D): Boolean {
        return contains(p.x, p.y)
    }

    /**
     * See the intersects(x, y, w, h) method.
     *
     * @see .intersects
     */
    override fun intersects(r: Rectangle2D): Boolean {
        return intersects(r.x, r.y, r.width, r.height)
    }

    /**
     * Returns a new PathIterator object.
     */
    override val pathIterator: org.kamaeleo.geom.PathIterator
        get() = getPathIterator(null)

    override fun getPathIterator(at: AffineTransform?): PathIterator {
        return ShapeMultiPathIterator(this, at)
    }

    /**
     * Returns a new PathIterator object.  The flatness parameter is ignored since a multi-path, by
     * definition, is already flat.
     */
    override fun getFlattenPathIterator(flatness: Double): PathIterator {
        return ShapeMultiPathIterator(this, null)
    }

    /**
     * Returns true if the point is contained inside the shape. Otherwise false is returned.
     */
    fun contains(x: Double, y: Double): Boolean {
        val cross = pointCrossingsForPath(pathIterator, x, y)
        return if (windingRule == PathIterator.WIND_NON_ZERO) cross != 0 else cross and 1 != 0
    }

    /**
     * Returns true only if the shape contains all points of the rectangle. First, if any of
     * the four corners is not contained in the shape then false is returned.  Now we know
     * that all four corners are inside the shape.  Next, we check to see if any line segment
     * of this shape intersects any of the 4 line segments formed by the rectangle.  If there
     * is an intersection, then false is returned.  Otherwise true is returned.
     */
    fun contains(x1: Double, y1: Double, w: Double, h: Double): Boolean {
        if (!contains(x1, y1)) return false
        val y2 = y1 + h
        if (!contains(x1, y2)) return false
        val x2 = x1 + w
        if (!contains(x2, y1)) return false
        if (!contains(x2, y2)) return false
        val n: Int = numPoints
        if (n == 0) return false
        var p: DoubleArray = get(0)
        var xb = p[ai0]
        var yb = p[ai1]
        for (i in 1 until n) {
            p = get(i)
            val xa = p[ai0]
            val ya = p[ai1]
            if (getType(i) === MultiPath.LINE_TO) {
                if (Geom.getSegSegIntersection(xa, ya, xb, yb, x1, y1, x2, y1, null) === Geom.INTERSECT) return false
                if (Geom.getSegSegIntersection(xa, ya, xb, yb, x1, y1, x1, y2, null) === Geom.INTERSECT) return false
                if (Geom.getSegSegIntersection(xa, ya, xb, yb, x1, y2, x2, y2, null) === Geom.INTERSECT) return false
                if (Geom.getSegSegIntersection(xa, ya, xb, yb, x2, y1, x2, y2, null) === Geom.INTERSECT) return false
            }
            xb = xa
            yb = ya
        }
        return true
    }
    //---------------------------------------------------------------
    /**
     * See the contains(x, y, w, h) method.
     *
     * @see .contains
     */
    operator fun contains(r: Rectangle2D): Boolean {
        return contains(r.x, r.y, r.width, r.height)
    }

    /**
     * This method returns true if any line segment in this multi-path intersects any of the
     * 4 line segments formed by the rectangle or any corner of the rectangle is inside the
     * shape or any point of the shape is inside the rectangle.  Otherwise false is returned.
     */
    fun intersects(x1: Double, y1: Double, w: Double, h: Double): Boolean {
        if (contains(x1, y1)) return true
        val y2 = y1 + h
        if (contains(x1, y2)) return true
        val x2 = x1 + w
        if (contains(x2, y1)) return true
        if (contains(x2, y2)) return true
        val n: Int = numPoints
        if (n == 0) return false
        var p: DoubleArray = get(0)
        var xb = p[ai0]
        var yb = p[ai1]
        for (i in 1 until n) {
            p = get(i)
            val xa = p[ai0]
            val ya = p[ai1]
            if (getType(i) === MultiPath.LINE_TO) {
                if (Geom.getSegSegIntersection(xa, ya, xb, yb, x1, y1, x2, y1, null) === Geom.INTERSECT) return true
                if (Geom.getSegSegIntersection(xa, ya, xb, yb, x1, y1, x1, y2, null) === Geom.INTERSECT) return true
                if (Geom.getSegSegIntersection(xa, ya, xb, yb, x1, y2, x2, y2, null) === Geom.INTERSECT) return true
                if (Geom.getSegSegIntersection(xa, ya, xb, yb, x2, y1, x2, y2, null) === Geom.INTERSECT) return true
                if (xa >= x1 && ya >= y1 && xa <= x2 && ya <= y2) return true
                if (xb >= x1 && yb >= y1 && xb <= x2 && yb <= y2) return true
            }
            xb = xa
            yb = ya
        }
        return false
    }

    companion object {
        /**
         * Calculates the number of times the given path
         * crosses the ray extending to the right from (px,py).
         * If the point lies on a part of the path,
         * then no crossings are counted for that intersection.
         * +1 is added for each crossing where the Y coordinate is increasing
         * -1 is added for each crossing where the Y coordinate is decreasing
         * The return value is the sum of all crossings for every segment in
         * the path.
         * The path must start with a SEG_MOVETO, otherwise an exception is
         * thrown.
         * The caller must check p[xy] for NaN values.
         * The caller may also reject infinite p[xy] values as well.
         */
        fun pointCrossingsForPath(
            pi: PathIterator,
            px: Double, py: Double
        ): Int {
            if (pi.isDone) {
                return 0
            }
            val coords = DoubleArray(6)
            check(!(pi.currentSegment(coords) != PathIterator.SEG_MOVETO)) {
                "missing initial moveto " +
                        "in path definition"
            }
            pi.next()
            var movx = coords[0]
            var movy = coords[1]
            var curx = movx
            var cury = movy
            var crossings = 0
            while (!pi.isDone) {
                var endy: Double
                var endx: Double
                when (pi.currentSegment(coords)) {
                    PathIterator.SEG_MOVETO -> {
                        if (cury != movy) {
                            crossings += pointCrossingsForLine(
                                px, py,
                                curx, cury,
                                movx, movy
                            )
                        }
                        run {
                            curx = coords[0]
                            movx = curx
                        }
                        run {
                            cury = coords[1]
                            movy = cury
                        }
                    }
                    PathIterator.SEG_LINETO -> {
                        endx = coords[0]
                        endy = coords[1]
                        crossings += pointCrossingsForLine(
                            px, py,
                            curx, cury,
                            endx, endy
                        )
                        curx = endx
                        cury = endy
                    }
                    PathIterator.SEG_QUADTO -> {
                        endx = coords[2]
                        endy = coords[3]
                        crossings += pointCrossingsForQuad(
                            px, py,
                            curx, cury,
                            coords[0], coords[1],
                            endx, endy, 0
                        )
                        curx = endx
                        cury = endy
                    }
                    PathIterator.SEG_CUBICTO -> {
                        endx = coords[4]
                        endy = coords[5]
                        crossings += pointCrossingsForCubic(
                            px, py,
                            curx, cury,
                            coords[0], coords[1],
                            coords[2], coords[3],
                            endx, endy, 0
                        )
                        curx = endx
                        cury = endy
                    }
                    PathIterator.SEG_CLOSE -> {
                        if (cury != movy) {
                            crossings += pointCrossingsForLine(
                                px, py,
                                curx, cury,
                                movx, movy
                            )
                        }
                        curx = movx
                        cury = movy
                    }
                }
                pi.next()
            }
            if (cury != movy) {
                crossings += pointCrossingsForLine(
                    px, py,
                    curx, cury,
                    movx, movy
                )
            }
            return crossings
        }
        //------------------------------------------------------------------------------------------
        // methods for Shape interface:
        /**
         * Calculates the number of times the line from (x0,y0) to (x1,y1)
         * crosses the ray extending to the right from (px,py).
         * If the point lies on the line, then no crossings are recorded.
         * +1 is returned for a crossing where the Y coordinate is increasing
         * -1 is returned for a crossing where the Y coordinate is decreasing
         */
        fun pointCrossingsForLine(
            px: Double, py: Double,
            x0: Double, y0: Double,
            x1: Double, y1: Double
        ): Int {
            if (py < y0 && py < y1) return 0
            if (py >= y0 && py >= y1) return 0
            // assert(y0 != y1);
            if (px >= x0 && px >= x1) return 0
            if (px < x0 && px < x1) return if (y0 < y1) 1 else -1
            val xintercept = x0 + (py - y0) * (x1 - x0) / (y1 - y0)
            if (px >= xintercept) return 0
            return if (y0 < y1) 1 else -1
        }

        /**
         * Calculates the number of times the quad from (x0,y0) to (x1,y1)
         * crosses the ray extending to the right from (px,py).
         * If the point lies on a part of the curve,
         * then no crossings are counted for that intersection.
         * the level parameter should be 0 at the top-level call and will count
         * up for each recursion level to prevent infinite recursion
         * +1 is added for each crossing where the Y coordinate is increasing
         * -1 is added for each crossing where the Y coordinate is decreasing
         */
        fun pointCrossingsForQuad(
            px: Double, py: Double,
            x0: Double, y0: Double,
            xc: Double, yc: Double,
            x1: Double, y1: Double, level: Int
        ): Int {
            var xc = xc
            var yc = yc
            if (py < y0 && py < yc && py < y1) return 0
            if (py >= y0 && py >= yc && py >= y1) return 0
            // Note y0 could equal y1...
            if (px >= x0 && px >= xc && px >= x1) return 0
            if (px < x0 && px < xc && px < x1) {
                if (py >= y0) {
                    if (py < y1) return 1
                } else {
                    // py < y0
                    if (py >= y1) return -1
                }
                // py outside of y01 range, and/or y0==y1
                return 0
            }
            // double precision only has 52 bits of mantissa
            if (level > 52) return pointCrossingsForLine(px, py, x0, y0, x1, y1)
            val x0c = (x0 + xc) / 2
            val y0c = (y0 + yc) / 2
            val xc1 = (xc + x1) / 2
            val yc1 = (yc + y1) / 2
            xc = (x0c + xc1) / 2
            yc = (y0c + yc1) / 2
            return if (xc.isNaN() || yc.isNaN()) {
                // [xy]c are NaN if any of [xy]0c or [xy]c1 are NaN
                // [xy]0c or [xy]c1 are NaN if any of [xy][0c1] are NaN
                // These values are also NaN if opposing infinities are added
                0
            } else pointCrossingsForQuad(
                px, py,
                x0, y0, x0c, y0c, xc, yc,
                level + 1
            ) +
                    pointCrossingsForQuad(
                        px, py,
                        xc, yc, xc1, yc1, x1, y1,
                        level + 1
                    )
        }

        /**
         * Calculates the number of times the cubic from (x0,y0) to (x1,y1)
         * crosses the ray extending to the right from (px,py).
         * If the point lies on a part of the curve,
         * then no crossings are counted for that intersection.
         * the level parameter should be 0 at the top-level call and will count
         * up for each recursion level to prevent infinite recursion
         * +1 is added for each crossing where the Y coordinate is increasing
         * -1 is added for each crossing where the Y coordinate is decreasing
         */
        fun pointCrossingsForCubic(
            px: Double, py: Double,
            x0: Double, y0: Double,
            xc0: Double, yc0: Double,
            xc1: Double, yc1: Double,
            x1: Double, y1: Double, level: Int
        ): Int {
            var xc0 = xc0
            var yc0 = yc0
            var xc1 = xc1
            var yc1 = yc1
            if (py < y0 && py < yc0 && py < yc1 && py < y1) return 0
            if (py >= y0 && py >= yc0 && py >= yc1 && py >= y1) return 0
            // Note y0 could equal yc0...
            if (px >= x0 && px >= xc0 && px >= xc1 && px >= x1) return 0
            if (px < x0 && px < xc0 && px < xc1 && px < x1) {
                if (py >= y0) {
                    if (py < y1) return 1
                } else {
                    // py < y0
                    if (py >= y1) return -1
                }
                // py outside of y01 range, and/or y0==yc0
                return 0
            }
            // double precision only has 52 bits of mantissa
            if (level > 52) return pointCrossingsForLine(px, py, x0, y0, x1, y1)
            var xmid = (xc0 + xc1) / 2
            var ymid = (yc0 + yc1) / 2
            xc0 = (x0 + xc0) / 2
            yc0 = (y0 + yc0) / 2
            xc1 = (xc1 + x1) / 2
            yc1 = (yc1 + y1) / 2
            val xc0m = (xc0 + xmid) / 2
            val yc0m = (yc0 + ymid) / 2
            val xmc1 = (xmid + xc1) / 2
            val ymc1 = (ymid + yc1) / 2
            xmid = (xc0m + xmc1) / 2
            ymid = (yc0m + ymc1) / 2
            return if (xmid.isNaN() || ymid.isNaN()) {
                // [xy]mid are NaN if any of [xy]c0m or [xy]mc1 are NaN
                // [xy]c0m or [xy]mc1 are NaN if any of [xy][c][01] are NaN
                // These values are also NaN if opposing infinities are added
                0
            } else pointCrossingsForCubic(
                px, py,
                x0, y0, xc0, yc0,
                xc0m, yc0m, xmid, ymid, level + 1
            ) +
                    pointCrossingsForCubic(
                        px, py,
                        xmid, ymid, xmc1, ymc1,
                        xc1, yc1, x1, y1, level + 1
                    )
        }
    }

    /**
     * Constructs a new ShapeMultiPath with the specified dimension requirement.
     */
    init {
    }
}