package org.kamaeleo.graphics

import com.macrofocus.common.math.convertDegreesToRadians
import com.macrofocus.common.math.isNaN
import org.kamaeleo.font.CPFont
import org.kamaeleo.font.CPFontMetrics
import org.kamaeleo.geom.*
import kotlin.math.abs
import kotlin.math.sign

abstract class AbstractIGraphics : IGraphics {
    protected var value: Any? = null
    protected var callback: ICallback<Any?>? = null

    override fun isThreadSafe(): Boolean {
        return false
    }

    override fun setLineDash(dashPatern: FloatArray?) {
        throw UnsupportedOperationException()
    }

    fun drawShape(shape: Shape?) {
        draw(shape)
    }

    open fun drawPoint(point: Point2D?) {}
    open fun fillCircle(rectangle: Rectangle2D) {
        fill(Ellipse2D.Double(rectangle.x, rectangle.y, rectangle.width, rectangle.height))
    }

    open fun fillRectangle2D(rectangle: Rectangle2D?) {
        fill(rectangle)
    }

    override fun fillRectangle(x: Int, y: Int, width: Int, height: Int) {
        fill(Rectangle(x, y, width, height))
    }

    override fun drawRectange(x: Int, y: Int, width: Int, height: Int) {
        draw(Rectangle(x, y, width, height))
    }

    fun fillShape(shape: Shape?) {
        fill(shape)
    }

    fun <T> pickDraw(geometry: Shape, point: Point2D?, value: T, callback: ICallback<T>) {
        pick(geometry, point, value, callback)
        draw(geometry)
    }

    fun <T> pickFill(geometry: Shape, point: Point2D?, value: T, callback: ICallback<T>) {
        pick(geometry, point, value, callback)
        fill(geometry)
    }

    private fun <T> pick(geometry: Shape, point: Point2D?, value: T, callback: ICallback<T>) {
        if (point != null && geometry.contains(point)) {
            this.value = value
            this.callback = callback as ICallback<Any?>
        }
    }

    protected abstract fun draw(geometry: Shape?)
    protected abstract fun fill(geometry: Shape?)
    override fun beginPick() {
        value = null
        callback = null
    }

    override fun endPick() {
        if (callback != null) {
            callback!!.callingBack(value)
        }
    }

    override fun setTextBaseline(baseline: IGraphics.TextBaseline?) {}
    override fun setFont(font: CPFont?) {
        throw UnsupportedOperationException("Not implemented")
    }

    open fun getStringBounds(text: String?): Rectangle2D {
        throw UnsupportedOperationException("Not implemented")
    }

    override fun getStringWidth(text: String?): Float {
        throw UnsupportedOperationException("Not implemented")
    }

    override fun getStringHeight(text: String?): Float {
        throw UnsupportedOperationException("Not implemented")
    }

    override fun drawString(text: String?, x: Float, y: Float) {
        throw UnsupportedOperationException("Not implemented")
    }

    /**
     *
     * @param text
     * @param centerX
     * @param centerY
     * @param radius
     * @param startAngle In degrees, Where the text will be shown. 0 degrees if the top of the circle
     * @param extent
     */
    override fun drawCircularString(
        text: String?,
        centerX: Float,
        centerY: Float,
        radius: Float,
        startAngle: Float,
        extent: Float
    ) {
        val clockwise = 1
        val inwardFacing = true
        val kerning = 0
        translate(centerX.toInt(), centerY.toInt())
        var rotated: Double = convertDegreesToRadians(normalizeDegrees(startAngle.toDouble()))
        rotate(rotated)
        val textHeight = getStringHeight(text)
        for (j in 0 until text!!.length) {
            val string = text[j].toString()
            val charWidth = getStringWidth(string)

            // rotate half letter
            val theta1 = charWidth / 2 / (radius - textHeight) * clockwise
            rotate(theta1.toDouble())
            rotated += theta1.toDouble()

            // draw the character at "top" or "bottom"
            // depending on inward or outward facing
            drawString(string, 0f, (if (inwardFacing) 1 else -1) * (0 - radius + textHeight / 2))
            val theta2 = (charWidth / 2 + kerning) / (radius - textHeight) * clockwise
            rotate(theta2.toDouble()) // rotate half letter
            rotated += theta2.toDouble()
        }
        rotate(-rotated)
        translate((-centerX).toInt(), (-centerY).toInt())
    }

    override fun getAscent(): Double {
        return 0.0
    }

    override fun getDescent(): Double {
        return 0.0
    }

    override fun getFontMetrics(): CPFontMetrics? {
        TODO("Not yet implemented")
    }

    companion object {
        /*
     * Normalizes the specified angle into the range -180 to 180.
     */
        fun normalizeDegrees(angle: Double): Double {
            var angle = angle
            if (angle > 180.0) {
                if (angle <= 180.0 + 360.0) {
                    angle = angle - 360.0
                } else {
                    angle = IEEEremainder(angle, 360.0)
                    // IEEEremainder can return -180 here for some input values...
                    if (angle == -180.0) {
                        angle = 180.0
                    }
                }
            } else if (angle <= -180.0) {
                if (angle > -180.0 - 360.0) {
                    angle = angle + 360.0
                } else {
                    angle = IEEEremainder(angle, 360.0)
                    // IEEEremainder can return -180 here for some input values...
                    if (angle == -180.0) {
                        angle = 180.0
                    }
                }
            }
            return angle
        }

        fun IEEEremainder(f1: Double, f2: Double): Double {
            val r: Double = abs(f1 % f2)
            return if (isNaN(r) || r == f2 || r <= abs(f2) / 2.0) {
                r
            } else {
                sign(f1) * (r - f2)
            }
        }
    }
}
