/*
 * Copyright (c) 2016 Vivid Solutions.
 * Copyright (c) 2022 Macrofocus GmbH and Luc Girardin.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * and Eclipse Distribution License v. 1.0 which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v20.html
 * and the Eclipse Distribution License is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package org.locationtech.jts.noding

import org.locationtech.jts.algorithm.LineIntersector
import org.locationtech.jts.algorithm.RobustLineIntersector
import org.locationtech.jts.geom.Coordinate
import org.locationtech.jts.geom.TopologyException
import org.locationtech.jts.io.WKTWriter
import kotlin.jvm.JvmStatic

/**
 * Validates that a collection of [SegmentString]s is correctly noded.
 * Indexing is used to improve performance.
 * By default validation stops after a single
 * non-noded intersection is detected.
 * Alternatively, it can be requested to detect all intersections
 * by using [.setFindAllIntersections].
 *
 * The validator does not check for topology collapse situations
 * (e.g. where two segment strings are fully co-incident).
 *
 * The validator checks for the following situations which indicated incorrect noding:
 *
 *  * Proper intersections between segments (i.e. the intersection is interior to both segments)
 *  * Intersections at an interior vertex (i.e. with an endpoint or another interior vertex)
 *
 *
 * The client may either test the [.isValid] condition,
 * or request that a suitable [TopologyException] be thrown.
 *
 * @version 1.7
 *
 * @see NodingIntersectionFinder
 */
class FastNodingValidator
/**
 * Creates a new noding validator for a given set of linework.
 *
 * @param segStrings a collection of [SegmentString]s
 */(private val segStrings: Collection<SegmentString>?) {
    private val li: LineIntersector = RobustLineIntersector()
    private var findAllIntersections = false
    private var segInt: NodingIntersectionFinder? = null
    private var isValid = true
    fun setFindAllIntersections(findAllIntersections: Boolean) {
        this.findAllIntersections = findAllIntersections
    }

    /**
     * Gets a list of all intersections found.
     * Intersections are represented as [Coordinate]s.
     * List is empty if none were found.
     *
     * @return a list of Coordinate
     */
    val intersections: MutableList<Coordinate>
        get() = segInt!!.getIntersections()

    /**
     * Checks for an intersection and
     * reports if one is found.
     *
     * @return true if the arrangement contains an interior intersection
     */
    fun isValid(): Boolean {
        execute()
        return isValid
    }

    /**
     * Returns an error message indicating the segments containing
     * the intersection.
     *
     * @return an error message documenting the intersection location
     */
    val errorMessage: String
        get() {
            if (isValid) return "no intersections found"
            val intSegs: Array<Coordinate?>? = segInt!!.intersectionSegments
            return (("found non-noded intersection between "
                    + WKTWriter.toLineString(intSegs!![0]!!, intSegs[1]!!)
                    ) + " and "
                    + WKTWriter.toLineString(intSegs[2]!!, intSegs[3]!!))
        }

    /**
     * Checks for an intersection and throws
     * a TopologyException if one is found.
     *
     * @throws TopologyException if an intersection is found
     */
    fun checkValid() {
        execute()
        if (!isValid) throw TopologyException(errorMessage, segInt!!.intersection)
    }

    private fun execute() {
        if (segInt != null) return
        checkInteriorIntersections()
    }

    private fun checkInteriorIntersections() {
        /**
         * MD - It may even be reliable to simply check whether
         * end segments (of SegmentStrings) have an interior intersection,
         * since noding should have split any true interior intersections already.
         */
        isValid = true
        segInt = NodingIntersectionFinder(li)
        segInt!!.setFindAllIntersections(findAllIntersections)
        val noder: MCIndexNoder = MCIndexNoder()
        noder.setSegmentIntersector(segInt)
        noder.computeNodes(segStrings!!)
        if (segInt!!.hasIntersection()) {
            isValid = false
            return
        }
    }

    companion object {
        /**
         * Gets a list of all intersections found.
         * Intersections are represented as [Coordinate]s.
         * List is empty if none were found.
         *
         * @param segStrings a collection of SegmentStrings
         * @return a list of Coordinate
         */
        @JvmStatic
        fun computeIntersections(segStrings: Collection<SegmentString>?): MutableList<Coordinate> {
            val nv = FastNodingValidator(segStrings)
            nv.setFindAllIntersections(true)
            nv.isValid()
            return nv.intersections
        }
    }
}