/*
 * Copyright (c) 2016 Vivid Solutions.
 * Copyright (c) 2022 Macrofocus GmbH and Luc Girardin.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * and Eclipse Distribution License v. 1.0 which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v20.html
 * and the Eclipse Distribution License is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package org.locationtech.jts.operation.overlay

import org.locationtech.jts.algorithm.PointLocator
import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.geom.GeometryFactory
import org.locationtech.jts.geomgraph.DirectedEdge
import org.locationtech.jts.geomgraph.DirectedEdgeStar
import org.locationtech.jts.geomgraph.Edge
import org.locationtech.jts.geomgraph.Node
import org.locationtech.jts.util.Assert.isTrue

/**
 * Forms JTS LineStrings out of a the graph of [DirectedEdge]s
 * created by an [OverlayOp].
 *
 * @version 1.7
 */
class LineBuilder(
    private val op: OverlayOp,
    private val geometryFactory: GeometryFactory,
    private val ptLocator: PointLocator
) {
    private val lineEdgesList: MutableList<Any?> = ArrayList()
    private val resultLineList: MutableList<Geometry> = ArrayList()

    /**
     * @return a list of the LineStrings in the result of the specified overlay operation
     */
    fun build(opCode: Int): List<Geometry> {
        findCoveredLineEdges()
        collectLines(opCode)
        //labelIsolatedLines(lineEdgesList);
        buildLines(opCode)
        return resultLineList
    }

    /**
     * Find and mark L edges which are "covered" by the result area (if any).
     * L edges at nodes which also have A edges can be checked by checking
     * their depth at that node.
     * L edges at nodes which do not have A edges can be checked by doing a
     * point-in-polygon test with the previously computed result areas.
     */
    private fun findCoveredLineEdges() {
        // first set covered for all L edges at nodes which have A edges too
        val nodeit: Iterator<*> = op.graph.nodes.iterator()
        while (nodeit.hasNext()) {
            val node = nodeit.next() as Node
            //node.print(System.out);
            (node.edges as DirectedEdgeStar).findCoveredLineEdges()
        }
        /**
         * For all L edges which weren't handled by the above,
         * use a point-in-poly test to determine whether they are covered
         */
        val it: Iterator<Any?> = op.graph.getEdgeEnds().iterator()
        while (it.hasNext()) {
            val de = it.next() as DirectedEdge
            val e = de.edge
            if (de.isLineEdge && !e.isCoveredSet) {
                val isCovered = op.isCoveredByA(de.coordinate!!)
                e.isCovered = isCovered
            }
        }
    }

    private fun collectLines(opCode: Int) {
        val it: Iterator<*> = op.graph.getEdgeEnds().iterator()
        while (it.hasNext()) {
            val de = it.next() as DirectedEdge
            collectLineEdge(de, opCode, lineEdgesList)
            collectBoundaryTouchEdge(de, opCode, lineEdgesList)
        }
    }

    /**
     * Collect line edges which are in the result.
     * Line edges are in the result if they are not part of
     * an area boundary, if they are in the result of the overlay operation,
     * and if they are not covered by a result area.
     *
     * @param de the directed edge to test
     * @param opCode the overlap operation
     * @param edges the list of included line edges
     */
    private fun collectLineEdge(de: DirectedEdge, opCode: Int, edges: MutableList<Any?>) {
        val label = de.label!!
        val e = de.edge
        // include L edges which are in the result
        if (de.isLineEdge) {
            if (!de.isVisited && OverlayOp.isResultOfOp(label, opCode) && !e.isCovered) {
//Debug.println("de: " + de.getLabel());
//Debug.println("edge: " + e.getLabel());
                edges.add(e)
                de.setVisitedEdge(true)
            }
        }
    }

    /**
     * Collect edges from Area inputs which should be in the result but
     * which have not been included in a result area.
     * This happens ONLY:
     *
     *  * during an intersection when the boundaries of two
     * areas touch in a line segment
     *  *  OR as a result of a dimensional collapse.
     *
     */
    private fun collectBoundaryTouchEdge(de: DirectedEdge, opCode: Int, edges: MutableList<Any?>) {
        val label = de.label
        if (de.isLineEdge) return  // only interested in area edges
        if (de.isVisited) return  // already processed
        if (de.isInteriorAreaEdge) return  // added to handle dimensional collapses
        if (de.edge.isInResult) return  // if the edge linework is already included, don't include it again

        // sanity check for labelling of result edgerings
        isTrue(!(de.isInResult || de.sym!!.isInResult) || !de.edge.isInResult)

        // include the linework if it's in the result of the operation
        if (OverlayOp.isResultOfOp(label!!, opCode)
            && opCode == OverlayOp.INTERSECTION
        ) {
            edges.add(de.edge)
            de.setVisitedEdge(true)
        }
    }

    private fun buildLines(opCode: Int) {
        val it: Iterator<*> = lineEdgesList.iterator()
        while (it.hasNext()) {
            val e = it.next() as Edge
            // Label label = e.getLabel();
            val line = geometryFactory.createLineString(e.getCoordinates())
            resultLineList.add(line)
            e.isInResult = true
        }
    }

    private fun labelIsolatedLines(edgesList: List<*>) {
        val it = edgesList.iterator()
        while (it.hasNext()) {
            val e = it.next() as Edge
            val label = e.label
            //n.print(System.out);
            if (e.isIsolated) {
                if (label!!.isNull(0)) labelIsolatedLine(e, 0) else labelIsolatedLine(e, 1)
            }
        }
    }

    /**
     * Label an isolated node with its relationship to the target geometry.
     */
    private fun labelIsolatedLine(e: Edge, targetIndex: Int) {
        val loc = ptLocator.locate(e.getCoordinate()!!, op.getArgGeometry(targetIndex)!!)
        e.label!!.setLocation(targetIndex, loc)
    }
}