/*
 * Copyright (c) 2016 Vivid Solutions.
 * Copyright (c) 2022 Macrofocus GmbH and Luc Girardin.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * and Eclipse Distribution License v. 1.0 which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v20.html
 * and the Eclipse Distribution License is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package org.locationtech.jts.triangulate

import org.locationtech.jts.geom.Coordinate
import org.locationtech.jts.geom.LineSegment

/**
 * A strategy for finding constraint split points which attempts to maximise the length of the split
 * segments while preventing further encroachment. (This is not always possible for narrow angles).
 *
 * @author Martin Davis
 */
class NonEncroachingSplitPointFinder : ConstraintSplitPointFinder {
    /**
     * A basic strategy for finding split points when nothing extra is known about the geometry of
     * the situation.
     *
     * @param seg the encroached segment
     * @param encroachPt the encroaching point
     * @return the point at which to split the encroached segment
     */
    override fun findSplitPoint(seg: Segment, encroachPt: Coordinate?): Coordinate? {
        val lineSeg: LineSegment = seg.getLineSegment()
        val segLen = lineSeg.length
        val midPtLen = segLen / 2
        val splitSeg: SplitSegment =
            SplitSegment(lineSeg)
        val projPt = projectedSplitPoint(seg, encroachPt)

        /**
         * Compute the largest diameter (length) that will produce a split segment which is not
         * still encroached upon by the encroaching point (The length is reduced slightly by a
         * safety factor)
         */
        val nonEncroachDiam = projPt.distance(encroachPt!!) * 2 * 0.8 // .99;
        var maxSplitLen = nonEncroachDiam
        if (maxSplitLen > midPtLen) {
            maxSplitLen = midPtLen
        }
        splitSeg.setMinimumLength(maxSplitLen)
        splitSeg.splitAt(projPt)
        return splitSeg.getSplitPoint()
    }

    companion object {
        /**
         * Computes a split point which is the projection of the encroaching point on the segment
         *
         * @param seg
         * @param encroachPt
         * @return a split point on the segment
         */
        fun projectedSplitPoint(
            seg: Segment,
            encroachPt: Coordinate?
        ): Coordinate {
            val lineSeg: LineSegment = seg.getLineSegment()
            return lineSeg.project(encroachPt!!)
        }
    }
}