/*
 * Copyright (c) 2016 Vivid Solutions.
 * Copyright (c) 2022 Macrofocus GmbH and Luc Girardin.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * and Eclipse Distribution License v. 1.0 which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v20.html
 * and the Eclipse Distribution License is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package org.locationtech.jts.geom

import org.locationtech.jts.io.OrdinateFormat
import org.locationtech.jts.legacy.Math.isNaN
import org.locationtech.jts.legacy.Math.min
import kotlin.jvm.JvmOverloads
import kotlin.jvm.JvmStatic

/**
 * Utility functions for manipulating [CoordinateSequence]s
 *
 * @version 1.7
 */
object CoordinateSequences {
    /**
     * Reverses the coordinates in a sequence in-place.
     *
     * @param seq the coordinate sequence to reverse
     */
    @JvmStatic
    fun reverse(seq: CoordinateSequence) {
        if (seq.size() <= 1) return
        val last: Int = seq.size() - 1
        val mid = last / 2
        for (i in 0..mid) {
            swap(seq, i, last - i)
        }
    }

    /**
     * Swaps two coordinates in a sequence.
     *
     * @param seq the sequence to modify
     * @param i the index of a coordinate to swap
     * @param j the index of a coordinate to swap
     */
    fun swap(seq: CoordinateSequence, i: Int, j: Int) {
        if (i == j) return
        for (dim in 0 until seq.dimension) {
            val tmp: Double = seq.getOrdinate(i, dim)
            seq.setOrdinate(i, dim, seq.getOrdinate(j, dim))
            seq.setOrdinate(j, dim, tmp)
        }
    }

    /**
     * Copies a section of a [CoordinateSequence] to another [CoordinateSequence].
     * The sequences may have different dimensions;
     * in this case only the common dimensions are copied.
     *
     * @param src the sequence to copy from
     * @param srcPos the position in the source sequence to start copying at
     * @param dest the sequence to copy to
     * @param destPos the position in the destination sequence to copy to
     * @param length the number of coordinates to copy
     */
    @JvmStatic
    fun copy(
        src: CoordinateSequence,
        srcPos: Int,
        dest: CoordinateSequence,
        destPos: Int,
        length: Int
    ) {
        for (i in 0 until length) {
            copyCoord(src, srcPos + i, dest, destPos + i)
        }
    }

    /**
     * Copies a coordinate of a [CoordinateSequence] to another [CoordinateSequence].
     * The sequences may have different dimensions;
     * in this case only the common dimensions are copied.
     *
     * @param src the sequence to copy from
     * @param srcPos the source coordinate to copy
     * @param dest the sequence to copy to
     * @param destPos the destination coordinate to copy to
     */
    fun copyCoord(
        src: CoordinateSequence,
        srcPos: Int,
        dest: CoordinateSequence,
        destPos: Int
    ) {
        val minDim: Int = min(src.dimension, dest.dimension)
        for (dim in 0 until minDim) {
            dest.setOrdinate(destPos, dim, src.getOrdinate(srcPos, dim))
        }
    }

    /**
     * Tests whether a [CoordinateSequence] forms a valid [LinearRing],
     * by checking the sequence length and closure
     * (whether the first and last points are identical in 2D).
     * Self-intersection is not checked.
     *
     * @param seq the sequence to test
     * @return true if the sequence is a ring
     * @see LinearRing
     */
    @JvmStatic
    fun isRing(seq: CoordinateSequence): Boolean {
        val n: Int = seq.size()
        if (n == 0) return true
        // too few points
        return if (n <= 3) false else (seq.getOrdinate(
            0,
            CoordinateSequence.X
        ) == seq.getOrdinate(n - 1, CoordinateSequence.X)
                && seq.getOrdinate(0, CoordinateSequence.Y) == seq.getOrdinate(
            n - 1,
            CoordinateSequence.Y
        ))
        // test if closed
    }

    /**
     * Ensures that a CoordinateSequence forms a valid ring,
     * returning a new closed sequence of the correct length if required.
     * If the input sequence is already a valid ring, it is returned
     * without modification.
     * If the input sequence is too short or is not closed,
     * it is extended with one or more copies of the start point.
     *
     * @param fact the CoordinateSequenceFactory to use to create the new sequence
     * @param seq the sequence to test
     * @return the original sequence, if it was a valid ring, or a new sequence which is valid.
     */
    @JvmStatic
    fun ensureValidRing(
        fact: CoordinateSequenceFactory,
        seq: CoordinateSequence
    ): CoordinateSequence {
        val n: Int = seq.size()
        // empty sequence is valid
        if (n == 0) return seq
        // too short - make a new one
        if (n <= 3) return createClosedRing(fact, seq, 4)
        val isClosed = (seq.getOrdinate(0, CoordinateSequence.X) == seq.getOrdinate(
            n - 1,
            CoordinateSequence.X
        )
                && seq.getOrdinate(
            0,
            CoordinateSequence.Y
        ) == seq.getOrdinate(n - 1, CoordinateSequence.Y))
        return if (isClosed) seq else createClosedRing(fact, seq, n + 1)
        // make a new closed ring
    }

    private fun createClosedRing(
        fact: CoordinateSequenceFactory,
        seq: CoordinateSequence,
        size: Int
    ): CoordinateSequence {
        val newseq: CoordinateSequence = fact.create(size, seq.dimension)
        val n: Int = seq.size()
        copy(seq, 0, newseq, 0, n)
        // fill remaining coordinates with start point
        for (i in n until size) copy(seq, 0, newseq, i, 1)
        return newseq
    }

    fun extend(
        fact: CoordinateSequenceFactory,
        seq: CoordinateSequence,
        size: Int
    ): CoordinateSequence {
        val newseq: CoordinateSequence = fact.create(size, seq.dimension)
        val n: Int = seq.size()
        copy(seq, 0, newseq, 0, n)
        // fill remaining coordinates with end point, if it exists
        if (n > 0) {
            for (i in n until size) copy(seq, n - 1, newseq, i, 1)
        }
        return newseq
    }

    /**
     * Tests whether two [CoordinateSequence]s are equal.
     * To be equal, the sequences must be the same length.
     * They do not need to be of the same dimension,
     * but the ordinate values for the smallest dimension of the two
     * must be equal.
     * Two `NaN` ordinates values are considered to be equal.
     *
     * @param cs1 a CoordinateSequence
     * @param cs2 a CoordinateSequence
     * @return true if the sequences are equal in the common dimensions
     */
    @JvmStatic
    fun isEqual(
        cs1: CoordinateSequence,
        cs2: CoordinateSequence
    ): Boolean {
        val cs1Size: Int = cs1.size()
        val cs2Size: Int = cs2.size()
        if (cs1Size != cs2Size) return false
        val dim: Int = min(cs1.dimension, cs2.dimension)
        for (i in 0 until cs1Size) {
            for (d in 0 until dim) {
                val v1: Double = cs1.getOrdinate(i, d)
                val v2: Double = cs2.getOrdinate(i, d)
                return if (cs1.getOrdinate(i, d) == cs2.getOrdinate(i, d)) {
                    continue
                } else if (isNaN(v1) && isNaN(v2)) {
                    // special check for NaNs
                    continue
                } else {
                    false
                }
            }
        }
        return true
    }

    /**
     * Creates a string representation of a [CoordinateSequence].
     * The format is:
     * <pre>
     * ( ord0,ord1.. ord0,ord1,...  ... )
    </pre> *
     *
     * @param cs the sequence to output
     * @return the string representation of the sequence
     */
    fun toString(cs: CoordinateSequence): String {
        val size: Int = cs.size()
        if (size == 0) return "()"
        val dim: Int = cs.dimension
        val builder: StringBuilder = StringBuilder()
        builder.append('(')
        for (i in 0 until size) {
            if (i > 0) builder.append(" ")
            for (d in 0 until dim) {
                if (d > 0) builder.append(",")
                builder.append(OrdinateFormat.DEFAULT.format(cs.getOrdinate(i, d)))
            }
        }
        builder.append(')')
        return builder.toString()
    }

    /**
     * Returns the minimum coordinate, using the usual lexicographic comparison.
     *
     * @param  seq  the coordinate sequence to search
     * @return  the minimum coordinate in the sequence, found using `compareTo`
     * @see Coordinate.compareTo
     */
    fun minCoordinate(seq: CoordinateSequence): Coordinate? {
        var minCoord: Coordinate? = null
        for (i in 0 until seq.size()) {
            val testCoord: Coordinate = seq.getCoordinate(i)
            if (minCoord == null || minCoord > testCoord) {
                minCoord = testCoord
            }
        }
        return minCoord
    }
    /**
     * Returns the index of the minimum coordinate of a part of
     * the coordinate sequence (defined by `from` and `to`,
     * using the usual lexicographic comparison.
     *
     * @param  seq   the coordinate sequence to search
     * @param  from  the lower search index
     * @param  to    the upper search index
     * @return  the index of the minimum coordinate in the sequence, found using `compareTo`
     * @see Coordinate.compareTo
     */
    /**
     * Returns the index of the minimum coordinate of the whole
     * coordinate sequence, using the usual lexicographic comparison.
     *
     * @param  seq  the coordinate sequence to search
     * @return  the index of the minimum coordinate in the sequence, found using `compareTo`
     * @see Coordinate.compareTo
     */
    @JvmOverloads
    @JvmStatic
    fun minCoordinateIndex(
        seq: CoordinateSequence,
        from: Int = 0,
        to: Int = seq.size() - 1
    ): Int {
        var minCoordIndex = -1
        var minCoord: Coordinate? = null
        for (i in from..to) {
            val testCoord: Coordinate = seq.getCoordinate(i)
            if (minCoord == null || minCoord > testCoord) {
                minCoord = testCoord
                minCoordIndex = i
            }
        }
        return minCoordIndex
    }

    /**
     * Shifts the positions of the coordinates until `firstCoordinate`
     * is first.
     *
     * @param  seq      the coordinate sequence to rearrange
     * @param  firstCoordinate  the coordinate to make first
     */
    fun scroll(seq: CoordinateSequence, firstCoordinate: Coordinate) {
        val i = indexOf(firstCoordinate, seq)
        if (i <= 0) return
        scroll(seq, i)
    }
    /**
     * Shifts the positions of the coordinates until the coordinate at  `firstCoordinateIndex`
     * is first.
     *
     * @param  seq      the coordinate sequence to rearrange
     * @param  indexOfFirstCoordinate
     * the index of the coordinate to make first
     * @param  ensureRing
     * makes sure that `` will be a closed ring upon exit
     */
    /**
     * Shifts the positions of the coordinates until the coordinate at  `firstCoordinateIndex`
     * is first.
     *
     * @param  seq      the coordinate sequence to rearrange
     * @param  indexOfFirstCoordinate  the index of the coordinate to make first
     */
    @JvmOverloads
    @JvmStatic
    fun scroll(
        seq: CoordinateSequence,
        indexOfFirstCoordinate: Int,
        ensureRing: Boolean = isRing(seq)
    ) {
        if (indexOfFirstCoordinate <= 0) return

        // make a copy of the sequence
        val copy: CoordinateSequence = seq.copy()

        // test if ring, determine last index
        val last: Int = if (ensureRing) seq.size() - 1 else seq.size()

        // fill in values
        for (j in 0 until last) {
            for (k in 0 until seq.dimension) seq.setOrdinate(
                j,
                k,
                copy.getOrdinate((indexOfFirstCoordinate + j) % last, k)
            )
        }

        // Fix the ring (first == last)
        if (ensureRing) {
            for (k in 0 until seq.dimension) seq.setOrdinate(last, k, seq.getOrdinate(0, k))
        }
    }

    /**
     * Returns the index of `coordinate` in a [CoordinateSequence]
     * The first position is 0; the second, 1; etc.
     *
     * @param  coordinate   the `Coordinate` to search for
     * @param  seq  the coordinate sequence to search
     * @return              the position of `coordinate`, or -1 if it is
     * not found
     */
    @JvmStatic
    fun indexOf(coordinate: Coordinate, seq: CoordinateSequence): Int {
        for (i in 0 until seq.size()) {
            if (coordinate.x == seq.getOrdinate(i, CoordinateSequence.X) &&
                coordinate.y == seq.getOrdinate(i, CoordinateSequence.Y)
            ) {
                return i
            }
        }
        return -1
    }
}